import { orderItemFilter, orderItemReducer } from './orderItem';

import { isPlusPaymentMethod } from '../../../../../helpers/data';
import { IOrder, IOrderItem, IOrderRefund, IPayment, OperationType } from '../../../../../types';
import { saftyParseFloat } from '../../../../lib/float';

const DEFAULT_SERVICE = {
    alias: 'unknown',
    id: 'unknown',
    name: 'Unknown',
    i18nKeyName: 'unknown',
    iconUrl: '',
};

function getValuesFromMap<K, V>(map: Map<K, V>): V[] {
    const result: V[] = new Array(map.size);
    let i = 0;

    for (const value of map.values()) {
        result[i] = value;
        i++;
    }

    return result;
}

interface IReducerOptions {
    isReceipt?: boolean;
}

export function orderReducer(
    order: Backend.Order,
    lookup: (groupId: string) => Backend.Order[],
    appContext: AppContext,
    options?: IReducerOptions,
): IOrder {
    const serviceData = order.service_data;
    const { isReceipt } = options || {};

    // Здесь мы собираем композитную транзакцию
    // Бэкэнд должен гарантировать наличие всех составляющих композитной транзакции
    // если они у него есть.
    const children =
        serviceData.payment_method === 'composite' && order.trust_payment_id ? lookup(order.trust_payment_id) : [order];

    const payments = children.map<IPayment>((child) => ({
        method: child.service_data.payment_method,
        price: saftyParseFloat(child.total),
        account: child.service_data.user_account,
        currency: child.currency,
    }));

    const [plusWithdraw, plusTopup, discountValue] = payments.reduce(
        (aux, child) => {
            const paymentMethod = child.method;

            switch (paymentMethod) {
                case 'yandex_account_withdraw':
                    return [aux[0] + child.price, aux[1], aux[2]];
                case 'yandex_account_topup':
                    return [aux[0], aux[1] + child.price, aux[2]];
                default:
                    if (appContext.discounts.has(paymentMethod)) {
                        return [aux[0], aux[1], aux[2] + child.price];
                    }

                    return aux;
            }
        },
        [0, 0, 0],
    );

    const cashback = order.cashback?.type === 'SUCCESS' ? saftyParseFloat(order.cashback.amount) : undefined;

    if (cashback && cashback > 0) {
        payments.push({
            method: 'cashback_edadeal',
            price: cashback,
            currency: 'RUB',
        });
    }

    const [items, refundOnly] = children.reduce<[IOrderItem[], boolean]>(
        (aux, child) => {
            const paymentMethod = child.service_data.payment_method;
            const isChildRefundOnly = child.items.length === 0;

            // Не учитываем плюсы в деталях транзакции так как
            // позиция в заказе будет дубликатом оплаченной деньгами
            if (isPlusPaymentMethod(paymentMethod) || appContext.discounts.has(paymentMethod)) {
                return [aux[0], aux[1] || isChildRefundOnly];
            }

            aux[0].push(...child.items.map(orderItemReducer));

            return [aux[0], aux[1] || isChildRefundOnly];
        },
        [[], false],
    );

    // Сохраняем всю информацию о рефандах
    const refunds = getValuesFromMap(
        children.reduce((aux, child) => {
            child.refunds.forEach(({ items, total }) => {
                const paymentMethod = child.service_data.payment_method;
                const currency = child.currency;

                if (appContext.discounts.has(paymentMethod)) {
                    return;
                }

                const key = `${paymentMethod}_${currency}_${child.service_data.user_account}`;

                if (!aux.has(key)) {
                    aux.set(key, {
                        paymentMethod,
                        total: 0,
                        items: [],
                        currency,
                        account: child.service_data.user_account,
                    });
                }

                const refund = aux.get(key);

                if (refund && refund.items) {
                    refund.total += saftyParseFloat(total);

                    if (isPlusPaymentMethod(paymentMethod)) {
                        refund.total = Math.round(refund.total);
                    } else {
                        // Не учитываем плюсы в деталях транзакции
                        refund.items = [...refund.items, ...items.filter(orderItemFilter).map(orderItemReducer)];
                    }
                }
            });

            return aux;
        }, new Map<string, IOrderRefund>()),
    );

    const service = appContext.getServiceById(order.subservice_id) || DEFAULT_SERVICE;

    let status: string = order.status;

    order.features?.is_advance && (status = 'advance');
    order.features?.operation_type === OperationType.ReturnArrival && (status = 'refunded');

    return {
        id: String(order.order_id),
        created: order.created,
        serviceId: order.subservice_id,
        refunds,
        items,
        orderId: String(order.order_id),
        status,
        total: saftyParseFloat(order.total) - plusWithdraw - discountValue,
        plus: plusTopup - plusWithdraw,
        currency: order.currency,
        payments,
        refundOnly,
        purchaseToken: order.trust_purchase_token,
        checksEnabled: appContext.withChecks(order.subservice_id),
        service: {
            id: service.alias,
            ...service,
        },
        initiatorUid: order.initiator_uid,
        isReceipt,
        receiptUrl: order.receipt_url,
        retailerName: order.retailer_name,
        retailerIcon: order.retailer_icon,
        cashback,
        source: order.source,
    };
}
