import { orderReducer } from './order';

import { isPlusPaymentMethod } from '../../../../../helpers/data';
import { IOrder, IOrderList } from '../../../../../types';
import { base64Encode } from '../../../../helpers';
import logger from '../../../../lib/logger';

const CURSOR_JOIN = '__';

const createCursor = (id: number | string, created: string) => base64Encode(id + CURSOR_JOIN + created);

function lookupFactory(orders: Backend.Order[]) {
    const cache = new Map<string, Backend.Order[]>();

    return function lookup(searchGroupId: string) {
        const cachedValue = cache.get(searchGroupId);

        if (cachedValue) {
            return cachedValue;
        }

        for (let i = 0; i < orders.length; i++) {
            const order = orders[i];
            const groupId = order.service_data.trust_group_id;

            if (!groupId) {
                continue;
            }

            const list = cache.get(groupId);

            if (list) {
                list.push(order);
            } else {
                cache.set(groupId, [order]);
            }
        }

        return cache.get(searchGroupId) || [];
    };
}

function castToOrderListEdgeNode(order: Omit<IOrder, 'service'>) {
    return order as IOrder;
}

function normalizeOrder(node: Backend.Order, order: Omit<IOrder, 'service'>) {
    const aux: IOrder[] = [];

    // Разделяем транзакции в которых есть одновременно оплата и возврат
    // Кроме случаев с отменой (когда стоимость items и refunds совпадает)
    if (order.refunds?.length && !order.refundOnly && order.status !== 'cancelled') {
        order.refunds.forEach((refund, index) => {
            const orderId = `${order.orderId}_refund_${index}`;
            const isPlusTransaction = isPlusPaymentMethod(refund.paymentMethod);

            aux.push(castToOrderListEdgeNode({
                ...order,
                id: orderId,
                orderId,
                refunds: [refund],
                items: [],
                total: isPlusTransaction ? 0 : refund.total,
                plus: isPlusTransaction ? refund.total : 0,
                payments: [
                    {
                        price: refund.total,
                        currency: refund.currency,
                        method: refund.paymentMethod,
                        account: refund.account,
                    },
                ],
                currency: refund.currency,
                fake: true,
                checksEnabled: false,
            }));
        });

        aux.push(castToOrderListEdgeNode({
            ...order,
            refunds: [],
            status: 'paid',
        }));
    } else {
        aux.push(castToOrderListEdgeNode(order));
    }

    return aux;
}

interface IReducerOptions {
    notIgnoreTrust?: boolean;
    isReceipt?: boolean;
}

export function ordersReducer(
    response: Backend.OrdersResponse['data'],
    appContext: AppContext,
    options?: IReducerOptions,
): IOrderList {
    const { orders, next } = response;
    const { notIgnoreTrust, isReceipt } = options || {};
    const lookup = lookupFactory(response.orders);
    const hasNextPage = Boolean(orders.length && next && next.order_id_keyset && next.created_keyset);

    const edges = orders.reduce<Array<IOrder>>((aux, node) => {
        // Транзакции с trust_group_id будут объединяться в одну в
        // orderReducer-е транзакции с payment_method === 'composite'
        if (node.service_data.trust_group_id && !notIgnoreTrust) {
            return aux;
        }

        aux.push(...normalizeOrder(node, orderReducer(node, lookup, appContext, { isReceipt })));

        return aux;
    }, []);

    if (!edges.length && hasNextPage) {
        logger.warn({ response }, 'Orders data was fully filtered but next exists');
    }

    return ({
        edges,
        pageInfo: {
            hasNextPage,
            startCursor: null,
            hasPreviousPage: false,
            endCursor: !hasNextPage ? null : createCursor(next.order_id_keyset, next.created_keyset),
        },
    });
}
