import { AxiosRequestConfig } from 'axios';
import type { NextFunction, Request, Response } from 'express';

import config from '../../../config';
import { createApiRequest } from '../../../lib/request';
import { apiErrorHandler } from '../../error/apiErrorHandler';
import { ApiMiddleware } from '../index';
import { suggestData } from './__mock__/suggest';

class SearchApi extends ApiMiddleware {
    constructor() {
        super();

        this.search = this.search.bind(this);
        this.request = createApiRequest({
            baseURL: config.search.url,
            timeout: config.search.timeout,
            responseType: 'json',
        });
    }

    getRequestOptions(req: Request): AxiosRequestConfig {
        const { tvm, blackbox } = req;
        const ticket = tvm?.frontend.tickets?.search?.ticket;

        const headers: AxiosRequestConfig['headers'] = {};

        if (ticket) {
            headers['X-Ya-Service-Ticket'] = ticket;
        }

        if (blackbox?.userTicket) {
            headers['X-Ya-User-Ticket'] = blackbox.userTicket;
        }

        return { headers };
    }

    async search(req: Request, res: Response, next: NextFunction) {
        try {
            const response = await this.request.get('/api/async/passport/multisearch', {
                ...this.getRequestOptions(req),
                params: {
                    request: req.query.text,
                    uid: req.blackbox?.uid,
                    length: 5,
                },
            });

            return res.json(response.data);
        } catch (error) {
            return apiErrorHandler(req, res, next)(error);
        }
    }

    async suggest(_req: Request, res: Response) {
        return res.json(suggestData);
    }
}

export const searchApi = new SearchApi();
