import { AxiosRequestConfig, AxiosResponse } from 'axios';
import type { Request } from 'express';

import { ApiMiddleware } from '..';
import config from '../../../config';
import { saftyParseFloat } from '../../../lib/float';
import logger from '../../../lib/logger';
import { createApiRequest } from '../../../lib/request';

class TrustApi extends ApiMiddleware {
    constructor() {
        super();

        this.request = createApiRequest({
            baseURL: config.trust.url,
            timeout: config.trust.timeout,
            responseType: 'json',
        });
    }

    getRequestOptions = (req: Request): AxiosRequestConfig => {
        const { tvm, blackbox } = req;
        const ticket = tvm?.frontend.tickets?.trust?.ticket;

        const headers: AxiosRequestConfig['headers'] = {
            'X-Request-Id': req.id,
        };

        if (ticket) {
            headers['X-Ya-Service-Ticket'] = ticket;
        }

        if (blackbox?.userTicket) {
            headers['X-Ya-User-Ticket'] = blackbox.userTicket;
        }

        return { headers };
    }

    getPlusBalance = async (req: Request) => {
        const amount = await this.request
            .get('/legacy/wallet-balance', {
                ...this.getRequestOptions(req),
                params: { uid: req.blackbox?.uid },
            })
            .then((response: AxiosResponse<TrustBackend.PlusBalance>) => {
                return response.data.balances?.[0]?.amount || '0';
            })
            .catch((e) => {
                const error = e?.response?.data;

                logger.error({ error }, '[TrustApi] getPlusBalance error');

                return '-1';
            });

        return saftyParseFloat(amount);
    }
}

const trust = new TrustApi();

export const getPlusBalance = trust.getPlusBalance;
