import type { RequestHandler } from 'express';
import * as bunker from 'express-bunker';
import * as wrapMiddleware from 'wrap-middleware';

import { getPayServicesPlusPromo, getPayServices, getPayServicesEnabled } from './pay-services';
import { parseUrlFactory } from './url';

import config from '../../config';
import { isYandexoid } from '../blackbox';

const parser: RequestHandler = (req, res, next) => {
    const { config } = req.bunker;

    const parseUrl = parseUrlFactory(req);

    const services = Object.keys(config.services).reduce<AppContext['services']>((aux, key) => {
        const item = config.services[key];

        if (!item) {
            return aux;
        }

        aux[key] = {
            alias: key,
            iconUrl: parseUrl(item.iconUrl),
            splashUrl: parseUrl(item.splashUrl),
            ids: new Set(item.ids),
            name: item.name,
            i18nKeyName: item.i18nKeyName,
            url: parseUrl(item.url),
            help: parseUrl(item.help),
        };

        return aux;
    }, {});

    const plusServiceData = {
        ...config.plusServiceData,
        iconUrl: parseUrl(config.plusServiceData.iconUrl),
        splashUrl: parseUrl(config.plusServiceData.splashUrl),
    };

    const servicesIdsWithoutCheck = new Set(config.servicesIdsWithoutCheck);
    const servicesIdsWithoutDetails = new Set(config.servicesIdsWithoutDetails);

    const appContext: AppContext = {
        services,
        payServicesEnabled: getPayServicesEnabled(req),
        payServices: getPayServices(req),
        payServicesPlusPromo: getPayServicesPlusPromo(req),
        servicesOrder: config.order,
        plusServiceData,
        withChecks(id: string) {
            return !servicesIdsWithoutCheck.has(id);
        },
        withDetails(id: string) {
            return !servicesIdsWithoutDetails.has(id);
        },
        discounts: new Set(config.discounts),
        getServiceById(id: string) {
            return Object.values(services).find((service) => {
                return service.ids.has(id);
            });
        },
        fnsEnabled: config.fnsEnabled || isYandexoid(req.blackbox?.raw),
        yandexPayEnabled: Boolean(config.yandexPayEnabled),
        newSecurity: config.newSecurity,
        newHelpdesk: config.newHelpdesk,
    };

    req.appContext = appContext;

    next();
};

const bunkerMiddleware = ((bunker as unknown) as Function)({
    api: config.bunker.url,
    project: config.bunker.project,
});

const handler: RequestHandler = (req, res, next) => {
    if (req.bunker.config.dead) {
        return res.redirect('https://any.yandex.ru');
    }

    parser(req, res, next);
};

export default ((wrapMiddleware as unknown) as Function)(bunkerMiddleware).after(handler);
