import type { Request, RequestHandler } from 'express';

import { Csrf } from '@yandex-int/csrf';

import config from '../config';

const csrf = new Csrf({ key: config.csrf.salt });

const getOptions = (req: Request) => {
    const {
        cookies: { yandexuid },
        blackbox,
    } = req;
    let uid = '0';

    if (blackbox && ['VALID', 'NEED_RESET'].includes(blackbox.status) && blackbox.raw) {
        const users = 'users' in blackbox.raw ? blackbox.raw.users : [];

        uid = users
            .map((user) => user.id)
            .sort()
            .join(',');
    }

    return {
        yandexuid,
        uid,
    };
};

export const generateToken: RequestHandler = (req, res, next) => {
    let token = 'csrf_token_generation_failed';

    try {
        const options = getOptions(req);

        token = csrf.generateToken(options);
    } catch (err) {
        req.log.error({
            err,
            type: 'csrf-generation',
            code: 'csrf_token.internal',
        });
    }

    res.locals.csrfToken = token;

    return next();
};

export const sendToken: RequestHandler = (req, res) => res.send({ token: res.locals.csrfToken });

export const checkToken: RequestHandler = (req, res, next) => {
    const {
        headers: { 'x-csrf-token': token },
    } = req;

    let isValid;
    let options;

    if (config.environment === 'stress') {
        return next();
    }

    try {
        options = getOptions(req);
        isValid = csrf.isTokenValid(token, options);
    } catch (err) {
        req.log.error({
            err,
            type: 'csrf-check',
            code: 'csrf_token.internal',
        }, 'csrf-check');
        isValid = false;
    }

    if (isValid) {
        return next();
    }

    req.log.error({
        type: 'csrf-check',
        code: 'csrf_token.invalid',
        ...options,
    }, 'csrf-check');

    return res.status(403).json({
        status: 'error',
        code: 'csrf_token.invalid',
    });
};
