import * as cookieParser from 'cookie-parser';
import * as express from 'express';
import type { Express } from 'express';

import * as expressYandexuid from '@yandex-int/express-yandexuid';

import * as billsApi from './api/bills';
import { getReceiptByPurchaseToken, getOrdersMiddleware, getOrdersApi, getPaymentById, familyMiddleware, getBaseData } from './api/ohio/ohio';
import { searchApi } from './api/search';
import baseTrailingSlash from './baseTrailingSlash';
import blackbox from './blackbox';
import bunkerMiddleware from './bunker';
import { generateToken, sendToken, checkToken } from './csrf';
import error from './error';
import headers from './headers';
import langDetect from './langDetect';
import loadableStats from './loadableStats';
import logger from './logger';
import params from './params';
import render from './render';
import render3dsComplete from './render/3ds-complete';
import frame3ds from './render/frame-3ds';
import services from './services';
import { setLang } from './setLang';
import tvm from './tvm';
import uatraits from './uatraits';
import userMatch from './userMatch';
import webAmMiddleware from './webAm';

import { getRoutes } from '../../routes';
import config from '../config';

const DEV = config.environment === 'dev';

const middleware = (app: Express) => {
    const routes = [...Object.values(getRoutes(config.baseUrl)), config.baseUrl + '/index.html'];

    if (DEV) {
        app.use('/favicon.ico', (req, res) => res.status(404).end());
    }

    app.disable('trust proxy');
    app.use(params, logger);
    app.use(express.json());
    app.use(express.urlencoded({ extended: false }));
    app.use(cookieParser());
    app.get(config.baseUrl + '/payment/frame-3ds', frame3ds); // не нужна авторизация
    app.get(config.baseUrl + '/payment/3ds-complete', render3dsComplete); // не нужна авторизация
    app.post(config.baseUrl + '/api/token', sendToken);
    app.use(config.baseUrl, express.static(config.paths.public));
    app.use(uatraits, expressYandexuid(), tvm, blackbox, langDetect, services, generateToken);
    app.use(bunkerMiddleware);
    app.get(config.baseUrl + '/api/orders', userMatch, checkToken, getOrdersApi);
    app.post(config.baseUrl + '/api/receipts', userMatch, checkToken, getReceiptByPurchaseToken);
    app.get(config.baseUrl + '/api/bills/documents', userMatch, checkToken, billsApi.getDocuments);
    app.post(config.baseUrl + '/api/bills/documents', userMatch, checkToken, billsApi.createDocument);
    app.put(config.baseUrl + '/api/bills/documents/:documentId', userMatch, checkToken, billsApi.updateDocument);
    app.delete(config.baseUrl + '/api/bills/documents/:documentId', userMatch, checkToken, billsApi.deleteDocument);
    app.get(config.baseUrl + '/api/bills/bills', userMatch, checkToken, billsApi.getBills);
    app.post(config.baseUrl + '/api/bills/search/bills', userMatch, checkToken, billsApi.restartSearch);
    app.post(config.baseUrl + '/api/bills/orders', userMatch, checkToken, billsApi.createOrder);
    app.post(config.baseUrl + '/api/bills/orders/:orderId/start', userMatch, checkToken, billsApi.startOrder);
    app.get(config.baseUrl + '/api/bills/transactions/:transactionId', userMatch, checkToken, billsApi.getTransaction);
    app.post(config.baseUrl + '/api/bills/notifications', userMatch, checkToken, billsApi.setNotificationsAgreement);
    app.get(config.baseUrl + '/api/suggest', searchApi.suggest);
    app.get(config.baseUrl + '/api/search', searchApi.search);
    app.use(config.baseUrl + '/api/*', (req, res) => res.sendStatus(404));
    app.use(headers);
    app.get(config.baseUrl + '/order/:paymentId', loadableStats, webAmMiddleware, familyMiddleware, getPaymentById, render);
    app.get(
        routes,
        loadableStats,
        baseTrailingSlash,
        getBaseData,
        familyMiddleware,
        getOrdersMiddleware,
        render,
    );
    app.get(config.baseUrl + '/setLang', setLang);
    app.get('*', (req, res) => res.redirect(config.baseUrl || '/'));
    app.use(error);
};

export default middleware;
