import * as fs from 'fs';
import * as https from 'https';
import { Transform as Stream } from 'stream';

import type { RequestHandler } from 'express';

import config from '../config';
import logger from '../lib/logger';

const getLoadableStats = () =>
    new Promise<void>((resolve) => {
        if (config.environment === 'dev' || fs.existsSync(config.paths.loadableStats)) {
            return resolve();
        }

        const onError = (err: Error) => {
            logger.error({ err, type: 'loadable-stats' });
            // eslint-disable-next-line
            setTimeout(() => getLoadableStats().then(resolve), 500);
        };
        const request = https.request(`${config.paths.versionized}/loadable-stats.json`, (res) => {
            const data = new Stream();

            res.on('data', (chunk) => data.push(chunk));
            res.on('end', () => {
                const content = data.read().toString();

                content === 'Not Found' ?
                    onError(new Error('file not found')) :
                    fs.writeFile(config.paths.loadableStats, content, (err) => (err ? onError(err) : resolve()));
            });
        });

        request.on('error', onError);
        request.end();
    });

const promise = getLoadableStats();

const handler: RequestHandler = (req, res, next) => promise.then(() => next());

export default handler;
