import type { Response, RequestHandler } from 'express';

import logger from '../lib/logger';
import Time from '../lib/time';

const levelFromStatus = (statusCode: number) => {
    const key = String(Math.floor(statusCode / 100));
    const levels: Record<string, 'info' | 'warn' | 'error'> = {
        '1': 'warn',
        '2': 'info',
        '3': 'warn',
        '4': 'error',
        '5': 'error',
    };

    return levels[key] || 'error';
};

const handler: RequestHandler = (req, res, next) => {
    const time = new Time();

    req.log = logger.child({
        requestId: req.id,
        ssrId: req.ssrId,
        yandexuid: req.cookies && req.cookies.yandexuid,
    });

    req.log.info({
        type: 'request',
        method: req.method,
        url: req.originalUrl,
    }, `Request → ${req.method} ${req.originalUrl}`);

    function onResFinished(this: Response) {
        const { statusCode } = res;
        const level = levelFromStatus(statusCode);
        const { raw } = req.blackbox || {};

        req.log[level]({
            type: 'response',
            ms: time.ms(),
            method: req.method,
            url: req.originalUrl,
            statusCode,
            uid: raw && 'default_uid' in raw ? raw.default_uid : null,
        }, `${statusCode} ← ${req.method} ${req.originalUrl}`);

        this.removeListener('finish', onResFinished);
    }

    function onResError(error: Error) {
        const { statusCode } = res;
        const level = levelFromStatus(statusCode);
        const { raw } = req.blackbox || {};

        req.log[level]({
            type: 'response',
            ms: time.ms(),
            method: req.method,
            url: req.originalUrl,
            statusCode,
            uid: raw && 'default_uid' in raw ? raw.default_uid : null,
        }, `${statusCode} ← ${req.method} ${req.originalUrl} | ${error.name} / ${error.message}`);
    }

    res.on('finish', onResFinished);
    res.on('error', onResError);

    return next();
};

export default handler;
