import type { RequestHandler, Request } from 'express';
import { isString } from 'lodash';

import { errorMiddlewareFactory } from '../error';

interface FormParams {
    action: string;
    method: 'get' | 'post';
}

const getHtml = (formParams: FormParams) => `<!doctype html>
<html style="height: 100%;">
    <head>
        <title>YandexID.Payment::3DS</title>
        <meta charset="utf-8"/>
    </head>
    <body style="height: 100%;">
        <style>
            html,body,iframe {
                display: block;
                width: 100%;
                min-width: 100%;
                height: 100%;
                min-height: 100%;
                margin: 0;
                padding: 0;
                border: none;
            }
            form {
                position: absolute;
                opacity: 0;
            }
        </style>
        <form target="frame-3ds-inner" action="${formParams.action}" method="${formParams.method}">
            <input type="submit" value="send" />
        </form>
        <iframe name="frame-3ds-inner" id="frame-3ds-inner" title="3DS"></iframe>
        <script>
            window.addEventListener('message', function(ev) {
                if (parent !== window && parent.postMessage) {
                    parent.postMessage(ev.data, "*");
                }
            });

            var form = document.getElementsByTagName('form')[0];

            form.submit();
            form.style.display = 'none';
        </script>
    </body>
</html>
`;

const getQuerySrc = (req: Request): string => {
    const { src } = req.query;

    if (!isString(src)) {
        throw new Error(`Invalid 3ds param - src="${JSON.stringify(src)}"`);
    }

    return src;
};

const getQueryMethod = (req: Request): 'post'| 'get' => {
    const { method } = req.query;

    if (!['post', 'get'].includes(method)) {
        throw new Error(`Invalid 3ds param - method="${JSON.stringify(method)}"`);
    }

    return method as 'post'| 'get';
};

const handler: RequestHandler = (req, res, next) => {
    let html;

    try {
        const src = getQuerySrc(req);
        const method = getQueryMethod(req);

        html = getHtml({ action: src, method });
    } catch (err) {
        return next(err);
    }

    res.send(html);
};

export default [handler, errorMiddlewareFactory('frame-3ds')];
