import type { Response, RequestHandler } from 'express';
import * as htmlescape from 'htmlescape';
import { renderToString } from 'react-dom/server';
import type { StaticRouterContext } from 'react-router';

import { parseTld } from '@yandex-int/express-langdetect/dist/lib/utils';

import type { Initialized } from './util';
import { getInitialized } from './util';

import Metrika from '../../../client/lib/metrika';
import { getRoutes } from '../../../routes';
import type { State, Constants, InitialSsrErrorMeta, ServerRenderProps } from '../../../types';
import config from '../../config';
import { createExtractor } from '../../helpers';
import { errorMiddlewareFactory } from '../error';
import { isYandexoid } from '../blackbox';

// eslint-disable-next-line @typescript-eslint/no-var-requires
const fontMeta = require('yandex-font/build/meta');

const handler: RequestHandler = async (req, res, next) => {
    const routerContext: StaticRouterContext = {};
    const webAmParams = res.locals.webAmParams;
    const services = req.services;
    const lang = webAmParams?.lang || services.misc.lang;
    const isPWA = req.headers['x-requested-by'] === 'sw';
    const { blackbox } = req;
    const userInfo = services.user.authInfo.user;

    if (blackbox && blackbox.status === 'VALID' && blackbox.userTicket) {
        userInfo.name = blackbox.displayName;
        userInfo.avatarId =
            blackbox.avatar && !blackbox.avatar.empty ? blackbox.avatar.default : undefined;
        userInfo.hasPlus =
            'attributes' in blackbox.raw
                ? blackbox.raw.attributes?.hasOwnProperty('1015')
                : undefined;
        userInfo.plusBalance = res.locals.plusBalance;
        userInfo.isFamilyAdmin = res.locals.isFamilyAdmin;
    }

    const tld = parseTld(req, req.hostname);
    const isLite = Boolean(req.query.isLite);
    const familyMode = Boolean(req.query.family);

    let alias = req.query.alias as string;

    if (req.path === getRoutes(config.baseUrl).receipts) {
        alias = 'receipts';
    }

    const constants: Constants = {
        env: config.environment,
        lang,
        isPWA,
        tld,
        nonce: req.nonce,
        baseUrl: config.baseUrl,
        platform: {
            isBrowser: req.uatraits.isBrowser,
            isMobile: req.uatraits.isMobile,
            isTouch: req.uatraits.isTouch,
            BrowserName: req.uatraits.BrowserName,
            OSFamily: req.uatraits.OSFamily,
        },
        config: {
            passportHost: config.passport.host.replace('{tld}', tld),
            avatarHost: config.images.avatarHost,
            yandexHost: config.yandex.host.replace('{tld}', tld),
            edadealHosts: config.edadeal.hosts,
        },
        payServicesEnabled: !isLite && !familyMode && req.appContext.payServicesEnabled,
        payServices: req.appContext.payServices,
        payServicesPlusPromo: req.appContext.payServicesPlusPromo,
        familyUsers: res.locals.familyUsers,
        plusServiceData: req.appContext.plusServiceData,
        webAmParams: res.locals.webAmParams,
        fnsData: {
            isBind: res.locals.fns?.status === 'has_binding',
            urls: res.locals.fns?.urls,
            amount: res.locals.fns?.amount,
        },
        flags: {
            billsOnly: req.query.billsOnly === '1',
            noHeader: req.query.noHeader === '1',
            fnsEnabled: !isLite && !familyMode && req.appContext.fnsEnabled && tld === 'ru',
            newSecurity: req.appContext.newSecurity,
            newHelpdesk: req.appContext.newHelpdesk,
            isYandexoid: isYandexoid(req.blackbox?.raw),
            hideReceiptBanner:
                isLite ||
                familyMode ||
                !req.appContext.fnsEnabled ||
                Boolean(req.cookies.ohio_hide_receipt_banner) ||
                !res.locals.fns ||
                res.locals.fns?.status === 'has_binding' ||
                res.locals.fns?.status === 'blocked',
        },
        isLite,
        familyMode,
        ...(isPWA
            ? {}
            : {
                ssrId: req.ssrId,
                metrikaId: config.metrika.id,
                authInfo: {
                    user: userInfo,
                },
            }),
        yandexPay: config.yandexPay,
        bills: config.bills,
    };

    const state: State = {
        orders: {
            isOffline: false,
            services: res.locals.services,
            orders: res.locals.orders,
            order: res.locals.order,
            alias,
            familyFilter: alias !== 'plus' ? (req.query.familyFilter as string) : undefined,
        },
        bills: {
            documents: res.locals.billsState?.documents,
            bills: res.locals.billsState?.bills,
            payment: {
                state: 'initial',
                transactionId: '',
            },
        },
    };
    const props: ServerRenderProps & { res: Response } = {
        req,
        res,
        state,
        constants,
        metrika: new Metrika(),
        context: routerContext,
    };
    const fontLoaded = req.cookies.font_loaded === fontMeta.fontVersion;
    let initialized: Initialized | null = null;
    let initialSsrErrorMeta: InitialSsrErrorMeta | undefined = undefined;

    if (!['rc', 'prod'].includes(config.environment)) {
        constants.debug = true;
    }

    try {
        initialized = await getInitialized(props);
    } catch (err) {
        req.log.error({ err, type: 'ssr' }, 'ssr');
        try {
            initialSsrErrorMeta = {
                requestId: req.id,
                timestamp: new Date().toISOString(),
            };
            // repeat render with no content, just header & footer and an error message
            initialized = await getInitialized({
                ...props,
                initialSsrErrorMeta,
            });
        } catch (e) {
            return next(e);
        }
    }

    if (!initialized) {
        return next(new Error('something went rent wrong with ssr'));
    }

    const { root, extractor } = initialized;
    const linkTags = extractor.getLinkTags();
    const styleTags = extractor.getStyleTags();
    const scriptTags = extractor.getScriptTags();

    const poolyfillExtractor = createExtractor({
        entrypoints: ['polyfill'],
        namespace: 'polyfill',
    });

    const htmlClasses = [];

    if (fontLoaded) {
        htmlClasses.push('font_loaded');
    }

    if (req.uatraits.isTouch) {
        htmlClasses.push('touch');
    }

    if (req.uatraits.isMobile) {
        htmlClasses.push('mobile');
    }

    if (req.uatraits.isBrowser) {
        htmlClasses.push('browser');
    }

    res.send(`
    <!doctype html>
    <html lang="${lang}" class="${htmlClasses.join(' ')}">
      <head>
        <meta charset="utf-8" />
        <meta
            name="viewport"
            content="width=device-width, initial-scale=1, maximum-scale=1, minimum-scale=1, shrink-to-fit=no, viewport-fit=cover"
        />
        <title>${lang === 'en' ? 'Payment history' : 'История платежей'}</title>
        ${isPWA ? '' : linkTags}
        ${styleTags}
      </head>
      <body>
        <div id="root" class="root">${renderToString(root)}</div>
        <script nonce="${req.nonce}">
          window.__INITIAL_SSR_ERROR_META__ = ${!initialSsrErrorMeta ? undefined : htmlescape(initialSsrErrorMeta)
        };
          window.__STATE__ = ${htmlescape(state)};
          window.__FONT_LOADED__ = ${htmlescape(fontLoaded)};
          window.__CONSTANTS__ = ${htmlescape(constants)};
          window.__CSRF__ = ${htmlescape(res.locals.csrfToken)};
        </script>
        ${poolyfillExtractor.getScriptTags().replace(/async/gm, '')}
        ${scriptTags}
      </body>
    </html>`);
};

export default [handler, errorMiddlewareFactory('ssr-static-500')];
