declare namespace ExpressBlackbox {
    interface PassportUser {
        id: string;
        family_info: any;
    }

    interface PassportSessionidMultiResponse {
        login?: undefined;
        family_info?: {
            admin_uid: string;
            family_id: string;
        };
        aliases?: PassportUser['aliases'];
        attributes?: PassportUser['attributes'];
    }
}

declare module 'wrap-middleware' {
    function wrapMiddleware(
        req,
        res,
        next,
    ): RequestHandler & {
        after(req, res, next): RequestHandler;
    };

    export default wrapMiddleware;
}

declare module 'express-bunker' {
    import { RequestHandler } from 'express';
    // eslint-disable-next-line import/no-extraneous-dependencies
    import { GotOptions } from 'got';

    // tslint:disable-next-line:no-any
    type Parser = (node: any, next: () => void) => void;

    interface Options {
        /** API endpoint */
        api: string;

        /** Project name */
        project: string;

        /** Version of bunker nodes (Default: latest) */
        version?: string;

        /** Update interval in milliseconds (Default: 5000) */
        updateInterval?: number;

        /** `true` if directories content should be loaded (Default: true) */
        directories?: boolean;

        /** `true` if bunker-cache should be enabled (Default: true) */
        cache?: boolean;

        /**
         * `true` if data should be deep freezed (Default: false). Incompatible with `cache: true`
         *
         * See:
         * https://developer.mozilla.org/ru/docs/Web/JavaScript/Reference/Global_Objects/Object/freeze
         */
        freeze?: boolean;

        /**
         * Array of bunker-to-js parsers
         * (Default: [
         *    BunkerToJs.filter.hidden,
         *    BunkerToJs.filter.empty,
         *    BunkerToJs.avatar,
         *    BunkerToJs.tjson,
         *    BunkerToJs.json,
         *    BunkerToJs.cat
         * ])
         *
         * See:
         * https://github.yandex-team.ru/toolbox/bunker-to-js/#parsers
         */
        parsers?: Parser[];

        /** Property name where data should be stored (Default: bunker) */
        propertyName?: string;
    }

    export type ExpressBunkerOptions = Options & GotOptions<string | null>;

    /**
     * Loads data from bunker project into req.bunker
     *
     * See:
     * https://github.yandex-team.ru/toolbox/express-bunker
     */
    function expressBunker(options?: ExpressBunkerOptions): RequestHandler;

    export default expressBunker;
}

interface RawBunker {
    config: {
        dead?: boolean;
        services: Record<
            string,
            {
                ids: string[];
                iconUrl?: string;
                splashUrl?: string;
                i18nKeyName?: string;
                name: string;
                url?: string;
                help?: string;
            }
        >;
        payServicesEnabled: boolean;
        payServicesWhitelist?: string[];
        payServices: {
            name: string;
            iconUrl: string;
            detailsIconUrl?: string;
            splashUrl?: string;
            path?: string;
            disabled?: boolean;
        }[];
        payServicesPlusPromo?: Record<
            string,
            {
                active: [string, string];
                label: {
                    text: string;
                    plusText: string;
                };
                description?: {
                    text: string;
                    legal: {
                        text: string;
                        href: string;
                    };
                };
            }
        >;
        discounts: string[];
        order: string[];
        servicesIdsWithoutCheck: string[];
        servicesIdsWithoutDetails: string[];
        plusServiceData: PlusServiceData;
        fnsEnabled?: boolean;
        yandexPayEnabled?: boolean;
        newSecurity: boolean;
        newHelpdesk: boolean;
    };
    images: Record<string, string>;
}

interface PayService {
    name: string;
    iconUrl: string;
    detailsIconUrl?: string;
    splashUrl?: string;
    path?: string;
    disabled?: boolean;
}

interface PayServicesPlusPromo {
    label: {
        text: string;
        plusText: string;
    };
    description?: {
        text: string;
        legal: {
            text: string;
            href: string;
        };
    };
}

interface BunkerService {
    alias: string;
    ids: Set<string>;
    iconUrl?: string;
    splashUrl?: string;
    i18nKeyName?: string;
    name: string;
    url?: string;
    help?: string;
}

interface AppContext {
    services: Record<string, BunkerService>;
    fnsEnabled?: boolean;
    payServices: PayService[];
    payServicesEnabled?: boolean;
    payServicesPlusPromo?: PayServicesPlusPromo;
    servicesOrder: string[];
    plusServiceData: PlusServiceData;
    discounts: Set<string>;
    yandexPayEnabled: boolean;
    newSecurity: boolean;
    newHelpdesk: boolean;
    withChecks(id: string): boolean;
    withDetails(id: string): boolean;
    getServiceById(id: string): BunkerService | undefined;
}

declare namespace Express {
    interface Request {
        id: string;
        ip: string;
        ssrId: string;
        nonce: string;
        log: import('pino').Logger;
        services: import('./services').Services;
        langdetect: {
            id: string;
            name: string;
        };
        appContext: AppContext;
        uatraits: Constants.platform;
        bunker: RawBunker;
        blackbox: ExpressBlackbox.Result;
    }
}

declare namespace Backend {
    type Currency = string;
    type OrderStatus = 'paid' | 'cancelled' | 'refunded' | 'hold';

    interface ListRequest { }

    interface OrderItem {
        amount: string;
        currency: Currency;
        name: string;
        nds: string;
        price: string;
        hidden?: boolean;
        alias?: string;
    }

    interface ServiceData {
        subservice_id: string;
        payment_method: 'composite' | string;
        user_account: string;
        trust_group_id: string | null;
    }

    interface Order {
        created: IsoDateTimeShift;
        currency: Currency;
        items: OrderItem[];
        order_id: number;
        refunds: OrderRefund[];
        service_data: ServiceData;
        subservice_id: string;
        status: OrderStatus;
        total: string;
        trust_payment_id: string;
        trust_purchase_token: string;
        initiator_uid?: string;
        receipt_url?: string;
        retailer_name?: string;
        retailer_icon?: string;
        cashback?: {
            type: 'SUCCESS' | 'FAIL';
            amount: string;
        };
        features?: {
            is_advance: boolean;
            operation_type: number;
        };
        source: 'trust' | 'yandexpay';
        merchant_id?: string;
        gateway_name?: string;
    }

    interface OrderRefund {
        items: OrderItem[];
        total: string;
        currency: string;
    }

    interface Response<D> {
        status: number;
        error?: string;
        data: D;
    }

    type OrdersResponse = Response<{
        next: {
            created_keyset: IsoDateTime;
            order_id_keyset: number;
        };
        orders: Order[];
    }>;

    type ItemResponse = Response<{
        order: Order;
    }>;

    type ServicesResponse = Response<{
        services: ServiceData[];
    }>;

    type CashbackBalanceResponse = Response<{
        amount: string;
    }>;

    interface ReceiptPayment {
        amount: string;
        payment_type: string;
    }

    interface Receipt {
        receipt_type: string;
        url: string;
        payments: ReceiptPayment[];
        timestamp: string;
    }

    interface ReceiptsData {
        receipts: Receipt[];
    }

    interface Service {
        service_id: string;
        subservice_id: string;
    }

    interface ServiceData {
        services: Service[];
    }

    interface FamilyUser {
        initiator_uid: number;
    }

    interface FamilyUsersData {
        familypay_users: FamilyUser[];
    }

    type FamilyResponse = Response<FamilyUsersData>;
    type ReceiptsResponse = Response<ReceiptsData>;
    type ServiceResponse = Response<ServiceData>;

    interface FnsBinding {
        status: 'blocked' | 'no_binding' | 'in_progress' | 'has_binding';
        urls: {
            touch: string;
            desktop?: string;
        };
    }
}

declare namespace BillsBackend {
    type DocumentType = 'VEHICLE_REGISTRATION_CERTIFICATE' | 'DRIVER_LICENSE';

    interface Document {
        document_id: string;
        type: DocumentType;
        title: string;
        value: string;
        created: string;
        updated: string;
    }

    interface Bill {
        bill_id: string;
        supplier_bill_id: string;
        document_id: string;
        discount_date: string;
        uid: number;
        kpp: string;
        account_number: string;
        bik: string;
        inn: string;
        paid_amount: number;
        discount_size: string;
        paid_date: string;
        treasure_branch: string;
        dep_type: string;
        payee_name: string;
        bill_date: string;
        status: string;
        amount: number;
        oktmo: string;
        kbk: string;
        div_id: number;
        purpose: string;
        amount_to_pay: number;
        payer_name: string;
        discounted_amount: number;
        fee_amount: number;
        offense_name: string;
        offense_place: string;
        offense_date: string;
        legal_act: string;
        payment_deadline: string;
    }

    interface PayRequest {
        transaction_id: string;
        payment_method: string;
        payment_token: string;
        payer_full_name: string;
        mpi_3ds_info: Mpi3dsInfo;
        return_url: string;
    }

    interface Mpi3dsInfo {
        browser_accept_header: string;
        browser_tz: string;
        browser_language: string;
        browser_screen_width: number;
        browser_user_agent: string;
        window_width: number;
        browser_javascript_enabled: boolean;
        browser_screen_height: number;
        window_height: number;
        browser_color_depth: number;
        browser_ip: string;
    }

    interface Response<T> {
        code: string;
        status: string;
        data: T;
    }

    interface DocumentData {
        document: Document;
    }

    interface DocumentsData {
        documents: Document[];
        has_documents: boolean;
    }

    interface OrderData {
        order: {
            status: string;
            created: string;
            order_id: string;
            bill_ids: string[];
            updated: string;
        };
    }

    interface BillsData {
        state: 'unregistered' | 'syncing' | 'completed';
        bills: Bill[];
    }

    type TransactionStatus = 'NEW' | 'PAID' | 'CANCELLED' | 'REFUNDED';

    interface BillPayment {
        transaction: {
            payer_full_name: string;
            acs_url: string;
            order_id: string;
            status: string;
            payment_method: string;
            amount: number;
            return_url: string;
            created: string;
            updated: string;
            transaction_id: string;
            payment_token: string;
            mpi_3ds_info: {
                browser_accept_header: string;
                browser_tz: string;
                browser_language: string;
                browser_screen_width: number;
                browser_user_agent: string;
                window_width: number;
                browser_javascript_enabled: boolean;
                browser_screen_height: number;
                window_height: number;
                browser_color_depth: number;
                browser_ip: string;
            };
        };
        status: TransactionStatus;
    }

    interface BillPaymentStatus {
        transaction: {
            status: TransactionStatus;
            transaction_id: string;
        };
    }

    type DeleteResponse = Response<{}>;
    type DocumentResponse = Response<DocumentData>;
    type DocumentsResponse = Response<DocumentsData>;
    type CreateOrderResponse = Response<OrderData>;
    type BillsResponse = Response<BillsData>;
    type PayResponse = Response<BillPayment>;
    type PaymentStatusResponse = Response<BillPaymentStatus>;
}

declare namespace TrustBackend {
    type Wallet = {
        wallet_id: string;
        amount: string;
        currency: string;
    };

    interface PlusBalance {
        balances?: Wallet[];
    }
}
