import type { Request } from 'express';
import type { StaticRouterContext } from 'react-router';

import type Metrika from './client/lib/metrika';
import type { Config as ServerConfig } from './server/config/types';
import type { UserInfo } from './server/middleware/api/blackbox/types';
import type { WebAmInitialData } from './server/middleware/webAm';

export enum OperationType {
    Arrival = 1,
    ReturnArrival,
    Expence,
    ExpenseRefund,
}

export type SecondaryUser = {
    uid: string;
};
export type User = SecondaryUser & {
    yu: string;
    name?: string;
    avatarId?: string;
    hasPlus?: boolean;
    login?: string;
    plusBalance?: number;
    isFamilyAdmin?: boolean;
};
export type AuthInfo = {
    user: User;
};

export type State = {
    orders: OrdersState;
    bills: BillsState;
}

export type OrdersState = {
    isOffline?: boolean;
    services?: IService[];
    orders?: IOrderList;
    order?: IOrder;
    alias?: string;
    familyFilter?: string;
    isUpdating?: boolean;
    isUpdateFailed?: boolean;
    firstLoading?: boolean;
};

export type BillsState = {
    documents: IDocumentsState;
    bills: IBillsState;
    payment: IBillPaymentState;
}

export type IDocumentStateType = 'initial' | 'syncing' | 'stable';

export type IDocumentsState = {
    state: IDocumentStateType;
    list: IDocument[];
}

export type IBillStateApiType = 'unregistered' | 'syncing' | 'completed';
export type IBillStateType = 'initial' | 'outdated' | 'emulation' | IBillStateApiType;

export type IBillsState = {
    state: IBillStateType;
    list: IBill[];
    selected: string[];
}

export type IBillStateWithoutSelected = Omit<IBillsState, 'selected'>;

export type IBillPaymentId = string;
export type IBillPaymentToken = string;
export type IBillPaymentBills = string[];
export type IBillPaymentAuth3ds = {
    method: 'get' | 'post';
    url: string;
}
export type IBillPaymentStateInitial = {
    state: 'initial';
    transactionId: '';
};
export type IBillPaymentStateWaitToken = {
    state: 'wait-token';
    transactionId: '';
};
export type IBillPaymentStateWaitPay = {
    state: 'wait-pay';
    transactionId: '';
};
export type IBillPaymentStateCreate = {
    state: 'create';
    transactionId: '';
    token: IBillPaymentToken;
};
export type IBillPaymentStateCancel = {
    state: 'cancel';
    transactionId: IBillPaymentId;
};
export type IBillPaymentStateProcess = {
    state: 'process';
    transactionId: IBillPaymentId;
    token: IBillPaymentToken;
};
export type IBillPaymentStateAuth3ds = {
    state: 'auth-3ds';
    transactionId: IBillPaymentId;
    token: IBillPaymentToken;
    auth3ds: IBillPaymentAuth3ds;
};
export type IBillPaymentStateFail = {
    state: 'fail';
    transactionId: IBillPaymentId;
    token: IBillPaymentToken;
    reason?: string;
};
export type IBillPaymentStateSuccess = {
    state: 'success';
    transactionId: IBillPaymentId;
    token: IBillPaymentToken;
};
export type IBillPaymentState =
    | IBillPaymentStateInitial
    | IBillPaymentStateWaitToken
    | IBillPaymentStateCreate
    | IBillPaymentStateCancel
    | IBillPaymentStateProcess
    | IBillPaymentStateAuth3ds
    | IBillPaymentStateFail
    | IBillPaymentStateSuccess
    | IBillPaymentStateWaitPay

export type ClientMpiInfo = {
    browser_tz: string;
    browser_screen_width: number;
    window_width: number;
    browser_screen_height: number;
    window_height: number;
    browser_color_depth: number;
}

export type IBillPayApiRequest = {
    payment_token: string;
    payment_method: string;
    payer_full_name: string;
    return_url: string;
    mpi_3ds_info: ClientMpiInfo;
}

export type IBillCreateOrderApiResult = {
    order: {
        status: string;
        created: string;
        order_id: string;
        bill_ids: string[];
        updated: string;
    };
}

export type TransactionStatus = 'NEW' | 'PAID' | 'CANCELLED' | 'REFUNDED';

export type IBillPayApiResult = {
    transaction: {
        payer_full_name: string;
        acs_url: string;
        order_id: string;
        status: string;
        payment_method: string;
        amount: number;
        return_url: string;
        created: string;
        updated: string;
        transaction_id: string;
        payment_token: string;
        mpi_3ds_info: {
            browser_accept_header: string;
            browser_tz: string;
            browser_language: string;
            browser_screen_width: number;
            browser_user_agent: string;
            window_width: number;
            browser_javascript_enabled: boolean;
            browser_screen_height: number;
            window_height: number;
            browser_color_depth: number;
            browser_ip: string;
        };
    };
    status: TransactionStatus;
}
export type IBillPaymentStatusApiResult = {
    transaction: {
        status: TransactionStatus;
        transaction_id: string;
    };
}

export type IDocumentType = 'VEHICLE_REGISTRATION_CERTIFICATE' | 'DRIVER_LICENSE';

export type IDocument = {
    id: string;
    type: IDocumentType;
    value: string;
    title: string;
}

export type IBill = {
    bill_id: string;
    supplier_bill_id: string;
    document_id: string;
    discount_date: string;
    uid: number;
    kpp: string;
    account_number: string;
    bik: string;
    inn: string;
    paid_amount: number;
    discount_size: string;
    paid_date: string;
    treasure_branch: string;
    dep_type: string;
    payee_name: string;
    bill_date: string;
    status: string;
    amount: number;
    oktmo: string;
    kbk: string;
    div_id: number;
    purpose: string;
    amount_to_pay: number;
    payer_name: string;
    discounted_amount: number;
    fee_amount: number;
    offense_name: string;
    offense_place: string;
    offense_date: string;
    legal_act: string;
    payment_deadline: string;
}

export type IBillsApiResult = {
    state: IBillStateApiType;
    bills: IBill[];
}

export type Lang = 'ru' | 'en' | 'kk' | 'be';
export interface Config {
    passportHost?: string;
    avatarHost?: string;
    yandexHost?: string;
    edadealHosts?: string[];
}
export type Constants = {
    env: ServerConfig['environment'];
    isPWA: boolean;
    ssrId?: string;
    baseUrl: string;
    debug?: boolean;
    lang: Lang;
    tld: string;
    metrikaId?: string;
    authInfo?: AuthInfo;
    platform: {
        isTouch: boolean;
        isMobile: boolean;
        isBrowser: boolean;
        BrowserName: string;
        OSFamily: string;
    };
    payServicesEnabled?: boolean;
    payServices: IPayService[];
    payServicesPlusPromo?: {
        label: {
            text: string;
            plusText: string;
        };
        description?: {
            text: string;
            legal: {
                text: string;
                href: string;
            };
        };
    };
    config: Config;
    plusServiceData: PlusServiceData;
    nonce: string;
    familyUsers?: Record<string, UserInfo>;
    webAmParams?: WebAmInitialData;
    fnsData?: {
        isBind?: boolean;
        urls?: {
            touch: string;
            desktop?: string;
        };
        amount: number;
    };
    flags: {
        hideReceiptBanner?: boolean;
        fnsEnabled?: boolean;
        billsOnly?: boolean;
        noHeader?: boolean;
        isYandexoid: boolean;
        newSecurity: boolean;
        newHelpdesk: boolean;
    };
    isLite?: boolean;
    familyMode?: boolean;
    yandexPay: {
        scriptSrc: string;
        merchantId: string;
        merchantName: string;
        gateway: string;
        gatewayId: string;
        env: 'TESTING' | 'PRODUCTION';
    };
    bills: {
        maxSelected: number;
    };
};
export type InitialSsrErrorMeta = {
    requestId: ID;
    timestamp: IsoDateTime;
};
export type RenderProps = {
    state: State;
    metrika: Metrika;
    constants: Constants;
    initialSsrErrorMeta?: InitialSsrErrorMeta;
};
export type ServerRenderProps = RenderProps & {
    req: Request;
    context: StaticRouterContext;
};

export interface IReceipt {
    url: string;
    amount?: string[];
    timestamp?: string;
    type?: string;
}

export interface IPayService {
    name: string;
    iconUrl: string;
    detailsIconUrl?: string;
    splashUrl?: string;
    path?: string;
    disabled?: boolean;
}

export interface IService {
    id: string;
    alias: string;
    iconUrl?: string;
    splashUrl?: string;
    name: string;
    i18nKeyName?: string;
    url?: string;
    help?: string;
}

export type IOrder = {
    id: string;
    total: number;
    status?: string;
    items?: IOrderItem[];
    refunds?: IOrderRefund[];
    created: IsoDateTimeShift;
    serviceId: string;
    service: IService;
    orderId: string;
    plus?: number;
    payments?: IPayment[];
    checksEnabled?: boolean;
    currency: string;
    fake?: boolean;
    refundOnly?: boolean;
    purchaseToken: string;
    initiatorUid?: string;
    isReceipt?: boolean;
    receiptUrl?: string;
    retailerName?: string;
    retailerIcon?: string;
    cashback?: number;
    source?: 'trust' | 'yandexpay';
};

export type IOrderRefund = {
    items?: IOrderItem[];
    total: number;
    currency: string;
    paymentMethod: string;
    account?: string;
};

export type IOrderItem = {
    nds?: string;
    name: string;
    price: number;
    currency?: string;
    hidden?: boolean;
    alias?: string;
};

export type IPayment = {
    method: string;
    price: number;
    account?: string;
    currency?: string;
};

export type IOrderList = {
    edges: IOrder[];
    pageInfo: IPageInfo;
};

export type IOrderListNext = {
    order_id_keyset: number;
    created_keyset: IsoDateTimeShift;
};

export type IOrderQuery = {
    services?: IService[];
    list: IOrderList;
};

export type IPageInfo = {
    endCursor?: string | null;
    startCursor?: string | null;
    hasNextPage: boolean;
    hasPreviousPage: boolean;
};
