const cp = require('child_process');
const path = require('path');

const LoadablePlugin = require('@loadable/webpack-plugin');
const StatoscopeWebpackPlugin = require('@statoscope/webpack-plugin').default;
const { checkModules, buildExcludeRegexp } = require('are-you-es5');
const MiniCssExtractPlugin = require('mini-css-extract-plugin');
const getCSSModuleLocalIdent = require('react-dev-utils/getCSSModuleLocalIdent');
const TsconfigPathsPlugin = require('tsconfig-paths-webpack-plugin');
const webpack = require('webpack');

const { bucket, service } = require('./.config/static');

const areYouEs5 = checkModules({
    path: '',
    checkAllNodeModules: true,
    ignoreBabelAndWebpackPackages: true,
});
/** Returns the regexp excluding all es6 modules */
const es6ExcludeRegexp = buildExcludeRegexp(areYouEs5.es6Modules);

class MiniCssExtractPluginCleanup {
    shouldPickStatChild(child) {
        return child.name.indexOf('mini-css-extract-plugin') !== 0;
    }

    apply(compiler) {
        compiler.hooks.done.tap('CleanUpStatsPlugin', (stats) => {
            const children = stats.compilation.children;

            if (Array.isArray(children)) {
                // eslint-disable-next-line no-param-reassign
                stats.compilation.children = children.filter((child) =>
                    this.shouldPickStatChild(child),
                );
            }
        });
    }
}

const cssRegex = /\.css$/;
const cssModuleRegex = /\.module\.css$/;
const nodeModulesRegex = /node_modules/;
const scope = '/order-history';

module.exports = ({ server, analyze, production, s3static } = {}) => {
    const publicPath = s3static
        ? `https://yastatic.net/s3/${bucket}/${service}/`
        : scope + '/assets/';

    const getStyleLoaders = (cssOptions) => [
        ...(server ? [] : [MiniCssExtractPlugin.loader]),
        {
            loader: 'css-loader',
            options: Object.assign(
                {
                    importLoaders: production ? 1 : 0,
                    onlyLocals: server,
                },
                cssOptions,
            ),
        },
        {
            loader: 'postcss-loader',
            options: {
                ident: 'postcss',
            },
        },
    ];

    return {
        entry: {
            [server ? 'index' : 'main']: path.resolve(__dirname, 'src', server ? 'root' : 'client'),
            ...(!server && { error: path.resolve(__dirname, 'src/client/error-entry.ts') }),
            ...(!server && { polyfill: path.resolve(__dirname, 'src/client/polyfill-entry.ts') }),
        },
        target: server ? 'node' : 'web',
        externals: server ? ['react', 'react-dom', 'react-helmet', '@loadable/component'] : [],
        output: {
            publicPath,
            filename: `${server ? '[name]' : '[name].[contenthash]'}.js`,
            libraryTarget: server ? 'umd' : 'var',
            path: server
                ? path.resolve(__dirname, 'dist/root')
                : path.resolve(__dirname, 'public/assets'),
        },
        optimization: {
            splitChunks: {
                chunks: server
                    ? 'initial'
                    : (chunk) => {
                          // не разбиваем error на чанки
                          return chunk.name !== 'error' && chunk.name !== 'polyfill';
                      },
            },
            // runtimeChunk: server ? undefined : {
            //     name: entrypoint => `runtime-${entrypoint.name}`,
            // },
        },
        resolve: {
            extensions: ['.js', '.jsx', '.ts', '.tsx'],
            plugins: [
                new TsconfigPathsPlugin({
                    configFile: path.resolve(__dirname, 'tsconfig.client.json'),
                }),
            ],
            alias: {
                'react-dom': require.resolve('react-dom'),
                react: require.resolve('react'),
            },
        },
        plugins: [
            ...(server
                ? [
                      new webpack.optimize.LimitChunkCountPlugin({
                          maxChunks: 1,
                      }),
                  ]
                : [
                      new LoadablePlugin({
                          writeToDisk: true,
                          filename: 'loadable-stats.json',
                      }),
                      new MiniCssExtractPlugin({
                          ignoreOrder: true,
                          filename: '[name].[contenthash].css',
                      }),
                      new MiniCssExtractPluginCleanup(),
                      {
                          apply: (compiler) => {
                              compiler.hooks.afterEmit.tapAsync('GenerateSWPlugin', (_, cb) => {
                                  cp.exec(
                                      'npm run generateSW' + (production ? '' : ' -- --dev'),
                                      (err, stdout, stderr) => {
                                          process.stdout.write('test\n');
                                          if (err && stderr) {
                                              process.stdout.write(stderr);
                                          }
                                          cb();
                                      },
                                  );
                              });
                          },
                      },
                  ]),
            new webpack.DefinePlugin({
                __SERVER__: server,
                __DEV__: !production,
                ...(!server
                    ? {}
                    : {
                          navigator: '{}',
                      }),
            }),
            ...(analyze ? [new StatoscopeWebpackPlugin()] : []),
        ],
        module: {
            rules: [
                {
                    test: /\.(js?|tsx?)$/,
                    exclude: es6ExcludeRegexp,
                    use: {
                        loader: 'babel-loader',
                        options: {
                            presets: [
                                [
                                    '@babel/env',
                                    {
                                        useBuiltIns: server ? 'usage' : 'entry',
                                        corejs: 3,
                                        targets: server
                                            ? {
                                                  node: 'current',
                                              }
                                            : {
                                                  browsers: ['> 0.5%', 'ie 11', 'not dead'],
                                                  uglify: true,
                                              },
                                    },
                                ],
                            ],
                        },
                    },
                },
                {
                    test: cssRegex,
                    exclude: [cssModuleRegex, nodeModulesRegex],
                    use: getStyleLoaders(),
                    // Don't consider CSS imports dead code even if the
                    // containing package claims to have no side effects.
                    // Remove this when webpack adds a warning or an error for this.
                    // See https://github.com/webpack/webpack/issues/6571
                    sideEffects: true,
                },
                // Adds support for CSS Modules (https://github.com/css-modules/css-modules)
                // using the extension .module.css
                {
                    test: cssModuleRegex,
                    use: getStyleLoaders({
                        modules: {
                            getLocalIdent: getCSSModuleLocalIdent,
                        },
                    }),
                },
                {
                    test: cssRegex,
                    exclude: [cssModuleRegex],
                    include: nodeModulesRegex,
                    loader: server ? 'null-loader' : undefined,
                    use: server ? undefined : getStyleLoaders(),
                },
                {
                    test: /\.svg$/,
                    issuer: /\.(t|j)sx?$/,
                    use: [
                        {
                            loader: 'file-loader',
                            options: {
                                emitFile: !server,
                                publicPath: publicPath + '_/',
                                outputPath: './_',
                                name: 'fl.[contenthash].[ext]',
                            },
                        },
                    ],
                },
                {
                    test: /\.(jpe?g|png|ttf|gif|eot|svg|woff(2)?)(\?[a-z0-9=&.]+)?$/,
                    issuer: cssRegex,
                    loader: 'base64-inline-loader',
                },
            ],
        },
    };
};
