import argparse
import os
import shutil
import subprocess


HOSTS_WITH_CUSTOM_HOSTNAMES = {
    'cerevra-dev-xenial.passport': 'cerevra-dev-xenial.passport.yandex.net',
    'dimaandreev-dev': 'dimaandreev-dev-bionic.passport.yandex.net',
    'kakabba-dev': 'kakabba-dev-bionic.passport.yandex.net',
    'gayurgin-dev': 'gayurgin-dev-bionic.passport.yandex.net',
    'skeef79-dev': 'skeef79-dev-bionic.passport.yandex.net',
    'pass-dev': 'pass-dev.passport.yandex.net',
    'pass-yateam-test-f1': 'pass-yateam-test-f1.passport.yandex.net',
    'pass-yateam-test-s1': 'pass-yateam-test-s1.passport.yandex.net',
    'sandbox-test': 'sandbox-test',
}


def get_host_ids(cfg_dir):
    res = {}

    for line in open(os.path.join(cfg_dir, 'hostid.txt')):
        line = line.strip()
        if len(line) != 0:
            tokens = line.split('=')
            res[tokens[0]] = tokens[1]

    return res


def build_host(cfg_dir, out_dir, env, host_id, host, synt_host=None):
    if synt_host is None:
        synt_host = host

    subprocess.check_call(
        [
            os.path.join(cfg_dir, 'buildconf'),
            synt_host,
            'blackbox',
            '',
            host_id,
        ],
        env=env,
    )

    hostname = HOSTS_WITH_CUSTOM_HOSTNAMES.get(host, '%s.sezam.yandex.net' % host)
    shutil.copy(
        os.path.join(cfg_dir, 'blackbox.daemon'),
        os.path.join(out_dir, hostname),
    )


def main():
    parser = argparse.ArgumentParser(description='build configs for blackbox')
    parser.add_argument('--xmllint_path', type=str, required=True)
    parser.add_argument('--xsltproc_path', type=str, required=True)
    parser.add_argument('--cfg_dir', type=str, required=True)
    parser.add_argument('--out_dir', type=str, required=True)
    parser.add_argument('--remove_cfg_dir', action='store_true', default=False, help='required for debian.json')
    args = parser.parse_args()

    assert os.path.isdir(args.cfg_dir)
    assert os.path.exists(os.path.join(args.xmllint_path, 'xmllint'))
    assert os.path.exists(os.path.join(args.xsltproc_path, 'xsltproc'))

    host_ids = get_host_ids(args.cfg_dir)
    env = os.environ.copy()
    env['PATH'] = ':'.join(
        [
            env.get('PATH', ''),
            os.path.abspath(args.xmllint_path),
            os.path.abspath(args.xsltproc_path),
        ]
    )

    for host in os.listdir(args.cfg_dir):
        dir_path = os.path.join(args.cfg_dir, host)
        if not os.path.isdir(dir_path) or host in ['db_hosts_repo', 'dev']:
            continue

        build_host(
            cfg_dir=args.cfg_dir,
            out_dir=args.out_dir,
            env=env,
            host_id=host_ids[host],
            host=host,
        )

    for host in os.listdir(os.path.join(args.cfg_dir, 'dev')):
        dir_path = os.path.join(args.cfg_dir, 'dev', host)
        if not os.path.isdir(dir_path):
            continue

        synt_host = 'dev/' + host
        build_host(
            cfg_dir=args.cfg_dir,
            out_dir=args.out_dir,
            env=env,
            host_id=host_ids[synt_host],
            host=host,
            synt_host=synt_host,
        )

    if args.remove_cfg_dir:
        shutil.rmtree(args.cfg_dir)


if __name__ == '__main__':
    main()
