#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import urllib.request
import sys
import socket
import subprocess
import json
from xml.dom import minidom
import argparse


CONDUCTOR_API_URL = "https://c.yandex-team.ru/api-cached/"

REPO_HOSTGROUPS = {
    "testing": {
        "passport_central": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "passport_shard1": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "passport_shard2": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "oauth_central": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "oauth_shard1": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "oauth_shard2": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "sess_kill": "passport-sezam-ufo-test",
        "kolmogor": "passport-sezam-db-badauth-test",
    },
    "production": {
        "passport_central": "passport-db-central-iva_slave,passport-db-central-myt_slave,passport-db-central-sas_slave,passport-db-central-vla_slave",
        "passport_shard1": "passport-db-sh1-iva_slave,passport-db-sh1-myt_slave,passport-db-sh1-sas_slave,passport-db-sh1-vla_slave",
        "passport_shard2": "passport-db-sh2-iva_slave,passport-db-sh2-myt_slave,passport-db-sh2-sas_slave,passport-db-sh2-vla_slave",
        "passport_shard3": "passport-db-sh3-iva_slave,passport-db-sh3-myt_slave,passport-db-sh3-sas_slave,passport-db-sh3-vla_slave",
        "passport_shard4": "passport-db-sh3-iva_slave,passport-db-sh3-myt_slave,passport-db-sh3-sas_slave,passport-db-sh3-vla_slave",
        "oauth_central": "passport-oauth-db-central-iva_slave,passport-oauth-db-central-myt_slave,passport-oauth-db-central-sas_slave,passport-oauth-db-central-vla_slave",
        "oauth_shard1": "passport-oauth-db-central-iva_slave,passport-oauth-db-central-myt_slave,passport-oauth-db-central-sas_slave,passport-oauth-db-central-vla_slave",
        "oauth_shard2": "passport-oauth-db-sh2-iva_slave,passport-oauth-db-sh2-myt_slave,passport-oauth-db-sh2-sas_slave,passport-oauth-db-sh2-vla_slave",
        "sess_kill": "passport-sezam-ufo-stable-prod",
        "kolmogor": "passport-sezam-db-badauth-stable-prod",
    },
    "yateam": {
        "passport_central": "passport-db-yateam_slave",
        "passport_shard1": "passport-db-yateam_slave",
        "oauth_central": "passport-db-yateam_slave",
        "oauth_shard1": "passport-db-yateam_slave",
        "kolmogor": "passport-sezam-db-badauth-stable-yateam",
        "perimeter": "passport-perimeter-stable",
    },
    "yateam_testing": {
        "passport_central": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "passport_shard1": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "oauth_central": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "oauth_shard1": "passport-db-test-sas_slave,passport-db-test-vla_slave",
        "kolmogor": "passport-sezam-db-badauth-test",
        "perimeter": "passport-perimeter-test",
    },
}

HOST_CACHE = {}
CONDUCTOR_OAUTH_TOKEN = ""
USE_SSH = True

TEMPLATE_DB_HOST_MYSQL = "<db_host dc=\"%s\" fqdn=\"%s\"%s>%s</db_host>"
TEMPLATE_DB_HOST_HTTP = "<db_host dc=\"%s\" fqdn=\"%s\"%s>[%s]</db_host>"
TEMPLATE_WEIGHT = "weight=\"%d\""
TEMPLATE_C_CPU_TAG = "hw:processor_count:%d"
FILE_NAME_TEMPLATE = "db_hosts_repo/%s.conf"


def get_ip_from_host(fqdn):
    return socket.getaddrinfo(fqdn, 0, socket.AF_INET6)[0][4][0]


def get_hosts_from_c_group(c_group):
    try:
        f = urllib.request.urlopen(CONDUCTOR_API_URL + "/groups2hosts/" + c_group)
        hosts = f.read().decode('utf-8').split()
        hosts.sort()
        return hosts
    except urllib.error.HTTPError as e:
        print(e)


def get_host_c_tags(fqdn):
    try:
        f = urllib.request.urlopen(CONDUCTOR_API_URL + "/get_host_tags/" + fqdn + "?recursion=false&format=json")
        return json.loads(f.read().decode('utf-8'))
    except urllib.error.HTTPError as e:
        print(e)


def get_host_c_info(fqdn):
    try:
        f = urllib.request.urlopen(CONDUCTOR_API_URL + "/hosts/" + fqdn + "?format=json")
        return json.loads(f.read().decode('utf-8'))
    except urllib.error.HTTPError as e:
        print(e)


def update_cpu_c_tag(fqdn, cpunum):
    # 1. Get tag: "https://c.yandex-team.ru/api/v1/tags/hw:processor_count:32", get tag id
    # 1a. If 404: create tag /api/v1/tags/ , get 201 and "id": 1234
    # 2. Get host tags: /api/v1/hosts/cnt-dbs-s1.passport.yandex.net/tags
    # 3. Check tag, if differs: Remove all tags like "hw:processor_count:"
    # 4. Add new tag by id
    pass


def process_host_group(host_group, type="mysql"):
    print("Processing group2host: %s" % host_group)
    xml = ""
    hosts = get_hosts_from_c_group(host_group)

    for host in hosts:
        print("Processing host: %s" % host)
        dc = ""
        weight = 0

        if host not in HOST_CACHE:
            ip = get_ip_from_host(host)
            c_hosts_info = get_host_c_info(host)[0]
            dc = c_hosts_info["root_datacenter"]

            # Get CPU number from ssh
            cpunum = 0
            if USE_SSH:
                p = subprocess.run(["ssh", "-o ConnectTimeout=5", host, "nproc"], capture_output=True)
                if p.returncode == 0:
                    cpunum = int(p.stdout.decode('UTF-8').strip())
                    # DRAFT:
                    # if UPDATE_CONDUCTOR:
                    #   update_cpu_c_tag(host, cpunum)
                else:
                    print("Error in ssh to host %s: %s" % (host, p.stderr.decode('UTF-8')))
            # DRAFT: Get CPU number from Conductor
            # else:
            #   c_tags = get_host_c_tags(host)
            #   for tag in c_tags:
            #       print(tag)
            #       if "hw:processor_count:" in tag:
            #           cpunum = tag.split(":")[-1]
            #           print(cpunum)

            # Fill host cache
            HOST_CACHE[host] = {"cpunum": cpunum, "ip": ip, "c_info": c_hosts_info}
        else:
            print("Cache hit")
            ip = HOST_CACHE[host]["ip"]
            dc = HOST_CACHE[host]["c_info"]["root_datacenter"]
            cpunum = HOST_CACHE[host]["cpunum"]

        if cpunum == 32:
            weight = 10
        elif cpunum == 40:
            weight = 13
        elif cpunum == 56:
            weight = 17
        elif cpunum == 80:
            weight = 32
        elif cpunum == 104:
            weight = 42
        elif cpunum == 128:
            weight = 52
        elif cpunum < 32:
            weight = None
        else:
            raise Exception("Unknown num of CPU cores: %d" % cpunum)

        if type == "mysql":
            xml += TEMPLATE_DB_HOST_MYSQL % (dc, host, (" " + TEMPLATE_WEIGHT % weight) if weight else "", ip)
        elif type == "http":
            xml += TEMPLATE_DB_HOST_HTTP % (dc, host, (" " + TEMPLATE_WEIGHT % weight) if weight else "", ip)
        else:
            raise Exception("Unknown DB type: %s" % type)

    return xml


# Generate xml
def gen_dbpool(environments):
    for env in environments:
        print("Processing env: %s" % env)
        xml = "<db_hosts>"

        for db_group in REPO_HOSTGROUPS[env]:
            xml += "<%s>" % db_group
            print("Processing group: %s: %s" % (db_group, REPO_HOSTGROUPS[env][db_group]))
            if "passport_" in db_group or "oauth_" in db_group:
                db_type = "mysql"
            else:
                db_type = "http"
            xml += process_host_group(REPO_HOSTGROUPS[env][db_group], db_type)
            xml += "</%s>" % db_group.split(" ")[0]

        xml += "</db_hosts>"
        print("Finished")

        try:
            xml = minidom.parseString(xml).toprettyxml(indent="    ")
            print(xml)
            with open(FILE_NAME_TEMPLATE % env, "w") as f:
                f.write(xml)
        except Exception as e:
            print(e)


def main():
    parser = argparse.ArgumentParser(
        description="Generates db host repository", formatter_class=argparse.RawTextHelpFormatter
    )
    parser.add_argument(
        '--run',
        dest='run',
        action='store_true',
        help='Run gathering info and build host repository.\nWARNING: You should be abble to ssh to productions hosts.',
    )
    parser.add_argument(
        '--section',
        dest='section',
        action='store',
        nargs="+",
        choices=REPO_HOSTGROUPS.keys(),
        help='Run gathering info for selected sections.\nDEFINED SECTIONS: %s' % ", ".join(REPO_HOSTGROUPS.keys()),
    )
    parser.add_argument(
        '--use-ssh',
        dest='use_ssh',
        action='store_true',
        help='Use SSH to host to gather info about CPU nubmers.\nWARNING: You should be abble to ssh to productions hosts.',
    )
    # parser.add_argument('--update-conductor', dest='update_conductor', action='store_true', help='Update Conductor tag with CPU nubmers from SSH')
    # parser.add_argument('--oauth-token', dest='c_oauth_token', type=argparse.FileType('r'), help='Conductor OAuth token file')
    args = vars(parser.parse_args())

    if len(sys.argv) == 1:
        parser.print_help()
        sys.exit(1)

    if args["use_ssh"]:
        USE_SSH = True

    # if args["update_conductor"] and not args["c_oauth_token"]:
    #    print('Enter OAuth token:')
    #    CONDUCTOR_OAUTH_TOKEN = input()
    # elif args["update_conductor"] and args["c_oauth_token"]:
    #    CONDUCTOR_OAUTH_TOKEN = args["c_oauth_token"].read().strip()

    if args["run"]:
        gen_dbpool(args["section"] if args["section"] else REPO_HOSTGROUPS.keys())
    else:
        parser.print_help()
        sys.exit(1)


if __name__ == '__main__':
    sys.exit(main())
