#!/bin/bash

# This script is used to build Blackbox configuration files
# The idea is to take a config template, apply local per-host settings,
# validate and reformat config to get custom config files for given host.

# expected environment settings:
# PREFIX - string prefix to add to config and log files, to run several instances on single host (not used and tested)
# KEYS_DB_NAME - name of mysql db to load keys from (used in yateam-test)
# for blackbox config also
# BLACKBOX_DAEMON_PORT
# BLACKBOX_MONITORING_PORT
# BLACKBOX_PASSPORTDB_PORT
# BLACKBOX_KOLMOGOR_PORT
# BLACKBOX_SESSKILL_PORT
# BLACKBOX_ROOT_DIRECTORY
# BLACKBOX_LOGS_DIRECTORY
# BLACKBOX_CONFIG_DIRECTORY


# script settings
set -o errexit

# directory to place configs
readonly CONF_DIR=../available

# supported module names
readonly MODULES_LIST="blackbox"

readonly RED='\033[0;31m'
readonly GREEN='\033[0;32m'
readonly NOCOLOR='\033[0m'

# -----------------------
# main script body

cd "$( dirname "${BASH_SOURCE[0]}" )"

# determine current hostname and host_id

HOSTNAME=$( (test -f hostname || ./gethostname) && cat hostname )

if [ -z "$HOSTNAME" ]; then
    echo "Hostname unknown, don't know what to generate!"
    exit 1
fi

BB_HOST_ID=$(grep "${HOSTNAME}=" hostid.txt | tail -1 | cut -d "=" -f 2)
export BB_HOST_ID

if [ -z "${BB_HOST_ID}" ]; then
    echo "Failed to find hostid in hostid.txt for hostname $HOSTNAME"
    exit 1
fi

if [ -n "${PREFIX}" ]; then
    PREFIX_VAR=${PREFIX}-
fi

# prepare directories

if [ ! -d "${CONF_DIR}" ]; then
    mkdir "${CONF_DIR}"
fi

logs_directory=${BLACKBOX_LOGS_DIRECTORY:-/opt/sezam-logs${PREFIX:+-${PREFIX}}}

if [ ! -d "${logs_directory}" ]; then
    mkdir "${logs_directory}"
fi

# generate configs

echo -e "${GREEN}Building configs for $HOSTNAME${NOCOLOR}"

for module in $MODULES_LIST; do
    if [ ! -f "$HOSTNAME/fastcgi.locals.$module" ]; then
        continue
    fi
    ./buildconf "$HOSTNAME" "$module" "$PREFIX"
    mv -f "$module.daemon" "$CONF_DIR/${PREFIX_VAR}$module.daemon"
    echo "Module $module config done."
done

echo -e "${GREEN}Done. Configs for $HOSTNAME generated.${NOCOLOR}"
