#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import urllib.request
from urllib.parse import urlencode
import sys
import json
import re
import argparse
import pprint

CONDUCTOR_API_URL = "https://c.yandex-team.ru/api-cached/"
L3MGR_API_URL = "https://l3-api.tt.yandex-team.ru/api/v1/"

HOST_CACHE = {}
L3MGR_OAUTH_TOKEN = ""

BLACKBOX_SLBS = {
    # WARNING! DANGEROUS SECTION BELOW
    "blackbox-mail.yandex.net": {
        "l3_vs_id": 3155,
        "c_groups": ["passport-sezam-front-stable-prod-dd"],
        "use_weights": True,
    },
    "blackbox.yandex.net": {
        "l3_vs_id": 3157,
        "c_groups": ["passport-sezam-front-stable-prod-farm"],
        "use_weights": True,
    },
    "blackbox.yandex-team.ru": {
        "l3_vs_id": 3156,
        "c_groups": ["passport-sezam-front-stable-yateam"],
        "use_weights": True,
    },
    # END OF DANGEROUS SECTION
    "blackbox-test.yandex.net": {
        "l3_vs_id": 3621,
        "c_groups": ["passport-sezam-front-test"],
        "use_weights": False,
    },
    "blackbox-test.yandex-team.ru": {
        "l3_vs_id": 1340,
        "c_groups": ["passport-sezam-front-yateam-test"],
        "use_weights": False,
    },
    "pass-test.yandex.ru": {
        "l3_vs_id": 1331,
        "c_groups": ["passport-sezam-front-test"],
        "use_weights": False,
    },
    "blackbox-mimino.yandex.net": {
        "l3_vs_id": 2642,
        "c_groups": ["passport-sezam-front-mimino"],
        "use_weights": False,
    },
    "blackbox-stress.yandex.net": {
        "l3_vs_id": 3091,
        "c_groups": ["passport-sezam-front-stress-for-all"],
        "use_weights": False,
    },
    "blackbox-rc.yandex.net": {
        "l3_vs_id": 2637,
        "c_groups": ["passport-sezam-front-rc"],
        "use_weights": False,
    },
}


def get_ip_from_host(fqdn):
    return socket.getaddrinfo(fqdn, 0, socket.AF_INET6)[0][4][0]


def get_hosts_from_c_group(c_group):
    try:
        f = urllib.request.urlopen(CONDUCTOR_API_URL + "/groups2hosts/" + c_group)
        hosts = f.read().decode('utf-8').split()
        hosts.sort()
        return hosts
    except urllib.error.HTTPError as e:
        print(e)


def get_host_l3_weigth(fqdn, l3_weight=10):
    short_hostname = fqdn.split('.', 1)[0]

    if 'test' in short_hostname:
        config_file = 'dev/' + short_hostname + '/fastcgi.locals.blackbox'
    else:
        config_file = short_hostname + '/fastcgi.locals.blackbox'

    try:
        with open(config_file) as f:
            config = f.readlines()
    except IOError:
        print("File not accessible")

    for line in config:
        if 'l3_weight/default_value' in line:
            l3_weight = re.findall(r'l3_weight/default_value==(\d+)', line)[0]

    return int(l3_weight)


def gen_host_list_for_l3(hosts_with_weights):
    result = []
    for host, l3_weight in hosts_with_weights.items():
        if l3_weight > 0:
            result.append('%s weight=%s' % (host, l3_weight))
        else:
            result.append('%s' % host)

    return "\n".join(result)


def push_host_list_to_l3(l3_vs_id, hosts_with_weights, auto_apply=False):
    try:
        req = urllib.request.Request(
            L3MGR_API_URL + 'service/' + str(l3_vs_id) + '/editrs',
            headers={'Authorization': 'OAuth %s' % L3MGR_OAUTH_TOKEN},
            data=urlencode({'groups': gen_host_list_for_l3(hosts_with_weights)}).encode(),
        )

        with urllib.request.urlopen(req) as f:
            data = json.loads(f.read().decode('utf-8'))
            # print(data)

        if data['result'] == 'OK':
            print("Request accepted by L3 API")
            print(
                "Check new config: https://l3.tt.yandex-team.ru/service/%d/config/%d" % (l3_vs_id, data['object']['id'])
            )

            if not auto_apply:
                print("Process? [y/Y=Yes, any=No]:")
                process = input()

            if auto_apply or process.lower() == "y":
                req = urllib.request.Request(
                    L3MGR_API_URL + 'service/' + str(l3_vs_id) + '/config/' + str(data['object']['id']) + '/process',
                    headers={'Authorization': 'OAuth %s' % L3MGR_OAUTH_TOKEN},
                    method="POST",
                )
                with urllib.request.urlopen(req) as f:
                    data = json.loads(f.read().decode('utf-8'))
                    print("L3 API process response: %s" % data)
            else:
                print("New config not applied.")
        else:
            raise Exception(data)
    except Exception as e:
        print("ERROR: L3 API: %s" % str(e))


def main():
    global L3MGR_OAUTH_TOKEN

    parser = argparse.ArgumentParser(
        description="Generates host list for L3 and push it", formatter_class=argparse.RawTextHelpFormatter
    )

    group = parser.add_mutually_exclusive_group()
    group.add_argument(
        '--c_groups',
        dest='c_groups',
        action='store',
        nargs="+",
        help='Conductor groups',
    )
    group.add_argument(
        '--vs',
        dest='vs',
        action='store',
        nargs="?",
        choices=BLACKBOX_SLBS.keys(),
        help='Run RS update for selected VS.',
    )
    parser.add_argument(
        '--l3-apply',
        dest='l3_apply',
        action='store_true',
        default=False,
        help='Push host list to L3 and apply',
    )
    parser.add_argument(
        '--disable-weights',
        dest='disable_weights',
        action='store_true',
        default=False,
        help='Disable adding \'weight\' param for hosts.',
    )
    parser.add_argument(
        '--l3mgr-oauth-token',
        dest='l3mgr_oauth_token',
        type=argparse.FileType('r'),
        help='L3 Mgr OAuth token file',
    )

    args = vars(parser.parse_args())
    # print(args)

    if len(sys.argv) == 1:
        parser.print_help()
        sys.exit(1)

    if args['c_groups']:
        c_groups = args['c_groups']
    elif args['vs']:
        if not args['l3mgr_oauth_token']:
            print("ERROR: In 'vs' mode you must provide oauth token for L3 API")
            return 1
        L3MGR_OAUTH_TOKEN = args["l3mgr_oauth_token"].read().strip()

        print("VS: %s\n%s" % (args['vs'], pprint.pformat(BLACKBOX_SLBS[args['vs']])))
        c_groups = BLACKBOX_SLBS[args['vs']]['c_groups']

    for c_group in c_groups:
        hosts = get_hosts_from_c_group(c_group)
        for host in hosts:
            if args['vs']:
                if BLACKBOX_SLBS[args['vs']]['use_weights']:
                    HOST_CACHE[host] = get_host_l3_weigth(host)
                else:
                    HOST_CACHE[host] = get_host_l3_weigth(host, -1)
            else:
                if args['disable_weights']:
                    HOST_CACHE[host] = get_host_l3_weigth(host, -1)
                else:
                    HOST_CACHE[host] = get_host_l3_weigth(host)

    print("\nPrepared hosts:")
    print(gen_host_list_for_l3(HOST_CACHE))
    print()

    if args['vs']:
        if not args['l3_apply']:
            print("Push hosts list in L3? [y/Y=Yes, any=No]:")
            process = input()

        if args['l3_apply'] or process.lower() == "y":
            push_host_list_to_l3(BLACKBOX_SLBS[args['vs']]['l3_vs_id'], HOST_CACHE, args['l3_apply'])


if __name__ == '__main__':
    sys.exit(main())
