#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import re
import json
import sys
import ipaddress
from passport.infra.daemons.blackbox.config.usernets.nets_provider import NetsProvider
from passport.infra.daemons.blackbox.config.usernets.network_description_filter import NetworkDescriptionFilter

INTERNAL_EXPLICIT_LIST = [
    {
        "object": "_UNIFIEDUSERNETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_DL_INT_BACK_PROD_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_C_YQL_PROD_",
        "reason": "PASSP-37505",
    },
]

ROBOT_EXPLICIT_LIST = [
    {
        "object": "_CHARTS_PROD_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_GENCFG_RTLINE_FRONTEND_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_CREW_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_SEARCHHITMANNETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_PGAASINTERNALNETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_DRIVE_RUNNER_TASK_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_C_KIWI_DEV_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_ZOMBIFICATORNETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_C_TAXI_PRESTABLE_STRONGBOX_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_WIKI_SEARCH_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_GENCFG_SEARCHPRODNETS_ROOT_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_C_TAXI_STRONGBOX_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_MUSIC_SUMMIT_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_THIRIUMNETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_YTHAHNNETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_SDCNETS_",
        "reason": "PASSP-37505",
        "work_rate": 0,
    },
    {
        "object": "_LOCDOC_PROD_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_MARKET_FOR_USERS_BOT_PROD_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_STARTREK_API_PRODUCTION_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_ISEARCH_PROD_NETS_",
        "reason": "PASSP-37505",
    },
    {
        "object": "_ZOMBIKOPALKANETS_",
        "reason": "PASSP-37505",
        "work_rate": 0,
    },
    {
        "object": "_GENCFG_MARKET_PROD_",
        "reason": "PASSP-37505",
        "work_rate": 0,
    },
    {
        "object": "_CLOUD_IL_FW_MGMT_NETS_",
        "reason": "PASSPINFRA-51",
    },
]

ALLOW_LIST = [
    {
        "object": "_YANDEXNETS_",
        "reason": "Allow all Yandex",
    },
    {
        "object": "_SDCAMINEVKANETS_",
        "reason": "Allow SDC garage",
    },
    {
        "object": "213.87.14.149",
        "reason": "SDC external (PASSP-17604)",
    },
    {
        "object": "213.87.14.162",
        "reason": "SDC external (PASSP-17604)",
    },
    {
        "object": "213.87.88.186",
        "reason": "SDC external (PASSP-17604)",
    },
    {
        "object": "31.173.17.40",
        "reason": "SDC external (PASSP-17604)",
    },
    {
        "object": "31.173.21.137",
        "reason": "SDC external (PASSP-17604)",
    },
    {
        "object": "91.224.124.170",
        "reason": "SDC external (PASSP-17604)",
    },
    {
        "object": "redrose.prius.sdc.yandex.net",
        "reason": "SDC external (PASSP-17604)",
    },
]

DENY_LIST = [
    {
        "object": "_YANDEXNETS_",
        "reason": "Forbid all Yandex",
        "work_rate": 0,
    },
    {
        "object": "_C_MAIL_RPOP_",
        "reason": "PASSP-20644",
    },
    {
        "object": "_TRBOSRVNETS_",
        "reason": "PASSP-20644",
    },
    {
        "object": "_CLOUD_YANDEX_CLIENTS_",
        "reason": "PASSP-31974",
    },
    {
        "object": "_SHAREDWORKSTATIONNETS_",
        "reason": "Shared workstations do not have client certificates",
    },
    {
        "object": "_2FATERMINALSNETS_",
        "reason": "Shared terminal servers do not have client certificates",
    },
    {
        "object": "_TAXI_TERM2FA_NETS_",
        "reason": "Shared terminal servers do not have client certificates",
    },
    {
        "object": "_POLAR_CC_TERMSRV_",
        "reason": "Shared terminal servers do not have client certificates",
    },
    {
        "object": "_MARKETWHLEPROUSNETS_",
        "reason": "NOCREQUESTS-19817",
    },
    {
        "object": "_GUESTSNETS_",
        "reason": "Deny from Guest networks",
    },
    {
        "object": "_WINTERMNETS_",
        "reason": "Shared terminal servers do not have client certificates",
    },
    {
        "object": "_PUBLICEVENTSTREAMINGNETS_",
        "reason": "video streaming nets are not trusted",
    },
    {
        "object": "_DESKTOPSPACE_",
        "reason": "PASSP-36609",
    },
    {
        "object": "_PRINTERS_",
        "reason": "PASSP-36609",
    },
    {
        "object": "_DATASCHOOLHADOOP_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_1CNETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_HYPERVSRVNETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_ADDC_PRJ_NETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_ADCS_NETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_ADFS_NETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_DSSNETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_WINADMINSANDBOXNETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_ZOOBROWSERNETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_ENDOCSNETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_ENDOCSTERM_NETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_BACKUP_CA_NETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_DIRECT_TERM2FA_NETS_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_BANKCLMSSQLPUB_",
        "reason": "PASSP-36824",
        "work_rate": 0,
    },
    {
        "object": "_MOBCERTNETS_",
        "reason": "PASSP-37211",
    },
]

ROBOTS_LIST = [
    {
        "object": "_SELENIUMGRIDNETS_",
        "reason": "PASSP-30610",
    },
    {
        "object": "_WIKI_SEARCH_NETS_",
        "reason": "PASSP-34134",
    },
    {
        "object": "_NIRVANANETS_",
        "reason": "PASSP-34134",
    },
    {
        "object": "_YTNETS_",
        "reason": "PASSP-34134",
    },
    {
        "object": "_CMSEARCHNETS_",
        "reason": "PASSP-34134",
    },
    {
        "object": "_NIRVANA_VANILLA_JOB_NETS_",
        "reason": "PASSP-36259",
        "work_rate": 0,
    },
    {
        "object": "_NIRVANA_REGULAR_JOB_NETS_",
        "reason": "PASSP-36259",
        "work_rate": 0,
    },
    {
        "object": "_NIRVANA_SPECIAL_JOB_NETS_",
        "reason": "PASSP-36259",
        "work_rate": 0,
    },
    {
        "object": "_NIRVANA_BASIC_NETS_",
        "reason": "PASSP-36259",
        "work_rate": 0,
    },
    {
        "object": "_KOLHOZ_AGENT_NETS_",
        "reason": "PASSP-36582",
    },
    {
        "object": "_MOBTESTNETS_",
        "reason": "PASSP-36582",
    },
    {
        "object": "_NOCNALIVKANETS_",
        "reason": "PASSP-36609",
    },
    {
        "object": "_DWDMMGMT_",
        "reason": "PASSP-36609",
    },
    {
        "object": "_THERMODENETS_",
        "reason": "PASSP-36609",
    },
    {
        "object": "_CCTVDCNETS_",
        "reason": "PASSP-36609",
    },
    {
        "object": "_CIPTNETS_",
        "reason": "PASSP-36741",
    },
    {
        "object": "_HITROBOTNETS_",
        "reason": "PASSP-37251",
    },
    {
        "object": "_AZURE_NETS_",
        "reason": "PASSP-37937",
    },
    {
        "object": "_AWS_NETS_",
        "reason": "PASSP-37937",
    },
]


def is_ip(ip):
    try:
        ipaddress.ip_address(ip)
        return True
    except:
        return False


def gen_list(object_list, nets_provider, network_description_filter):
    macro_dictionary = {}

    for item in object_list:
        macro_name = item["object"]
        percent = item.get("work_rate") if "work_rate" in item else 100
        ip_to_rate_dictionary = {}
        ip_adresses = []

        if re.match("_[A-Z0-9_]*_", macro_name):
            print("Item: \"%s\" is processing as Macro" % macro_name)
            for net in nets_provider.get_networklist_from_hbf(macro_name):
                if re.match("[a-zA-Z.]{3,255}", net):
                    ip_adresses += nets_provider.get_server_ips2(net)
                else:
                    ip_adresses += [net]
        elif re.match("[a-zA-Z.]{3,255}", macro_name):
            print("Item: \"%s\" is processing as Host" % macro_name)
            ip_adresses += nets_provider.get_server_ips2(macro_name)
        elif is_ip(macro_name):
            print("Item: \"%s\" is processing as IP address" % macro_name)
            ip_adresses += [macro_name]
        else:
            print("ERROR: Unknown how to process item: \"%s\"" % macro_name)
            return {}

        for ip_adress in ip_adresses:
            ip_to_rate_dictionary[ip_adress] = percent

        macro_dictionary[macro_name] = ip_to_rate_dictionary

    return network_description_filter.make_unique_rates(macro_dictionary)


def build_network_description_array(network_dictionary):
    network_description_array = []

    for macro_name, ip_dictionary in network_dictionary.items():
        for ip, work_rate in ip_dictionary.items():
            network_description = {"macro_name": macro_name, "ip": ip}
            if work_rate != 100:
                network_description["work_rate"] = work_rate
            network_description_array.append(network_description)

    return network_description_array


def dump_json_file(filename, internal_explicit, robot_explicit, allow_nets, deny_nets, robots_nets):
    print("Write result to \"%s\"" % filename)

    doc = {
        "internal_explicit": build_network_description_array(internal_explicit),
        "robot_explicit": build_network_description_array(robot_explicit),
        "allow": build_network_description_array(allow_nets),
        "deny": build_network_description_array(deny_nets),
        "robots": build_network_description_array(robots_nets),
    }
    with open(filename, "w") as f:
        f.write(json.dumps(doc, indent=4, sort_keys=True) + "\n")


def main():
    nets_provider = NetsProvider()
    network_description_filter = NetworkDescriptionFilter()

    print("Process INTERNAL_EXPLICIT_LIST")
    internal_explicit = gen_list(INTERNAL_EXPLICIT_LIST, nets_provider, network_description_filter)

    print("Process ROBOT_EXPLICIT_LIST")
    robot_explicit = gen_list(ROBOT_EXPLICIT_LIST, nets_provider, network_description_filter)

    print("Process Allow list")
    allow_nets = gen_list(ALLOW_LIST, nets_provider, network_description_filter)

    print("Process Deny list")
    deny_nets = gen_list(DENY_LIST, nets_provider, network_description_filter)

    print("Process Robot list")
    robots_nets = gen_list(ROBOTS_LIST, nets_provider, network_description_filter)

    dump_json_file(
        filename="usernets.json",
        internal_explicit=internal_explicit,
        robot_explicit=robot_explicit,
        allow_nets=allow_nets,
        deny_nets=deny_nets,
        robots_nets=robots_nets,
    )


if __name__ == '__main__':
    sys.exit(main())
