#include "async_db_writer.h"

#include "kolmogor.h"

#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/datetime/base.h>
#include <util/string/cast.h>
#include <util/system/thread.h>

namespace NPassport::NBb {
    TAsyncDbWriter::TAsyncDbWriter(ui32 workerCount,
                                   NDbPool::TDbPool* dbpool,
                                   ui32 queueLimit,
                                   const TDuration writeTimeout)
        : Db_(dbpool)
        , QueueLimit_(queueLimit)
        , WriteTimeout_(writeTimeout)
    {
        Y_ENSURE(workerCount, "Invalid AsyncDbWriter params");
        Y_ENSURE(dbpool, "Invalid AsyncDbWriter params");
        Y_ENSURE(queueLimit, "Invalid AsyncDbWriter params");

        WriterId_ = "AsyncDbWriter (DSN: " + Db_->GetDbInfo().Serialized + ")";
        Threads_.reserve(workerCount);

        unsigned n = 0;

        try {
            do {
                Threads_.push_back(std::thread([this, n]() { this->Writer("bb_async_wr_" + IntToString<10>(n)); }));
            } while (++n < workerCount);

        } catch (...) {
            TLog::Error("%s: failed to initialize thread #%d, stopping", WriterId_.c_str(), n);
            Stop();
            throw;
        }

        TLog::Info("%s: started %d workers", WriterId_.c_str(), n);
    }

    TAsyncDbWriter::~TAsyncDbWriter() {
        try {
            Stop();
        } catch (...) {
        }
    }

    bool TAsyncDbWriter::Append(const TString& request) {
        std::unique_lock sl(Mutex_);
        if (Stopped_) {
            return false;
        }

        ui32 queueSize = RequestQueue_.size();

        if (queueSize >= QueueLimit_) {
            TLog::Warning("%s: request queue is full, ignored request '%s'",
                          WriterId_.c_str(),
                          request.c_str());
            return false;
        }

        RequestQueue_.push(request);
        RequestQueueSize_ = RequestQueue_.size();

        Condition_.notify_one();

        return true;
    }

    void TAsyncDbWriter::Stop() {
        if (Stopped_) {
            return;
        }

        std::unique_lock sl(Mutex_);

        Stopped_ = true;

        // clear queue
        while (!RequestQueue_.empty()) {
            RequestQueue_.pop();
        }

        Condition_.notify_all();
        sl.unlock();

        for (unsigned i = 0; i < Threads_.size(); ++i) {
            Threads_[i].join();
        }
        Threads_.clear();

        TLog::Info("%s: all workers stopped", WriterId_.c_str());
    }

    void TAsyncDbWriter::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(RequestQueueSize_);
    }

    void TAsyncDbWriter::Writer(TString name) {
        TThread::SetCurrentThreadName(name.c_str());

        TString error;
        TString response;

        while (true) {
            try {
                std::unique_lock sl(Mutex_);
                while (RequestQueue_.empty()) {
                    Condition_.wait(sl);
                    if (Stopped_) {
                        return;
                    }
                }
                TString query = std::move(RequestQueue_.front());
                RequestQueue_.pop();
                RequestQueueSize_ = RequestQueue_.size();
                sl.unlock();

                // send a query to server
                if (!Db_->IsOk()) {
                    throw yexception() << "DBpool not feeling good, query ignored. Query='" << query << "'";
                }

                NDbPool::TBlockingHandle sqlh(*Db_);

                NDbPool::TQuery httpQuery = TKolmogor::CreateIncQuery(std::move(query));
                const TString path = httpQuery.Query();

                if (!NDbPool::NUtils::FetchBodyFromHttpResult(
                        sqlh.Query(std::move(httpQuery), WriteTimeout_),
                        path,
                        response,
                        error))
                {
                    throw yexception() << error << ". " << response;
                }
            } catch (const std::exception& e) {
                TLog::Warning("%s: Exception processing queued query: %s", WriterId_.c_str(), e.what());
            } catch (...) {
                TLog::Error("%s: Unknown exception processing queued query", WriterId_.c_str());
            }
        }
    }
}
