#include "facade.h"

#include "async_db_writer.h"
#include "helper.h"
#include "utils.h"

#include <passport/infra/daemons/blackbox/src/ip/ipacl_white_list.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/dbpool/destination.h>
#include <passport/infra/libs/cpp/tvm/common/service_tickets.h>
#include <passport/infra/libs/cpp/tvm/dbpool/create_service_ticket_opt.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

namespace NPassport::NBb {
    TBadauthFacade::TBadauthFacade(std::shared_ptr<NTvmAuth::TTvmClient> tvmClient)
        : TvmClient_(std::move(tvmClient))
    {
    }

    TBadauthFacade::~TBadauthFacade() = default;

    void TBadauthFacade::Init(const NXml::TConfig& config,
                              const TString& xpathPrefix,
                              std::shared_ptr<NDbPool::TDbPoolCtx> ctx) {
        InitBadAuthWhiteList(config, xpathPrefix);
        InitBadauthSettings(config, xpathPrefix);
        InitKolmogor(config, xpathPrefix, ctx);
    }

    bool TBadauthFacade::IsWhitelisted(const TString& ip) const {
        if (ip.empty()) {
            return false;
        }

        NUtils::TIpAddr addr;
        if (!addr.Parse(ip)) {
            return false;
        }

        return BadauthWhitelist_->Find(addr).get();
    }

    TBadauthHelper TBadauthFacade::FactoryWrapper(const TString& login,
                                                  const TString& userIp,
                                                  const TString& passwdHash,
                                                  const TString& authType) const {
        TBadauthHelper ins(KolmogorDb_.get(),
                           KolmogorWriter_.get(),
                           *BadauthSettings_);
        ins.SetLogin(login);
        NUtils::TIpAddr ip(userIp);
        ins.UserIp_ = ip.ToString();
        ins.HalfIp_ = ip.ToStringHalfV6();
        if (ins.HalfIp_ == ins.UserIp_) {
            // if userip if half zeroes don't increment the same counter twice
            ins.HalfIp_.clear();
        }
        ins.PasswdHash_ = passwdHash;
        ins.AuthType_ = authType;

        return ins;
    }

    void TBadauthFacade::AddUnistat(NUnistat::TBuilder& builder) const {
        if (KolmogorWriter_) {
            KolmogorWriter_->AddUnistat(builder);
        }
    }

    NDbPool::TDbPool* TBadauthFacade::GetKolmogor() const {
        return KolmogorDb_.get();
    }

    void TBadauthFacade::InitBadAuthWhiteList(const NXml::TConfig& config, const TString& xpathPrefix) {
        BadauthWhitelist_ = std::make_unique<TIpAclWhitelist>();

        std::vector<TString> entries = config.SubKeys(xpathPrefix + "/badauth/white_list/entry");

        if (entries.empty()) {
            TLog::Info("BlackBox: Badauth whitelist not configured");
            return;
        }

        for (const TString& en : entries) {
            TString entry = config.AsString(en + "/@id", TStrings::EMPTY);
            TString name = config.AsString(en + "/@comment", TStrings::EMPTY);
            if (!entry.empty()) {
                BadauthWhitelist_->AddEntry(entry, name);
            }
        }

        BadauthWhitelist_->Print();
    }

    void TBadauthFacade::InitBadauthSettings(const NXml::TConfig& config, const TString& xpathPrefix) {
        BadauthSettings_ = std::make_unique<TBadauthSettings>();

        const TString kolmoPath = xpathPrefix + "/badauth/kolmogor_settings";

        BadauthSettings_->SpaceNameCommon = config.AsString(kolmoPath + "/db_name", "badauth");
        BadauthSettings_->SpaceNameShort = config.AsString(kolmoPath + "/db_short_name", "badauth_short");
        BadauthSettings_->SpaceNameUniq = config.AsString(kolmoPath + "/db_name_for_uniq", "badauth_uniq");
        BadauthSettings_->PwdLength = config.AsNum<size_t>(xpathPrefix + "/badauth/pwd_length_for_short_ip_counter", 16);

        const TString limitsPath = xpathPrefix + "/badauth/limits";

        if (!config.Contains(limitsPath)) {
            TLog::Info("Badauth limits: OFF");
            return;
        }

        BadauthSettings_->RepeatedRequestLimit = config.AsInt(limitsPath + "/repeated_request_limit", 0);
        BadauthSettings_->LoginPwdIpAuthHr = config.AsInt(limitsPath + "/login_pwd_ip_authtype_hour", 0);
        BadauthSettings_->LoginIpHr = config.AsInt(limitsPath + "/login_ip_hour", 0);
        BadauthSettings_->PwdIpHr = config.AsInt(limitsPath + "/pwd_ip_hour", 0);
        BadauthSettings_->LoginHrCombined = config.AsInt(limitsPath + "/login_hour_combined", 0);
        BadauthSettings_->PwdHrCombined = config.AsInt(limitsPath + "/pwd_hour_combined", 0);
        BadauthSettings_->IpHrCombined = config.AsInt(limitsPath + "/ip_hour_combined", 0);
        BadauthSettings_->IpHr = config.AsInt(limitsPath + "/ip_hour", 0);
        BadauthSettings_->IpMin = config.AsInt(limitsPath + "/ip_minute", 0);
        BadauthSettings_->AuthlogLimit = config.AsInt(limitsPath + "/authlog_limit", 0);

        TLog::Info("Badauth limits: %s", BadauthSettings_->Str().c_str());
    }

    void TBadauthFacade::InitKolmogor(const NXml::TConfig& config,
                                      const TString& xpathPrefix,
                                      std::shared_ptr<NDbPool::TDbPoolCtx> ctx) {
        const TString dbPath = xpathPrefix + "/badauth/kolmogor_settings";

        if (!config.Contains(dbPath)) {
            TLog::Info("BlackBox: kolmogor_badauth not configured");
            return;
        }

        const TString writerPath = xpathPrefix + "/badauth/kolmogor_writer";

        NDbPool::TQueryOpts opts = {NTvmDbPool::CreateServiceTicketOpt(
            TvmClient_,
            NTvmCommon::TServiceTickets::KOLMOGOR_)};

        try {
            KolmogorDb_ = ctx->CreateDb(config, dbPath, std::move(opts));
            KolmogorDb_->TryPing();
        } catch (const std::exception& e) {
            TLog::Info() << "BlackBox: kolmogor db not available on start, ignoring: " << e.what();
        }

        ui32 workerCount = config.AsInt(writerPath + "/worker_threads", 1);
        ui32 queueLength = config.AsInt(writerPath + "/max_queue_length", 1000);

        TDuration writeTimeout = TDuration::MilliSeconds(config.AsInt(writerPath + "/write_timeout", 250));
        KolmogorWriter_ = std::make_unique<TAsyncDbWriter>(workerCount,
                                                           KolmogorDb_.get(),
                                                           queueLength,
                                                           writeTimeout);

        TLog::Info() << "BlackBox: kolmogor_badauth: set up kolmogor: "
                     << KolmogorDb_->GetDbInfo();
    }
}
