#include "helper.h"

#include "async_db_writer.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>

#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/datetime/base.h>

// uncomment for debugging
//#define LOG_QUERIES

namespace NPassport::NBb {
    TBadauthHelper::TBadauthHelper(NDbPool::TDbPool* kolmogorDb,
                                   TAsyncDbWriter* kolmogorWriter,
                                   const TBadauthSettings& badauthSettings)
        : KolmogorDb_(kolmogorDb)
        , KolmogorWriter_(kolmogorWriter)
        , Counts_(badauthSettings)
        , BadauthSettings_(badauthSettings)
    {
        if (!kolmogorDb) {
            Counts_.Failed = true; // ignore all operations if we have no db configured
        }
    }

    void TBadauthHelper::IncrementCounters(TString& errMsg, size_t pwdLength) {
        if (Counts_.Failed) {
            TLog::Info("BadauthHelper: failed to add record to queue: without attempt");
            return; // ignore if db not accessible
        }

        TString poolMsg;
        if (!KolmogorDb_->IsOk(&poolMsg)) {
            errMsg = std::move(poolMsg);
            Counts_.Failed = true;
            TLog::Info("BadauthHelper: failed to add record to queue: db is bad");
            return;
        }

        TKolmogor::TKeys keys;
        TKolmogor::TKeys shortKeys;
        TKolmogor::TKeys uniqKeys;

        bool firstTimeBan = (Counts_.RepeatCount == 0);

        // if not banned by (login,pwd,ip,authtype)
        if (Counts_.IncrementRepeated) {
            ++Counts_.RepeatCount;
            keys.push_back(CreateKeyForLoginPwdIpAuthtype());
        }

        // if not repeated (login,pwd,ip) pair, increment all counters
        if (firstTimeBan) {
            // if IP gets here for the first time, increment it
            if (Counts_.LoginIp == 0 || Counts_.PwdIp == 0) {
                ++Counts_.Ip;
                keys.push_back(CreateKeyForIp());
                if (HalfIp_) {
                    keys.push_back(CreateKeyForIpHalfIp());
                }
                if (pwdLength < BadauthSettings_.PwdLength) {
                    ++Counts_.IpShort;
                    shortKeys.push_back(CreateKeyForIp());
                    if (HalfIp_) {
                        shortKeys.push_back(CreateKeyForIpHalfIp());
                    }
                }
            }

            // if login gets here with new IP, increment it
            if (Counts_.LoginIp == 0) {
                ++Counts_.Login;
                keys.push_back(CreateKeyForLogin());
            }

            // if passwd gets here with new IP, increment it
            if (Counts_.PwdIp == 0) {
                ++Counts_.Pwd;
                keys.push_back(CreateKeyForPwd());
            }

            // increment (login,ip) and (pwd,ip)
            ++Counts_.LoginIp;
            keys.push_back(CreateKeyForLoginIp());
            if (HalfIp_) {
                keys.push_back(CreateKeyForLoginIpHalfIp());
            }

            ++Counts_.PwdIp;
            keys.push_back(CreateKeyForPwdIp());
            if (HalfIp_) {
                keys.push_back(CreateKeyForPwdIpHalfIp());
            }
        }

        if (keys.empty() && shortKeys.empty() && uniqKeys.empty()) {
            return;
        }

        TString query = QueryForKolmogorInc(std::move(uniqKeys), std::move(keys), std::move(shortKeys));

#ifdef LOG_QUERIES
        TLog::Info("BadauthHelper: kolmogor INC keys %s", query.c_str());
#endif
        KolmogorWriter_->Append(query);
    }

    void TBadauthHelper::IncrementAuthlog(TString& errMsg) {
        if (Counts_.Failed) {
            TLog::Info("BadauthHelper: failed to add authlog record to queue: without attempt");
            return; // ignore if db not accessible
        }

        TString poolMsg;
        if (!KolmogorDb_->IsOk(&poolMsg)) {
            errMsg = std::move(poolMsg);
            Counts_.Failed = true;
            TLog::Info("BadauthHelper: failed to add authlog record to queue: db is bad");
            return;
        }

        ++Counts_.Authlog;

        try {
            TString query = QueryForKolmogorInc(TKolmogor::TKeys{{CreateKeyForAuthlog()}});

#ifdef LOG_QUERIES
            TLog::Info("BadauthHelper: kolmogor INC keys %s", query.c_str());
#endif
            KolmogorWriter_->Append(query);

        } catch (const std::exception& e) {
            Counts_.Failed = true;
            TLog::Debug("BadauthHelper: failed to add authlog to queue: exception: %s",
                        e.what());
        }
    }

    void TBadauthHelper::GetTotalCounts(TString& errMsg) {
        if (Counts_.Failed) {
            TLog::Info("BadauthHelper: failed to get counts: without attempt");
            return; // ignore if db not accessible
        }

        TString poolMsg;
        if (!KolmogorDb_->IsOk(&poolMsg)) {
            errMsg = std::move(poolMsg);
            Counts_.Failed = true;
            TLog::Info("BadauthHelper: failed to get counts: db is bad");
            return;
        }

        TString response;
        try {
            NDbPool::TQuery httpQuery = TKolmogor::CreateGetQuery(QueryForKolmogorGet());

            response = NDbPool::NUtils::GetHttpBody(*KolmogorDb_,
                                                    std::move(httpQuery),
                                                    "kolmogor");
        } catch (const std::exception& e) {
            TLog::Debug("BadauthHelper: failed to get from kolmogor: wait exception: %s", e.what());
            Counts_.Failed = true;
            return;
        }

        Counts_.Failed = !ParseResponseForKolmogorGet(response);
    }

    bool TBadauthHelper::ParseResponseForKolmogorGet(TStringBuf body) {
        rapidjson::Document doc;
        if (!NJson::TReader::DocumentAsObject(body, doc)) {
            TLog::Error() << "BadauthHelper: failed to get from kolmogor: invalid json: "
                          << body;
            return false;
        }

        const rapidjson::Value* space = nullptr;
        if (!NJson::TReader::MemberAsObject(doc, BadauthSettings_.SpaceNameCommon.c_str(), space)) {
            TLog::Error() << "BadauthHelper: failed to get from kolmogor: missing badauth space in response: "
                          << body;
            return false;
        }

        auto get = [&](const TString& key, ui64& out) {
            const rapidjson::Value* obj = nullptr;
            return NJson::TReader::MemberAsObject(*space, key.c_str(), obj) &&
                   NJson::TReader::MemberAsUInt64(*obj, "value", out);
        };

        bool success = get(CreateKeyForLoginPwdIpAuthtype(), Counts_.RepeatCount) &&
                       get(CreateKeyForLoginIp(), Counts_.LoginIp) &&
                       get(CreateKeyForPwdIp(), Counts_.PwdIp) &&
                       get(CreateKeyForLogin(), Counts_.Login) &&
                       get(CreateKeyForPwd(), Counts_.Pwd) &&
                       get(CreateKeyForIp(), Counts_.Ip);

        if (HalfIp_) {
            success = success &&
                      get(CreateKeyForLoginIpHalfIp(), Counts_.LoginIpHalfIp) &&
                      get(CreateKeyForPwdIpHalfIp(), Counts_.PwdIpHalfIp) &&
                      get(CreateKeyForIpHalfIp(), Counts_.IpHalfIp);
        }

        if (!NJson::TReader::MemberAsObject(doc, BadauthSettings_.SpaceNameShort.c_str(), space)) {
            TLog::Error() << "BadauthHelper: failed to get from kolmogor: missing badauth_short space in response: "
                          << body;
            return false;
        }
        success = success && get(CreateKeyForIp(), Counts_.IpShort);

        if (HalfIp_) {
            success = success && get(CreateKeyForIpHalfIp(), Counts_.IpShortHalfIp);
        }

        if (!NJson::TReader::MemberAsObject(doc, BadauthSettings_.SpaceNameUniq.c_str(), space)) {
            TLog::Error() << "BadauthHelper: failed to get from kolmogor: missing badauth_uniq space in response: "
                          << body;
            return false;
        }
        success = success && get(CreateKeyForAuthlog(), Counts_.Authlog);

        if (!success) {
            TLog::Error() << "BadauthHelper: failed to get from kolmogor: invalid format of value: "
                          << body;
            return false;
        }

#ifdef LOG_QUERIES
        TLog::Info("BadauthHelper: just got kolmogor counts: %s ", counts_.str().c_str());
#endif
        return true;
    }

    TString TBadauthHelper::QueryForKolmogorGet() {
        TKolmogor::TSpaces spaces;
        spaces.push_back({
            .Name = BadauthSettings_.SpaceNameUniq,
            .Keys = TKolmogor::TKeys{{
                CreateKeyForAuthlog(),
            }},
        });
        spaces.push_back({
            .Name = BadauthSettings_.SpaceNameCommon,
            .Keys = TKolmogor::TKeys{{
                CreateKeyForLoginPwdIpAuthtype(),
                CreateKeyForLoginIp(),
                CreateKeyForPwdIp(),
                CreateKeyForLogin(),
                CreateKeyForPwd(),
                CreateKeyForIp(),
            }},
        });
        if (HalfIp_) {
            spaces.back().Keys.push_back(CreateKeyForLoginIpHalfIp());
            spaces.back().Keys.push_back(CreateKeyForPwdIpHalfIp());
            spaces.back().Keys.push_back(CreateKeyForIpHalfIp());
        }
        spaces.push_back({
            .Name = BadauthSettings_.SpaceNameShort,
            .Keys = TKolmogor::TKeys{{
                CreateKeyForIp(),
            }},
        });
        if (HalfIp_) {
            spaces.back().Keys.push_back(CreateKeyForIpHalfIp());
        }

        return TKolmogor::PrepareGetQuery(spaces);
    }

    TString TBadauthHelper::QueryForKolmogorInc(TKolmogor::TKeys&& uniqKeys,
                                                TKolmogor::TKeys&& keys,
                                                TKolmogor::TKeys&& shortKeys) {
        Y_ENSURE(!keys.empty() || !shortKeys.empty() || !uniqKeys.empty());

        TKolmogor::TSpaces spaces;
        if (!keys.empty()) {
            spaces.push_back({
                .Name = BadauthSettings_.SpaceNameCommon,
                .Keys = std::move(keys),
            });
        }
        if (!shortKeys.empty()) {
            spaces.push_back({
                .Name = BadauthSettings_.SpaceNameShort,
                .Keys = std::move(shortKeys),
            });
        }
        if (!uniqKeys.empty()) {
            spaces.push_back({
                .Name = BadauthSettings_.SpaceNameUniq,
                .Keys = std::move(uniqKeys),
            });
        }

        return TKolmogor::PrepareIncQuery(spaces);
    }

    static const TString LOGIN_PWD_IP_AUTHTYPE_ = "LGNPWDIPAT_";
    static const TString LOGIN_IP_ = "LGNIP_";
    static const TString PWD_IP_ = "PWDIP_";
    static const TString LOGIN_ = "LGN_";
    static const TString PWD_ = "PWD_";
    static const TString IP_ = "IP_";
    static const TString AUTHLOG_ = "LOG_";
    static const TString DELIMITER_ = "_";

    template <typename T>
    static const TString& CreateKey(TString& value, T func) {
        if (!value) {
            value = func();
        }

        return value;
    }

    const TString& TBadauthHelper::CreateKeyForLoginPwdIpAuthtype() {
        return CreateKey(
            Keys_.LoginPwdIpAuthtype,
            [this]() {
                return NUtils::CreateStr(
                    LOGIN_PWD_IP_AUTHTYPE_,
                    Login_,
                    DELIMITER_,
                    PasswdHash_,
                    DELIMITER_,
                    UserIp_,
                    DELIMITER_,
                    AuthType_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForLoginIp() {
        return CreateKey(
            Keys_.LoginIp,
            [this]() {
                return NUtils::CreateStr(
                    LOGIN_IP_,
                    Login_,
                    DELIMITER_,
                    UserIp_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForPwdIp() {
        return CreateKey(
            Keys_.PwdIp,
            [this]() {
                return NUtils::CreateStr(
                    PWD_IP_,
                    PasswdHash_,
                    DELIMITER_,
                    UserIp_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForLogin() {
        return CreateKey(
            Keys_.Login,
            [this]() {
                return NUtils::CreateStr(
                    LOGIN_,
                    Login_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForPwd() {
        return CreateKey(
            Keys_.Pwd,
            [this]() {
                return NUtils::CreateStr(
                    PWD_,
                    PasswdHash_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForIp() {
        return CreateKey(
            Keys_.Ip,
            [this]() {
                return NUtils::CreateStr(
                    IP_,
                    UserIp_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForLoginIpHalfIp() {
        return CreateKey(
            Keys_.LoginIpHalfIp,
            [this]() {
                return NUtils::CreateStr(
                    LOGIN_IP_,
                    Login_,
                    DELIMITER_,
                    HalfIp_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForPwdIpHalfIp() {
        return CreateKey(
            Keys_.PwdIpHalfIp,
            [this]() {
                return NUtils::CreateStr(
                    PWD_IP_,
                    PasswdHash_,
                    DELIMITER_,
                    HalfIp_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForIpHalfIp() {
        return CreateKey(
            Keys_.IpHalfIp,
            [this]() {
                return NUtils::CreateStr(
                    IP_,
                    HalfIp_);
            });
    }

    const TString& TBadauthHelper::CreateKeyForAuthlog() {
        return CreateKey(
            Keys_.Authlog,
            [this]() {
                return NUtils::CreateStr(
                    AUTHLOG_,
                    Login_,
                    DELIMITER_,
                    AuthType_,
                    DELIMITER_,
                    UserIp_);
            });
    }

    void TBadauthHelper::SetLogin(const TString& login) {
        Login_ = login;
        Counts_.SetLogin(login);
    }
}
