#include "utils.h"

#include <passport/infra/daemons/blackbox/src/output/badauth_counters_chunk.h>

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    // counter names
    static const TString CTR_LOGIN_PWD_IP_AUTHTYPE = "login,pwd,ip,authtype";
    static const TString CTR_LOGIN_IP = "login,ip";
    static const TString CTR_PWD_IP = "pwd,ip";
    static const TString CTR_LOGIN = "login";
    static const TString CTR_PWD = "pwd";
    static const TString CTR_IP = "ip";
    static const TString CTR_IP_SHORT = "ip_short";
    static const TString CTR_AUTHLOG = "authlog";

    TString TBadauthSettings::Str() const {
        TStringStream out;
        out << RepeatedRequestLimit << " " << CTR_LOGIN_PWD_IP_AUTHTYPE << "/10min, ";
        out << LoginPwdIpAuthHr << " " << CTR_LOGIN_PWD_IP_AUTHTYPE << "/hr, ";
        out << LoginIpHr << " " << CTR_LOGIN_IP << "/hr, ";
        out << PwdIpHr << " " << CTR_PWD_IP << "/hr, ";
        out << LoginHrCombined << " " << CTR_LOGIN << "/hr combined, ";
        out << PwdHrCombined << " " << CTR_PWD << "/hr combined, ";
        out << IpHrCombined << " " << CTR_IP << "/hr combined, ";
        out << IpHr << " " << CTR_IP << "/hr, ";
        out << IpMin << " " << CTR_IP << "/min, ";
        out << AuthlogLimit << " " << CTR_AUTHLOG << "/10min";

        return out.Str();
    }

    namespace {
        TString AuthMsg(const TString& name, ui64 val, ui64 limit) {
            return NUtils::CreateStr(name, " within hour: ", val, "/", limit);
        }

        TString AuthMsg2(const TString& name1, const TString& name2, ui64 val1, ui64 val2, ui64 limit1, ui64 limit2) {
            return NUtils::CreateStr(name1, " AND ", name2, " within hour: ", val1, ",", val2, "/", limit1, ",", limit2);
        }

        TString AuthMsgMin(const TStringBuf name, ui64 val, ui64 limit) {
            return NUtils::CreateStr(name, " within minute: ", val, "/", limit);
        }
    }

    TBadauthCheckResult TBadauthCounts::RepeatInLimits(TString& authlogMsg) {
        // check (login,pwd,ip) counter
        if (RepeatCount >= Settings_.LoginPwdIpAuthHr) {
            IncrementRepeated = false;
            // Note: this message says 'N within hour', leave it as is, someone may parse it
            authlogMsg = AuthMsg(CTR_LOGIN_PWD_IP_AUTHTYPE, RepeatCount, Settings_.LoginPwdIpAuthHr);
            return TBadauthCheckResult("woodpecker", RepeatCount);
        }

        return {};
    }

    TBadauthCheckResult TBadauthCounts::CountsInLimits(bool whitelisted, size_t pwdLength, TString& authlogMsg) const {
        // check (login,ip)
        if (LoginIp >= Settings_.LoginIpHr) {
            authlogMsg = AuthMsg(CTR_LOGIN_IP, LoginIp, Settings_.LoginIpHr);
            return TBadauthCheckResult("ip_login_tuple", LoginIp);
        }

        // check (pwd,ip)
        if (!whitelisted && PwdIp >= Settings_.PwdIpHr) {
            authlogMsg = AuthMsg(CTR_PWD_IP, PwdIp, Settings_.PwdIpHr);
            return TBadauthCheckResult("ip_pwd_tuple", PwdIp);
        }

        // check login and ip combined
        if (Login >= Settings_.LoginHrCombined && Ip >= Settings_.IpHrCombined) {
            authlogMsg = AuthMsg2(CTR_LOGIN, CTR_IP, Login, Ip, Settings_.LoginHrCombined, Settings_.IpHrCombined);
            return TBadauthCheckResult("ip_and_login", Ip, Login);
        }

        // check pwd and ip combined
        if (!whitelisted && Pwd >= Settings_.PwdHrCombined && Ip >= Settings_.IpHrCombined) {
            authlogMsg = AuthMsg2(CTR_PWD, CTR_IP, Pwd, Ip, Settings_.PwdHrCombined, Settings_.IpHrCombined);
            return TBadauthCheckResult("ip_and_pwd", Ip, Pwd);
        }

        // check ip
        if (!whitelisted && Ip >= Settings_.IpHr) {
            authlogMsg = AuthMsg(CTR_IP, Ip, Settings_.IpHr);
            return TBadauthCheckResult("ip", Ip);
        }

        // check ip - short counter
        if (!whitelisted && pwdLength < Settings_.PwdLength && IpShort >= Settings_.IpMin) {
            authlogMsg = AuthMsgMin(CTR_IP_SHORT, IpShort, Settings_.IpMin);
            return TBadauthCheckResult("ip_short", IpShort);
        }

        return {}; // ok, not banned
    }

    TBadauthCheckResult TBadauthCounts::CountsInLimitsHalfIp(bool whitelisted, size_t pwdLength, TString& authlogMsg) const {
        // check (login,ip)
        if (LoginIpHalfIp >= Settings_.LoginIpHr) {
            authlogMsg = AuthMsg(CTR_LOGIN_IP, LoginIpHalfIp, Settings_.LoginIpHr);
            return TBadauthCheckResult("ip_login_tuple", LoginIpHalfIp);
        }

        // check (pwd,ip)
        if (!whitelisted && PwdIpHalfIp >= Settings_.PwdIpHr) {
            authlogMsg = AuthMsg(CTR_PWD_IP, PwdIpHalfIp, Settings_.PwdIpHr);
            return TBadauthCheckResult("ip_pwd_tuple", PwdIpHalfIp);
        }

        // check login and ip combined
        if (Login >= Settings_.LoginHrCombined && IpHalfIp >= Settings_.IpHrCombined) {
            authlogMsg = AuthMsg2(CTR_LOGIN, CTR_IP, Login, IpHalfIp, Settings_.LoginHrCombined, Settings_.IpHrCombined);
            return TBadauthCheckResult("ip_and_login", IpHalfIp, Login);
        }

        // check pwd and ip combined
        if (!whitelisted && Pwd >= Settings_.PwdHrCombined && IpHalfIp >= Settings_.IpHrCombined) {
            authlogMsg = AuthMsg2(CTR_PWD, CTR_IP, Pwd, IpHalfIp, Settings_.PwdHrCombined, Settings_.IpHrCombined);
            return TBadauthCheckResult("ip_and_pwd", IpHalfIp, Pwd);
        }

        // check ip
        if (!whitelisted && IpHalfIp >= Settings_.IpHr) {
            authlogMsg = AuthMsg(CTR_IP, IpHalfIp, Settings_.IpHr);
            return TBadauthCheckResult("ip", IpHalfIp);
        }

        // check ip - short counter
        if (!whitelisted && pwdLength < Settings_.PwdLength && IpShortHalfIp >= Settings_.IpMin) {
            authlogMsg = AuthMsgMin(CTR_IP_SHORT, IpShortHalfIp, Settings_.IpMin);
            return TBadauthCheckResult("ip_short", IpShortHalfIp);
        }

        return {}; // ok, not banned
    }

    TBadauthCounts::TBadauthCounts(const TBadauthSettings& settings)
        : Settings_(settings)
    {
    }

    TString TBadauthCounts::Str() const {
        TStringStream out;
        out << CTR_LOGIN_PWD_IP_AUTHTYPE << "=" << RepeatCount << ", ";
        out << CTR_LOGIN_IP << "=" << LoginIp << ", ";
        out << CTR_PWD_IP << "=" << PwdIp << ", ";
        out << CTR_LOGIN << "=" << Login << ", ";
        out << CTR_PWD << "=" << Pwd << ", ";
        out << CTR_IP << "=" << Ip << ", ";
        out << CTR_IP_SHORT << "=" << IpShort << ", ";
        out << CTR_AUTHLOG << "=" << Authlog;

        return out.Str();
    }

    void TBadauthCounts::GetValues(TBadauthCountersChunk& chunk) const {
        if (Failed) {
            return;
        }

        chunk.Counters = {
            {CTR_LOGIN_PWD_IP_AUTHTYPE, RepeatCount, Settings_.LoginPwdIpAuthHr},
            {CTR_LOGIN_IP, LoginIp, Settings_.LoginIpHr},
            {CTR_PWD_IP, PwdIp, Settings_.PwdIpHr},
            {CTR_LOGIN, Login, Settings_.LoginHrCombined},
            {CTR_PWD, Pwd, Settings_.PwdHrCombined},
            {CTR_IP, Ip, Settings_.IpHr},
            {CTR_IP_SHORT, IpShort, Settings_.IpMin},
        };
    }

    bool TBadauthCounts::IsOk() const {
        return !Failed;
    }

    void TBadauthCounts::SetLogin(const TString& login) {
        LoginStr_ = login;
    }

    bool TBadauthCounts::AuthLogInLimits() const {
        // limit is off or counter in limits
        return Settings_.AuthlogLimit == 0 || Authlog < Settings_.AuthlogLimit;
    }

}
