#include "blackbox.h"

#include "blackbox_impl.h"
#include "grants/consumer_info.h"
#include "grants/grants_checker.h"
#include "methods/check_device_signature.h"
#include "methods/check_has_plus.h"
#include "methods/check_rfc_totp.h"
#include "methods/check_sign.h"
#include "methods/checkip.h"
#include "methods/create_oauth_token.h"
#include "methods/create_pwd_hash.h"
#include "methods/createsession.h"
#include "methods/decrease_sessionid_lifetime.h"
#include "methods/deletion_operations.h"
#include "methods/edit_totp.h"
#include "methods/editsession.h"
#include "methods/email_bindings.h"
#include "methods/family_info.h"
#include "methods/find_by_phone_numbers.h"
#include "methods/find_pdd_accounts.h"
#include "methods/generate_public_id.h"
#include "methods/get_all_tracks.h"
#include "methods/get_debug_user_ticket.h"
#include "methods/get_device_public_key.h"
#include "methods/get_max_uid.h"
#include "methods/get_oauth_tokens.h"
#include "methods/get_recovery_keys.h"
#include "methods/get_track.h"
#include "methods/hosted_domains.h"
#include "methods/lcookie.h"
#include "methods/login.h"
#include "methods/loginoccupation.h"
#include "methods/oauth.h"
#include "methods/phone_bindings.h"
#include "methods/phone_operations.h"
#include "methods/prove_key_diag.h"
#include "methods/pwdhistory.h"
#include "methods/sessionid.h"
#include "methods/sign.h"
#include "methods/test_pwd_hashes.h"
#include "methods/user_ticket.h"
#include "methods/userinfo.h"
#include "methods/webauthn_credentials.h"
#include "methods/yakey_backup.h"
#include "misc/exception.h"
#include "misc/experiment.h"
#include "misc/strings.h"
#include "misc/utils.h"
#include "output/all_tracks_result.h"
#include "output/check_device_signature_result.h"
#include "output/check_grants_result.h"
#include "output/check_has_plus_result.h"
#include "output/check_rfc_totp_result.h"
#include "output/check_sign_result.h"
#include "output/create_oauth_token_result.h"
#include "output/create_session_result.h"
#include "output/decrease_sessionid_lifetime_result.h"
#include "output/edit_totp_result.h"
#include "output/family_info_result.h"
#include "output/find_by_phone_numbers_result.h"
#include "output/get_debug_user_ticket_result.h"
#include "output/get_device_public_key_result.h"
#include "output/get_max_uid_result.h"
#include "output/get_oauth_tokens_result.h"
#include "output/json_serializer.h"
#include "output/lcookie_result.h"
#include "output/list_result.h"
#include "output/login_result.h"
#include "output/loginoccupation_result.h"
#include "output/oauth_result.h"
#include "output/phone_bindings_result.h"
#include "output/prove_key_diag_result.h"
#include "output/pwdhistory_result.h"
#include "output/session_result.h"
#include "output/table_result.h"
#include "output/test_pwd_hashes_result.h"
#include "output/track_result.h"
#include "output/typed_value_result.h"
#include "output/user_info_result.h"
#include "output/user_ticket_result.h"
#include "output/webauthn_credentials_result.h"
#include "output/xml_serializer.h"

#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/digest/old_crc/crc.h>

#include <util/datetime/base.h>
#include <util/system/fs.h>
#include <util/system/hostname.h>

namespace NPassport::NBb {
    static const TString BB_NO_GRANTS("\r\n<html><head><title>blackbox monitoring status</title></head><body><h2>Not operational: don't have valid grants config</h2></body></html>\r\n");
    static const TString PING_BB_ERR_HEAD("\r\n<html><head><title>blackbox monitoring status</title></head><body><h2>Error: databases: ");
    static const TString PING_BB_ERR_TAIL("</h2></body></html>\r\n");
    static const TString PING_BB_OK("\r\n<html><head><title>blackbox monitoring status</title></head><body><h2>OK</h2></body></html>\r\n");
    static const TString BB_FORCED_DOWN("\r\n<html><head><title>blackbox monitoring status</title></head><body><h2>Not operational: externally forced out-of-service</h2></body></html>\r\n");
    static const TString PING_BB_ERR_COMMON_HEAD("\r\n<html><head><title>blackbox monitoring status</title></head><body><h2>Error: ");

    static const TString BLACKBOX("/blackbox");
    static const TString NAGIOS("/blackbox/nagios");
    static const TString PING("/blackbox/ping");
    static const TString HEALTH_CHECK("/blackbox/healthcheck");
    static const TString CHECK_GRANTS("/blackbox/check_grants");

    static const TString DBPOOLSTATS("dbpoolstats");
    static const TString DBPOOL_EXTENDED_STATS("dbpool_extended_stats");
    static const TString GET_DEBUG_USER_TICKET("get_debug_user_ticket");
    static const TString DECREASE_SESSIONID_LIFETIME("decrease_sessionid_lifetime");

    static const TString X_CONN_REQS("X-Conn-Reqs");
    static const TString CONTENT_TYPE("Content-type");
    static const TString REQUEST_ID("X-Request-Id");
    static const TString CONTENT_TYPE_HTML("text/html");
    static const TString CONTENT_TYPE_JSON("application/json");
    static const TString CONTENT_TYPE_PLAIN("text/plain");
    static const TString CONTENT_TYPE_XML("text/xml");
    static const TString FORMAT("format");
    static const TString JSON("json");

    TBlackbox::TBlackbox()
        : Hostname_(FQDNHostName())
    {
        AddMethod<TCheckDeviceSignatureProcessor>("check_device_signature");
        AddMethod<TCheckHasPlusProcessor>("check_has_plus");
        AddMethod<TCheckIpProcessor>("checkip");
        AddMethod<TCheckRfcTotpProcessor>("check_rfc_totp");
        AddMethod<TCheckSignProcessor>("check_sign");
        AddMethod<TCreateOAuthTokenProcessor>("create_oauth_token");
        AddMethod<TCreatePwdHashProcessor>("create_pwd_hash");
        AddMethod<TCreateSessionProcessor>("createsession");
        AddMethod<TDecreaseSessionidLifetimeProcessor>(DECREASE_SESSIONID_LIFETIME);
        AddMethod<TDeletionOperationsProcessor>("deletion_operations");
        AddMethod<TEditSessionProcessor>("editsession");
        AddMethod<TEditTotpProcessor>("edit_totp");
        AddMethod<TEmailBindingsProcessor>("email_bindings");
        AddMethod<TFamilyInfoProcessor>("family_info");
        AddMethod<TFindByPhoneNumbersProcessor>("find_by_phone_numbers");
        AddMethod<TFindPddAccountsProcessor>("find_pdd_accounts");
        AddMethod<TGeneratePublicIdProcessor>("generate_public_id");
        AddMethod<TGetAllTracksProcessor>("get_all_tracks");
        AddMethod<TGetDebugUserTicketProcessor>(GET_DEBUG_USER_TICKET);
        AddMethod<TGetDevicePublicKeyProcessor>("get_device_public_key");
        AddMethod<TGetMaxUidProcessor>("get_max_uid");
        AddMethod<TGetOAuthTokensProcessor>("get_oauth_tokens");
        AddMethod<TGetRecoveryKeysProcessor>("get_recovery_keys");
        AddMethod<TGetTrackProcessor>("get_track");
        AddMethod<THostedDomainsProcessor>("hosted_domains");
        AddMethod<TLCookieProcessor>("lcookie");
        AddMethod<TLoginOccupationProcessor>("loginoccupation");
        AddMethod<TLoginProcessor>("login");
        AddMethod<TOAuthProcessor>("oauth");
        AddMethod<TPhoneBindingsProcessor>("phone_bindings");
        AddMethod<TPhoneOperationsProcessor>("phone_operations");
        AddMethod<TProveKeyDiagProcessor>("prove_key_diag");
        AddMethod<TPwdHistoryProcessor>("pwdhistory");
        AddMethod<TSessionidProcessor>("sessionid");
        AddMethod<TSignProcessor>("sign");
        AddMethod<TTestPwdHashesProcessor>("test_pwd_hashes");
        AddMethod<TUserInfoProcessor>("userinfo");
        AddMethod<TUserTicketProcessor>("user_ticket");
        AddMethod<TWebauthnCredentialsProcessor>("webauthn_credentials");
        AddMethod<TYakeyBackupProcessor>("yakey_backup");
    }

    TBlackbox::~TBlackbox() = default;

    void TBlackbox::Init(const NXml::TConfig& config) {
        try {
            const TString componentXPath = "/config/components/component";

            InitLoggers(config, componentXPath);

            TLog::Info("Starting blackbox module");

            Impl = std::make_unique<TBlackboxImpl>();

            ForceProvideRequestId_ = config.AsBool(componentXPath + "/__force_provide_request_id", false);

            Impl->Init(config, componentXPath);

            // file used for forcing out-of-service state
            ForceDownFilePath_ = config.AsString(componentXPath + "/force_down_file", "");
            TLog::Info("Forced out-of-service file: %s",
                       ForceDownFilePath_.empty() ? "unconfigured" : ForceDownFilePath_.c_str());

            InitCache(config, componentXPath);
            InitHttpCodes(config, componentXPath);
        } catch (const std::exception& e) {
            TLog::Error("Blackbox module init failed: %s", e.what());
            throw;
        } catch (...) {
            TLog::Error("Blackbox module init failed: unknown exception");
            ythrow yexception() << "Blackbox module init failed: unknown exception";
        }
    }

    void TBlackbox::LogAccess(const NCommon::TRequest& request, TDuration timeSpent, NCache::EStatus cacheStatus) {
        TStringBuf path = request.GetPath();
        path.ChopSuffix("/");
        if (path.StartsWith(NAGIOS) || path.StartsWith(HEALTH_CHECK)) {
            return;
        }

        if (CHECK_GRANTS == path) {
            LogAccessCheckGrants(request, timeSpent);
        } else {
            LogAccessCommon(request, timeSpent, cacheStatus);
        }
    }

    static const THashSet<TString> ARGS_TO_REPLACE_WITH_STAR = {
        "hashes",
        "password",
        "pin",
        "secret",
        "value",
    };

    static bool NeedToCutSignature(const TStringBuf arg) {
        return arg.EndsWith(TStrings::SESSION_ID) ||
               arg.Contains("sign") ||
               arg == TStrings::OAUTH_TOKEN ||
               arg == TStrings::SESSGUARD ||
               arg == TStrings::NONCE;
    }

    namespace {
        struct TRequestParams {
            TString Get;
            TString Post;
        };
    }

    static TRequestParams PrepareRequestParams(const NCommon::TRequest& request) {
        std::vector<TString> argvec;
        request.ArgNames(argvec);

        TRequestParams res;

        for (const TString& arg : argvec) {
            const NCommon::TExtendedArg& val = request.GetExtendedArg(arg);
            TString& params = val.IsGet ? res.Get : res.Post;
            if (params.empty()) {
                params.reserve(25 * argvec.size());
            }

            NUtils::AppendSeparated(params, '&', arg).push_back('=');
            if (val.Value.empty()) {
                continue;
            }

            // 1. secret parameters: dont log at all
            if (ARGS_TO_REPLACE_WITH_STAR.contains(arg)) {
                params.push_back('*');
                continue;
            }

            // 2. signed credentials: cut signature
            // cut off 10 symbols of sessionid/sessguard/oauth_token if it is not noauth or malformed
            if (NeedToCutSignature(arg) && val.Value.size() > 20) {
                size_t dotpos = val.Value.rfind('.');
                if (dotpos == TString::npos) {
                    dotpos = val.Value.size() - 10;
                }
                params.append(val.Value, 0, dotpos).append("...");
                continue;
            }

            // 3. too long param - cut unless it is dbfields/attributes/uid list
            if (val.Value.length() > 255 && arg != TStrings::ATTRIBUTES && arg != TStrings::DBFIELDS && arg != TStrings::UID) { // cut off too long values (except attributes and dbfields and uid)
                params.append(val.Value, 0, 255).append("...");
                continue;
            }

            params.append(val.Value); // else just write the whole value
        }

        NUtils::EscapeUnprintable(res.Get);
        NUtils::EscapeUnprintable(res.Post);

        return res;
    }

    const char* PrepareNonEmptyStr(const TString& str) {
        return str.empty() ? "-" : str.c_str();
    };

    void TBlackbox::LogAccessCommon(const NCommon::TRequest& request, TDuration timeSpent, NCache::EStatus cacheStatus) {
        if (!AccessLogger_) {
            return;
        }

        const TRequestParams requestParams = PrepareRequestParams(request);

        char cacheChar;
        switch (cacheStatus) {
            case NCache::EStatus::Hit:
                cacheChar = 'h';
                break;
            case NCache::EStatus::Miss:
                cacheChar = 'm';
                break;
            case NCache::EStatus::Unacceptable:
                cacheChar = '-';
                break;
        }

        const TString& requestNumberThroughOneConnection = request.GetHeader(X_CONN_REQS);

        AccessLogger_->Error("%s\thttp%s\t%s\t%.1f\t%s\t%lu\t%c\t%s\t%s\t%s",
                             request.GetRequestId().c_str(),
                             request.IsSecure() ? "s" : "",
                             request.GetRemoteAddr().c_str(),
                             timeSpent.MicroSeconds() / 1000.,
                             request.GetConsumerFormattedName().c_str(),
                             request.GetResponseSize(),
                             cacheChar,
                             PrepareNonEmptyStr(requestNumberThroughOneConnection),
                             PrepareNonEmptyStr(requestParams.Get),
                             PrepareNonEmptyStr(requestParams.Post));
    }

    void TBlackbox::LogAccessCheckGrants(const NCommon::TRequest& request, TDuration timeSpent) {
        if (!CheckGrantsLogger_) {
            return;
        }

        const TRequestParams requestParams = PrepareRequestParams(request);

        const TString& requestNumberThroughOneConnection = request.GetHeader(X_CONN_REQS);

        CheckGrantsLogger_->Error(
            "%s\thttp%s\t%s\t%.1f\t%s\t%lu\t%s\t%d\t%s\t%s",
            request.GetRequestId().c_str(),
            request.IsSecure() ? "s" : "",
            request.GetRemoteAddr().c_str(),
            timeSpent.MicroSeconds() / 1000.,
            request.GetConsumerFormattedName().c_str(),
            request.GetResponseSize(),
            PrepareNonEmptyStr(requestNumberThroughOneConnection),
            (int)request.GetStatusCode(),
            PrepareNonEmptyStr(requestParams.Get),
            PrepareNonEmptyStr(requestParams.Post));
    }

    void TBlackbox::AddUnistat(NUnistat::TBuilder& builder) {
        ErrorStats_.FillStats(builder);
        MethodCounters_.AddUnistat(builder);
        Impl->AddUnistat(builder);

        if (CacheCtx_) {
            CacheCtx_->AddUnistat(builder);
        }
        for (const auto& [method, cache] : CacheByMethod_) {
            if (cache.Json) {
                cache.Json->AddUnistat(builder);
            }
            if (cache.Xml) {
                cache.Xml->AddUnistat(builder);
            }
        }
    }

    void TBlackbox::AddUnistatExtended(const TString& path, NUnistat::TBuilder& builder) {
        if (path == "/consumers/") {
            Consumers_.All.AddUnistat(builder);
        } else if (path == "/consumers/denied/") {
            Consumers_.Denied.AddUnistat(builder);
        } else if (path == "/consumers/cache/") {
            Consumers_.Cache.AddUnistat(builder);
        } else if (path == "/dbpool_ext/") {
            Impl->AddExtendedUnistatForDbPool(builder);
        }
    }

    void TBlackbox::InitLoggers(const NXml::TConfig& config, const TString& componentXPath) {
        config.InitCommonLog(componentXPath + "/logger_common");
        AccessLogger_ = config.CreateLogger(componentXPath + "/logger_access");

        if (config.Contains(componentXPath + "/logger_checkgrants")) {
            CheckGrantsLogger_ = config.CreateLogger(componentXPath + "/logger_checkgrants");
        }
    }

    void TBlackbox::InitCache(const NXml::TConfig& config, const TString& componentXPath) {
        if (!config.Contains(componentXPath + "/response_cache")) {
            TLog::Info() << "Response cache is OFF";
            return;
        }

        double limit = 0;
        const TString limitKey = componentXPath + "/response_cache/total_memory_limit_gb";
        Y_ENSURE(TryFromString(config.AsString(limitKey), limit),
                 "Failed to get cache memory limit: it is not int or double: " << limitKey);
        TLog::Info() << "Response cache is ON. Total memory limit: " << limit << " Gb";

        limit *= 1024 * 1024 * 1024;
        CacheCtx_ = TResponseCache::TContext::Create(limit, "response");

        for (const TString& xpath : config.SubKeys(componentXPath + "/response_cache/method")) {
            int timeBucketCount = config.AsInt(xpath + "/time_bucket_count", 3);
            int keyBucketCount = config.AsInt(xpath + "/key_bucket_count", 128);
            int defaultLifeTime = config.AsInt(xpath + "/default_life_time_ms", 2000);
            const bool dryRun = config.AsBool(xpath + "/dry_run", true);
            const TString name = config.AsString(xpath + "/@name");
            TLog::Info() << "Response cache for method=" << name
                         << ". time_buckets=" << timeBucketCount
                         << ". key_buckets=" << keyBucketCount
                         << ". default_lifetime=" << defaultLifeTime << "ms"
                         << ". dry_run=" << (dryRun ? "on" : "off");

            TResponseCache caches(
                CacheCtx_->CreateCache(timeBucketCount, keyBucketCount, name + ".json"),
                CacheCtx_->CreateCache(timeBucketCount, keyBucketCount, name + ".xml"),
                TDuration::MilliSeconds(defaultLifeTime),
                dryRun);
            CacheByMethod_.insert({name, std::move(caches)});
        }

        CacheCtx_->StartCleaning(TDuration::Seconds(1));
    }

    void TBlackbox::InitHttpCodes(const NXml::TConfig& config, const TString& path) {
        if (config.Contains(path + "/http_codes")) {
            HttpCodes_ = THttpCodes{
                .AccessDenied = (HttpCodes)config.AsInt(path + "/http_codes/access_denied", HTTP_OK),
                .DbException = (HttpCodes)config.AsInt(path + "/http_codes/db_exception", HTTP_OK),
                .FatalException = (HttpCodes)config.AsInt(path + "/http_codes/fatal_exception", HTTP_OK),
                .EtcErrors = (HttpCodes)config.AsInt(path + "/http_codes/etc_errors", HTTP_OK),
            };
        } else if (config.Contains(path + "/exception_status")) {
            HttpCodes code = (HttpCodes)config.AsInt(path + "/exception_status", HTTP_OK);
            HttpCodes_ = THttpCodes{
                .AccessDenied = code,
                .DbException = code,
                .FatalException = code,
                .EtcErrors = code,
            };
        }
    }

    void TBlackbox::HandleRequest(NCommon::TRequest& bbRequest) {
        bbRequest.ScanCgiFromBody();
        if (ForceProvideRequestId_) {
            bbRequest.ForceProvideRequestId();
        }

        const TString& path = bbRequest.GetPath();

        if (bbRequest.GetRequestMethod() != "POST" &&
            bbRequest.GetRequestMethod() != "GET") {
            TLog::Warning("Suspicious HTTP method '%s'", bbRequest.GetRequestMethod().c_str());
            SendErrorStatus(bbRequest, TBlackboxError::EType::Unknown, "Unsupported HTTP method", HTTP_BAD_REQUEST);
            return;
        }

        if (!path.StartsWith(BLACKBOX)) {
            TLog::Warning("Unknown path: %s", path.c_str());
            SendErrorStatus(bbRequest, TBlackboxError::EType::Unknown, "Unknown path: " + path, HTTP_BAD_REQUEST);
            return;
        }

        TInstant start = TInstant::Now();
        NCache::EStatus cacheStatus = Handle(bbRequest);
        TInstant stop = TInstant::Now();

        LogAccess(bbRequest, stop - start, cacheStatus);
    }

    static const TString ACCESS_DENIED_MSG =
        "Please request grants here: https://forms.yandex-team.ru/surveys/4901 . "
        "If you are sure you have grants, please mail to passport-admin@yandex-team.ru";
    void TBlackbox::SendErrorStatus(NCommon::TRequest& request,
                                    const TBlackboxError::EType status,
                                    const TString& msg,
                                    const HttpCodes httpStatus,
                                    const TString& method) const {
        TExceptionInfo info;
        info.Msg = msg;
        info.Status = status;
        if (status == TBlackboxError::EType::AccessDenied) {
            info.ExtendedDescription = ACCESS_DENIED_MSG;
        }
        info.Method = method;
        info.Host = request.GetHost();
        info.Hostname = Hostname_;

        TString body;
        if (request.GetArg(FORMAT) == JSON) {
            body = NBb::TJsonSerializer::SerializeException(info);
            body.push_back('\n');
        } else {
            body = NBb::TXmlSerializer::SerializeException(info);
        }

        WriteResult(request, httpStatus, body);

        TLog::Debug() << "BlackBox: return error: status=" << TBlackboxError::StatusStr(info.Status)
                      << "; consumer name=" << request.GetConsumerFormattedName()
                      << "; consumer IP=" << request.GetRemoteAddr()
                      << "; message=" << msg;
    }

    void TBlackbox::SendBlackboxPingError(NCommon::TRequest& request, const TString& msg) {
        TLog::Warning("blackbox ping error: %s", msg.c_str());
        request.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
        request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_HTML);
        TString body(PING_BB_ERR_COMMON_HEAD);
        body.append(msg).append(PING_BB_ERR_TAIL);
        request.Write(body);
    }

    void TBlackbox::CheckDuplicatedArgs(const NCommon::TRequest& req) {
        NCommon::TRequest::TDuplicatedArgs dups = req.GetDuplicatedArgs();

        TString msgToThrow;
        for (auto [key, value] : dups) {
            // cut off 10 symbols of sessionid/sessguard/oauth_token if it is not noauth or malformed
            if (NeedToCutSignature(key) && value.size() > 20) {
                size_t dotpos = value.rfind('.');
                if (dotpos == TString::npos) {
                    dotpos = value.size() - 10;
                }

                value = value.substr(0, dotpos);
            } else if (ARGS_TO_REPLACE_WITH_STAR.contains(key)) {
                value = "*";
            }

            NUtils::AppendSeparated(msgToThrow, '&', NUtils::CreateStr(key, "=", value));
        }

        if (msgToThrow) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Duplicate args not allowed, got '" << msgToThrow << "'";
        }
    }

    static const std::set<TString> METHODS_WITHOUT_GRANTS = {
        DECREASE_SESSIONID_LIFETIME,
        GET_DEBUG_USER_TICKET,
    };

    NCache::EStatus TBlackbox::Handle(NCommon::TRequest& request) {
        TStringBuf path = request.GetPath();
        path.ChopSuffix("/");

        if (CHECK_GRANTS == path) {
            MethodCounters_.IncCheckGrants();
            HandleCheckGrants(request);
            return NCache::EStatus::Unacceptable;
        }

        TConsumerInfo consumerInfo; // required for access log and yasm
        TString method;
        try {
            // Handle ping first
            if (path.StartsWith(NAGIOS) || path.StartsWith(PING)) {
                MethodCounters_.IncPing();
                ProcessPing(request);
                return NCache::EStatus::Unacceptable;
            }

            if (path.StartsWith(HEALTH_CHECK)) {
                MethodCounters_.IncHealthCheck();
                ProcessHealthCheck(request);
                return NCache::EStatus::Unacceptable;
            }

            consumerInfo = Impl->GetConsumer(request);

            method = request.GetArg(TStrings::METHOD);
            if (METHODS_WITHOUT_GRANTS.contains(method)) {
                UpdateConsumerInfoForMethodsWithoutGrants(request, consumerInfo);
            }

            Consumers_.All.Add(consumerInfo);
            if (!consumerInfo.IsOk()) {
                // unknown consumer or consumer came with valid TVM ticket but illegal IP
                throw TBlackboxError(
                    consumerInfo.IsServiceTicketBad()
                        ? TBlackboxError::EType::Unknown // Never change it: PASSP-29436
                        : TBlackboxError::EType::AccessDenied)
                    << consumerInfo.GetError();
            }

            // Requested method (historically, ignore this for some older methods
            // by still may rely on it in the future)
            if (method.empty()) {
                if (request.HasArg(TStrings::SESSION_ID)) {
                    // If sessionid argument is present, we're requested to validate Session_id cookie
                    method = TStrings::SESSION_ID;
                    TLog::Debug() << "Substitution occured: method=" << TStrings::SESSION_ID
                                  << ". Conumer=" << consumerInfo.GetName()
                                  << ". Ip=" << request.GetRemoteAddr();
                }
            }

            MethodCounters_.Inc(method);

            auto it = Handlers_.find(method);
            if (it == Handlers_.end()) {
                ErrorStats_.IncUnrecognizedRequest();
                // None of the supported paths has been taken which is an error
                throw NBb::TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Unrecognized method: " << InvalidValue(method);
            }

            // Run check here:
            // 1. to detect consumer
            // 2. to increment unistat values
            CheckDuplicatedArgs(request);

            TResponseCache::TCacheHolder cache = GetCache(request.GetArg(FORMAT) == JSON, consumerInfo.GetConsumer(), method);
            cache.BuildKey(request, consumerInfo.GetConsumer().GetName());

            NCache::EStatus status = NCache::EStatus::Unacceptable;
            const std::shared_ptr<const TString> maybeResponse = cache.GetValue(status);
            if (status == NCache::EStatus::Hit) {
                Consumers_.Cache.Add(consumerInfo);
            }

            if (maybeResponse) {
                WriteResult(request, HTTP_OK, *maybeResponse);
                return status;
            }

            TString response = it->second(request, consumerInfo.GetConsumer());
            WriteResult(request, HTTP_OK, response);

            cache.PutValue(std::make_shared<const TString>(std::move(response)));
            return status;
        } catch (const NBb::TDbpoolError& e) {
            ErrorStats_.IncCounter(e.Status());
            TString msg("Fatal BlackBox error: ");
            msg.append(e.what());
            // Send out error XML containing both error code and text message
            SendErrorStatus(request, TBlackboxError::EType::DbException, msg, HttpCodes_.DbException, method);
        } catch (const NBb::TBlackboxError& e) {
            ErrorStats_.IncCounter(e.Status());
            HttpCodes httpCode = HttpCodes_.EtcErrors;
            if (e.Status() == TBlackboxError::EType::AccessDenied) {
                Consumers_.Denied.Add(consumerInfo);
                httpCode = HttpCodes_.AccessDenied;
            }

            TString msg = NUtils::CreateStr(
                "BlackBox error: ",
                e.Status() == TBlackboxError::EType::AccessDenied ? "Access denied: " : "",
                e.what());
            // Send out error XML containing both error code and text message
            SendErrorStatus(request, e.Status(), msg, httpCode, method);
        } catch (const NDbPool::TException& e) {
            ErrorStats_.IncCounter(TBlackboxError::EType::DbException);
            TString msg = NUtils::CreateStr("Fatal BlackBox error: dbpool exception: ", e.what());
            // Send out error XML containing both error code and text message
            SendErrorStatus(request, TBlackboxError::EType::DbException, msg, HttpCodes_.DbException, method);
        } catch (const std::exception& e) {
            ErrorStats_.IncFatal();
            TString msg = NUtils::CreateStr("Fatal BlackBox error: unexpected exception: ", e.what());
            // Send out error XML containing both error code and text message
            SendErrorStatus(request, TBlackboxError::EType::Unknown, msg, HttpCodes_.FatalException, method);
        }

        return NCache::EStatus::Unacceptable;
    }

    template <typename... Args>
    static TString SerializeCheckGrants(const NCommon::TRequest& request,
                                        const TCheckGrantsResult& result,
                                        const Args&... args) {
        if (request.GetArg(FORMAT) == JSON) {
            TString res = NBb::TJsonSerializer::SerializeCheckGrants(result, args...);
            res.push_back('\n');
            return res;
        }

        return NBb::TXmlSerializer::SerializeCheckGrants(result, args...);
    }

    void TBlackbox::HandleCheckGrants(NCommon::TRequest& request) {
        TCheckGrantsResult result;
        TExceptionInfo info;

        try {
            info.Method = request.GetArg(TStrings::METHOD);

            const TConsumerInfo consumerInfo = Impl->GetConsumer(request);

            auto it = CheckGrantsHandlers_.find(info.Method);
            if (it == CheckGrantsHandlers_.end()) {
                throw NBb::TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Unrecognized method: " << InvalidValue(info.Method);
            }

            const TConsumer& consumer = consumerInfo.GetOptionalConsumer()
                                            ? *consumerInfo.GetOptionalConsumer()
                                            : TConsumer::Unknown();
            result = it->second(request, consumer);

            if (consumerInfo.GetError()) {
                result.Errors.insert(consumerInfo.GetError());
            }

            if (result.Errors.empty()) {
                // Consumer has all required grants
                WriteResult(request, HTTP_OK, SerializeCheckGrants(request, result));
                return;
            }

            // Provide correct error info
            info.Msg = NUtils::CreateStr(
                "Missing some grants for consumer: ",
                consumer.PrintId(request.GetRemoteAddr()));
            info.Status = TBlackboxError::EType::AccessDenied;
        } catch (const NBb::TBlackboxError& e) {
            info.Msg = e.what();
            info.Status = e.Status();
        } catch (const std::exception& e) {
            info.Msg = e.what();
            info.Status = TBlackboxError::EType::Unknown;
        }

        info.Host = request.GetHost();
        info.Hostname = Hostname_;

        HttpCodes status = HTTP_INTERNAL_SERVER_ERROR; // 500

        // Set correct HTTP status for expected exception statuses
        if (info.Status == TBlackboxError::EType::AccessDenied) {
            status = HTTP_FORBIDDEN; // 403
            info.ExtendedDescription = ACCESS_DENIED_MSG;
        } else if (info.Status == TBlackboxError::EType::InvalidParams) {
            status = HTTP_BAD_REQUEST; // 400
        }

        TLog::Debug()
            << "BlackBox: CheckGrants failed: <" << info.Msg
            << ">, <" << (result.Errors.empty() ? TString() : *result.Errors.begin())
            << "...>; consumer IP=<" << request.GetRemoteAddr() << ">";

        WriteResult(request, status, SerializeCheckGrants(request, result, &info));
    }

    static const TString IP_GRANTS_MTIME = "ip_grants_mtime";
    static const TString TVM_GRANTS_MTIME = "tvm_grants_mtime";
    void TBlackbox::ProcessPing(NCommon::TRequest& request) {
        // for monitoring
        // must be protected from outer access
        // by proper auth.conf
        if (request.HasArg(DBPOOL_EXTENDED_STATS)) {
            request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);
            request.Write(Impl->DbpoolExtendedStats());
            request.SetStatus(HTTP_OK);
            return;
        }

        request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_HTML);

        if (request.HasArg(DBPOOLSTATS)) {
            request.Write(Impl->DbpoolStats());
            request.SetStatus(HTTP_OK);
            return;
        }

        const time_t ipGrantsMTime = Impl->GetIpGrantsMTime();
        const time_t tvmGrantsMTime = Impl->GetTvmGrantsMTime();
        if (request.HasArg(IP_GRANTS_MTIME)) {
            request.Write(TStringBuilder() << ipGrantsMTime << Endl);
            request.SetStatus(HTTP_OK);
            return;
        }
        if (request.HasArg(TVM_GRANTS_MTIME)) {
            request.Write(TStringBuilder() << tvmGrantsMTime << Endl);
            request.SetStatus(HTTP_OK);
            return;
        }

        if (!ipGrantsMTime || !tvmGrantsMTime) {
            TLog::Warning() << "blackbox ping error: no valid grants config.";
            request.Write(BB_NO_GRANTS);
            request.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            return;
        }

        if (ForcedDown()) {
            TLog::Debug() << "blackbox ping error: externally forced out-of-service";
            request.Write(BB_FORCED_DOWN);
            request.SetStatus(HTTP_SERVICE_UNAVAILABLE);
            return;
        }

        // If any of the critical errors occurred, log this and return 500
        TString msg;
        if (!Impl->DbOk(msg)) {
            TLog::Debug() << "blackbox ping error: " << msg;
            request.Write(NUtils::CreateStr(PING_BB_ERR_HEAD, msg, PING_BB_ERR_TAIL));
            request.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            return;
        }

        TLog::Info() << "blackbox ping OK: " << msg;

        request.Write(PING_BB_OK);
        request.SetStatus(HTTP_OK);
    }

    void TBlackbox::ProcessHealthCheck(NCommon::TRequest& request) const {
        NUtils::TIpAddr remoteAddress;
        if (!remoteAddress.Parse(request.GetRemoteAddr()) || !remoteAddress.IsLoopback()) {
            request.SetStatus(HTTP_FORBIDDEN);
            return;
        }

        request.SetStatus(HTTP_OK);
        request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_PLAIN);

        request.Write(Impl->GetJugglerStatus());
    }

    template <typename ResultType>
    TString TBlackbox::ProcessRequest(NCommon::TRequest& request,
                                      const TConsumer& consumer,
                                      THandler<ResultType> handler) {
        std::unique_ptr<ResultType> result = (Impl.get()->*handler)(request, consumer);
        return serializeResult(request.GetArg(FORMAT) == JSON, *result);
    }

    template <class Processor>
    void TBlackbox::AddMethod(const TString& method) {
        MethodCounters_.Add(method);

        Handlers_.insert({
            method,
            [this](NCommon::TRequest& request, const TConsumer& consumer) {
                Processor p(*this->Impl, request);
                return SerializeResult(request.GetArg(FORMAT) == JSON, *p.Process(consumer));
            },
        });

        CheckGrantsHandlers_.insert({
            method,
            [this](NCommon::TRequest& request, const TConsumer& consumer) {
                Processor p(*this->Impl, request);
                return p.CheckGrants(consumer, false).Extract();
            },
        });
    }

    template <typename Result>
    TString TBlackbox::SerializeResult(bool isJson, const Result& result) {
        TString body;
        if (isJson) {
            body = NBb::TJsonSerializer::Serialize(result);
            body.push_back('\n');
        } else {
            body = NBb::TXmlSerializer::Serialize(result);
        }
        return body;
    }

    void TBlackbox::WriteResult(NCommon::TRequest& request, HttpCodes status, const TString& response) const {
        request.SetStatus(status);
        request.SetHeader(CONTENT_TYPE, request.GetArg(FORMAT) == JSON ? CONTENT_TYPE_JSON : CONTENT_TYPE_XML);
        if (ForceProvideRequestId_) {
            request.SetHeader(REQUEST_ID, request.GetRequestId());
        }
        request.Write(response);
    }

    bool TBlackbox::ForcedDown() const {
        return NFs::Exists(ForceDownFilePath_);
    }

    TResponseCache::TCacheHolder TBlackbox::GetCache(const bool isJson,
                                                     const TConsumer& consumer,
                                                     const TString& method) const {
        auto it = CacheByMethod_.find(method);
        if (it == CacheByMethod_.end()) {
            return {};
        }

        if (!consumer.IsMethodCacheable(method)) {
            return {};
        }

        return it->second.GetCache(isJson, it->second.DefaultLifeTime);
    }

    void TBlackbox::UpdateConsumerInfoForMethodsWithoutGrants(const NCommon::TRequest& request,
                                                              TConsumerInfo& consumerInfo) {
        // This check should be here to provide correct error info:
        //   * for unknown consumer without ServiceTicket:
        //       it can't get to TGrantsChecker anyway
        //   * for valid IP-consumer
        //       it can get to TGrantsChecker but it would be copy-paste
        if (request.GetHeader(TStrings::X_YA_SERVICE_TICKET).empty()) {
            const TConsumer& cons = consumerInfo.GetOptionalConsumer()
                                        ? *consumerInfo.GetOptionalConsumer()
                                        : TConsumer::Unknown();

            throw TBlackboxError(TBlackboxError::EType::AccessDenied)
                << "method=" << request.GetArg(TStrings::METHOD)
                << " allowed only with header '" << TStrings::X_YA_SERVICE_TICKET << "'. "
                << cons.PrintId(request.GetRemoteAddr());
        }

        // Consumer uses TVM - it is enough to use this method.
        // ServiceTicket must be valid, other params are not restricted:
        //   * is it in grants
        //   * is remote addr belongs to known consumer
        //   * etc

        // Consumer is totally valid - it can use method
        if (consumerInfo.IsOk()) {
            return;
        }

        const bool isServiceTicketValid = consumerInfo.GetTvmId() != 0;
        // Consumer uses invalid ServiceTicket:
        //   request will be failed with error info from `consumerInfo`
        if (!isServiceTicketValid) {
            return;
        }

        if (consumerInfo.GetOptionalConsumer()) {
            // Consumer is known but it came from illegal network
            consumerInfo.ClearError();
        } else {
            // Consumer is unknown. We need to create synthetic consumer
            //   because all code later depends on valid consumer
            auto consumer = std::make_shared<TConsumer>(consumerInfo.GetTvmId());
            consumer->SetName(NUtils::CreateStr("_unknown_tvmid=", consumerInfo.GetTvmId()));
            consumerInfo = TConsumerInfo(
                std::move(consumer),
                {},
                TConsumerInfo::EAuthType::Tvm2);
        }
    }
}
