#pragma once

#include "misc/exception.h"
#include "misc/response_cache.h"
#include "unistat/consumer_stats.h"
#include "unistat/error_stats.h"
#include "unistat/method_counters.h"

#include <passport/infra/libs/cpp/unistat/diff.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <util/generic/string.h>
#include <util/system/types.h>

#include <map>
#include <memory>
#include <optional>
#include <set>
#include <unordered_map>
#include <unordered_set>
#include <vector>

class TBlackboxHelper;

namespace NPassport::NUtils {
    class TFileLogger;
}

namespace NPassport::NCommon {
    class TRequest;
}

namespace NPassport::NUnistat {
    class TBuilder;
}

namespace NPassport::NXml {
    class TConfig;
}

namespace NPassport::NBb {
    class TBlackboxError;
    class TBlackboxImpl;
    class TCheckGrantsResult;
    class TConsumer;
    class TConsumerInfo;

    struct THttpCodes {
        HttpCodes AccessDenied = HTTP_OK;
        HttpCodes DbException = HTTP_OK;
        HttpCodes FatalException = HTTP_OK;
        HttpCodes EtcErrors = HTTP_OK;
    };

    class TBlackbox {
    public:
        explicit TBlackbox();
        ~TBlackbox();

        void Init(const NXml::TConfig& config);
        void HandleRequest(NCommon::TRequest& request);

        void AddUnistat(NUnistat::TBuilder& builder);
        void AddUnistatExtended(const TString& path, NUnistat::TBuilder& builder);

    private:
        void InitLoggers(const NXml::TConfig& config, const TString& componentXPath);
        void InitCache(const NXml::TConfig& config, const TString& componentXPath);
        void InitHttpCodes(const NXml::TConfig& config, const TString& path);

        void LogAccess(const NCommon::TRequest& request, TDuration timeSpent, NCache::EStatus cacheStatus);
        void LogAccessCommon(const NCommon::TRequest& request, TDuration timeSpent, NCache::EStatus cacheStatus);
        void LogAccessCheckGrants(const NCommon::TRequest& request, TDuration timeSpent);

        void SendErrorStatus(NCommon::TRequest& request,
                             const TBlackboxError::EType status,
                             const TString& msg,
                             const HttpCodes httpStatus,
                             const TString& method = TString()) const;
        static void SendBlackboxPingError(NCommon::TRequest& request, const TString& msg);

        static void CheckDuplicatedArgs(const NCommon::TRequest& req);

        NCache::EStatus Handle(NCommon::TRequest& request);
        void HandleCheckGrants(NCommon::TRequest& request);
        void ProcessPing(NCommon::TRequest& request);
        void ProcessHealthCheck(NCommon::TRequest& request) const;

        // template helper to call Blackbox method and serialize result
        template <typename ResultType>
        using THandler = std::unique_ptr<ResultType> (TBlackboxImpl::*)(const NCommon::TRequest&, const TConsumer&) const;

        template <typename ResultType>
        TString ProcessRequest(NCommon::TRequest& request,
                               const TConsumer& consumer,
                               THandler<ResultType> handler);

        template <class Processor>
        void AddMethod(const TString& method);

        template <typename Result>
        static TString SerializeResult(bool isJson, const Result& result);

        void WriteResult(NCommon::TRequest& request, HttpCodes status, const TString& response) const;

        bool ForcedDown() const;

        TResponseCache::TCacheHolder GetCache(const bool isJson, const TConsumer& consumer, const TString& method) const;

        static void UpdateConsumerInfoForMethodsWithoutGrants(const NCommon::TRequest& request,
                                                              TConsumerInfo& consumerInfo);

    public: // for tests
        std::unique_ptr<TBlackboxImpl> Impl;

    private:
        std::unique_ptr<NUtils::TFileLogger> AccessLogger_;
        std::unique_ptr<NUtils::TFileLogger> CheckGrantsLogger_;

        THttpCodes HttpCodes_;

        TString ForceDownFilePath_;

        TMethodCounters MethodCounters_;
        TErrorStats ErrorStats_;
        TConsumerStats Consumers_;

        template <class T>
        using THandlerMap = std::unordered_map<TString, std::function<T>>;

        THandlerMap<TString(NCommon::TRequest&, const TConsumer&)> Handlers_;
        THandlerMap<TCheckGrantsResult(NCommon::TRequest&, const TConsumer&)> CheckGrantsHandlers_;

        std::map<TString, TResponseCache> CacheByMethod_;
        TResponseCache::TContextPtr CacheCtx_;

        const TString Hostname_;
        bool ForceProvideRequestId_ = false;

        friend class ::TBlackboxHelper;
    };
}
