#include "blackbox_impl.h"

#include "badauth/facade.h"
#include "domain/domain_fetcher.h"
#include "gammas/fetcher.h"
#include "grants/consumer_info.h"
#include "grants/ip_grants.h"
#include "grants/tvm_grants.h"
#include "ip/usernets_list.h"
#include "loggers/authlog.h"
#include "loggers/tskvlog.h"
#include "misc/anonymiser.h"
#include "misc/attributes.h"
#include "misc/db_fetcher.h"
#include "misc/db_types.h"
#include "misc/dbfields_converter.h"
#include "misc/dbfields_settings.h"
#include "misc/exception.h"
#include "misc/experiment.h"
#include "misc/hosts_list.h"
#include "misc/ip_comparator.h"
#include "misc/lrandoms_dumper.h"
#include "misc/partitions.h"
#include "misc/passport_wrapper.h"
#include "misc/password_checker.h"
#include "misc/perimeter.h"
#include "misc/rate_limiter.h"
#include "misc/sess_kill_wrapper.h"
#include "misc/session_utils.h"
#include "misc/shards_map.h"
#include "misc/signer.h"
#include "misc/stopwords_list.h"
#include "misc/strings.h"
#include "misc/synthetic_attrs.h"
#include "misc/utils.h"
#include "misc/validator.h"
#include "oauth/config.h"
#include "staff/staff_info.h"
#include "totp/checker.h"
#include "totp/totp_encryptor.h"

#include <passport/infra/libs/cpp/auth_core/keymap.h>
#include <passport/infra/libs/cpp/auth_core/keyring.h>
#include <passport/infra/libs/cpp/auth_core/oauth_token_parser.h>
#include <passport/infra/libs/cpp/auth_core/public_id_encryptor.h>
#include <passport/infra/libs/cpp/auth_core/sessguard_parser.h>
#include <passport/infra/libs/cpp/auth_core/session.h>
#include <passport/infra/libs/cpp/auth_core/sessionparser.h>
#include <passport/infra/libs/cpp/auth_core/sessionsigner.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/tvm/common/private_key.h>
#include <passport/infra/libs/cpp/tvm/common/service_tickets.h>
#include <passport/infra/libs/cpp/tvm/dbpool/create_service_ticket_opt.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/regular_task.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/tvmauth/checked_service_ticket.h>
#include <library/cpp/tvmauth/src/rw/keys.h>

#include <util/string/ascii.h>
#include <util/system/env.h>
#include <util/system/fs.h>

namespace NPassport::NBb {
    static const TString MAIL_MACRO("mail");
    static const TString SMTP_MACRO("smtp");
    static const TString PASSPORT_MACRO("passport");
    static const TString RESTORE_MACRO("restore");
    static const TString MK_MACRO("mk");
    static const TString POP_MACRO("pop");

    static const TString NAROD_DOMAIN("narod.ru");

    static const TString SMTP_HOSTED_SIDS("2,105");

    class TStringSet {
    public:
        TStringSet(std::unordered_set<TString>&& entries)
            : Data_(std::move(entries))
        {
        }

        TStringSet(const TString& entryList) {
            std::vector<TString> entries = NUtils::ToVector(entryList, ",;");
            Data_.insert(entries.begin(), entries.end());
        }

        bool Contains(const TString& entry) const {
            return Data_.find(entry) != Data_.end();
        }

    private:
        std::unordered_set<TString> Data_;
    };

    TBlackboxImpl::TBlackboxImpl()
        : DefaultGuardSpaceOptions_(0, 0)
        , StrongpwdExpireTime_(90 * 24 * 3600)
        , YakeyBackupExpireTime_(90 * 24 * 3600)
        , IpComparator_(std::make_unique<TDirectIpComparator>())
        , DbCtx_(NDbPool::TDbPoolCtx::Create())
    {
    }

    TBlackboxImpl::~TBlackboxImpl() = default;

    void TBlackboxImpl::Init(const NXml::TConfig& config, const TString& xpathPrefix) {
        SetHostId(config.AsString(xpathPrefix + "/hostid"));
        InitLoggers(config, xpathPrefix);
        InitExperiments(config, xpathPrefix);
        InitTvm(config, xpathPrefix);

        // init DB connections before doing any other init!
        InitDb(config, xpathPrefix);

        InitRandomKeys(config, xpathPrefix);

        SupportLiteSess_ = config.AsBool(xpathPrefix + "/support_lite_sess", true);
        MinCharsCountBeforeWildcard_ = config.AsInt(xpathPrefix + "/min_chars_count_before_wildcard", 1);

        // init everything else only when db are set up
        InitPddRanges(config, xpathPrefix);
        InitAcl(config, xpathPrefix);
        InitGeobase(config, xpathPrefix);
        InitYandexIpList(config, xpathPrefix);
        InitMailDomains(config, xpathPrefix);
        InitAnonymiser(config, xpathPrefix);
        InitPerimeter(config, xpathPrefix);
        InitStaffInfo(config, xpathPrefix);
        InitOAuth(config, xpathPrefix);
        InitStrongPwdLimits(config, xpathPrefix);
        InitBadauth(config, xpathPrefix);
        InitSessKillWrapper(config, xpathPrefix);
        InitPassportWrapper(config, xpathPrefix);
        InitStopWords(config, xpathPrefix);
        InitMail4Domains(config, xpathPrefix);
        InitHosts(config, xpathPrefix);
        InitPartitionsSettings(config, xpathPrefix);
        InitAttrSettings(config, xpathPrefix);
        InitDbFieldsSettings(config, xpathPrefix);
        InitMiscOptions(config, xpathPrefix);
        InitTotpSettings(config, xpathPrefix);
        InitRfcTotpChecker(config, xpathPrefix);
        InitPasswordChecker(config, xpathPrefix);
        InitPublicIdSettings(config, xpathPrefix);
        InitLRandomsDumper(config, xpathPrefix);
        InitSigner(config, xpathPrefix);
        InitUserTicketCache(config, xpathPrefix);
        InitRateLimiter(config, xpathPrefix);
    }

    void TBlackboxImpl::InitDb(const NXml::TConfig& config, const TString& xpathPrefix) {
        // init central db
        CentralDb_ = DbCtx_->CreateDb(config,
                                      xpathPrefix + "/db_conf/central",
                                      xpathPrefix + "/centraldb_settings");
        CentralDb_->TryPing();

        CentralSlowDb_ = DbCtx_->CreateDb(config,
                                          xpathPrefix + "/db_conf/central",
                                          xpathPrefix + "/centraldb_slow_settings");
        CentralSlowDb_->TryPing();

        // init sharded db
        ShardsMap_ = std::make_unique<TRangedShardsMap>();
        ShardsMap_->Init(xpathPrefix, config, DbCtx_);
    }

    static NAuth::EEnvironmentType GetEnvironmentType(const TString& env) {
        if (env == "production") {
            return NAuth::EEnvironmentType::Production;
        }
        if (env == "testing") {
            return NAuth::EEnvironmentType::Testing;
        }
        if (env == "production_yateam") {
            return NAuth::EEnvironmentType::ProductionYateam;
        }
        if (env == "testing_yateam") {
            return NAuth::EEnvironmentType::TestingYateam;
        }
        if (env == "load") {
            return NAuth::EEnvironmentType::Load;
        }

        ythrow yexception() << "Unknown environment " << env;
    }

    void TBlackboxImpl::InitRandomKeys(const NXml::TConfig& config, const TString& xpathPrefix) {
        NAuth::TKeyRingSettings settings{
            .Signkeydepth = config.AsInt(xpathPrefix + "/sign_key_depth", 3),
            .GammaKeeper = NGammas::TFetcher::CreateKeeper(
                NGammaFetcher::TIssPartBaseFetcher::Create(config, xpathPrefix, TvmClient_),
                config,
                xpathPrefix),
        };

        OAuthEnv_ = GetEnvironmentType(config.As<TString>(xpathPrefix + "/gamma_fetcher/env_type"));

        TLog::Info("Init libauth (sign with key #%d)",
                   settings.Signkeydepth);

        SessSigner_ = NUtils::TRegularTaskDecorator<NAuth::TSessionSigner>::CreateUnique(
            {
                TDuration::Seconds(60),
                "sess_signer",
            },
            *CentralSlowDb_,
            settings);
        SessParser_ = std::make_unique<NAuth::TSessionParser>(*SessSigner_, OAuthEnv_);
        SessguardParser_ = std::make_unique<NAuth::TSessGuardParser>(*SessSigner_);
        InitKeyspaces(config, xpathPrefix);
        SessSigner_->CheckBeforeStart();

        OauthParser_ = NUtils::TRegularTaskDecorator<NAuth::TOAuthTokenParser>::CreateUnique(
            {
                TDuration::Seconds(60),
                "oauth_signer",
            },
            *CentralSlowDb_,
            settings);
    }

    void TBlackboxImpl::InitKeyspaces(const NXml::TConfig& config, const TString& xpathPrefix) {
        time_t warningTimeout = config.AsInt(xpathPrefix + "/key_age_warning__min", 125);

        std::unordered_set<TString> mdaDomains;
        // init keyspaces
        for (const TString& key : config.SubKeys(xpathPrefix + "/serve_kspaces/entry")) {
            TString keyspace = config.AsString(key + "/@id");
            time_t timeout = config.AsInt(key + "/@warning_timeout__min", warningTimeout);

            SessSigner_->AddKeyspace(keyspace, TDuration::Minutes(timeout));

            if (config.AsBool(key + "/@mda", false)) {
                mdaDomains.insert(keyspace);
            }
        }

        MdaDomains_ = std::make_unique<TStringSet>(std::move(mdaDomains));

        // init guardspaces
        for (const TString& key : config.SubKeys(xpathPrefix + "/serve_guard_spaces/guard_space")) {
            TString id = config.AsString(key + "/@id");
            TString keyspace = config.AsString(key + "/@name");

            std::vector<TString> hostnames = NUtils::ToVector(config.AsString(key + "/@hostnames"), ';');
            SessguardParser_->AddGuardSpace(id, keyspace, hostnames);

            GuardSpaceOptions_.emplace(
                keyspace,
                TSessGuardOptions(
                    config.As<size_t>(key + "/@enable_rate", 100),
                    config.As<size_t>(key + "/@need_reset_rate", 0)));
        }
    }

    void TBlackboxImpl::InitAcl(const NXml::TConfig& config, const TString& xpathPrefix) {
        TString aclpath = config.AsString(xpathPrefix + "/acl_path", "/etc/fastcgi2/available/grants.xml");
        ui32 aclperiod = config.AsInt(xpathPrefix + "/acl_check_period", 5000);
        IpGrants_ = std::make_unique<TIpGrants>(aclpath, TDuration::MilliSeconds(aclperiod));

        TString consGrantsPath = config.AsString(xpathPrefix + "/consumer_grants_path");
        ui32 consGrantsPeriod = config.AsInt(xpathPrefix + "/consumer_grants_check_period", 5);
        TvmGrants_ = std::make_unique<TTvmGrants>(consGrantsPath, TDuration::Seconds(consGrantsPeriod));
    }

    void TBlackboxImpl::InitGeobase(const NXml::TConfig& config, const TString& xpathPrefix) {
        const TString geobasePath = config.AsString(xpathPrefix + "/geobase", "");
        if (geobasePath.empty()) {
            return;
        }

        Y_ENSURE(NFs::Exists(geobasePath),
                 "File " << geobasePath << " does not exist");

        try {
            Lookup_ = std::make_unique<NGeobase::TLookup>(geobasePath.c_str());
        } catch (const std::exception& e) {
            TLog::Error("BlackBox error: exception during geobase load: %s, %s", geobasePath.c_str(), e.what());
        }

        if (Lookup_) {
            IpComparator_ = std::make_unique<TAsIpComparator>(*Lookup_);
            TLog::Info("Geobase configured at %s", geobasePath.c_str());
        }
    }

    void TBlackboxImpl::InitYandexIpList(const NXml::TConfig& config, const TString& xpathPrefix) {
        EnableYandexIpCheck_ = config.AsBool(xpathPrefix + "/enable_yandex_ip_check", false);

        TString usernetsPath = config.AsString(xpathPrefix + "/usernets_path");

        UsernetsList_ = std::make_unique<TUsernetsList>(NUtils::ReadFile(usernetsPath));
    }

    void TBlackboxImpl::InitMailDomains(const NXml::TConfig& config, const TString& xpathPrefix) {
        std::vector<TString> domains = config.SubKeys(xpathPrefix + "/mail_domains/domain");

        if (domains.empty()) {
            throw yexception() << "Failed to load mail domains list: it must not be empty";
        }

        for (const TString& dom : domains) {
            TString name = config.AsString(dom + "/@name");
            TString strCountries = config.AsString(dom + "/@countries", TStrings::EMPTY);
            TString strSid = config.AsString(dom + "/@sid", "");
            TString strDomid = config.AsString(dom + "/@domid", "");
            TString strPhoneCountry = config.AsString(dom + "/@phone_country", "");

            if (name.empty() || name[0] == '.') {
                throw yexception() << "Failed to load mail domains list: got invalid domain name='" << name << "'";
            }

            TYandexDomains::TDomainData data(name);

            if (AsciiEqualsIgnoreCase(strCountries, "any")) {
                data.SetAny(true);
            } else { // need to parse country list
                for (const TString& c : NUtils::NormalizeListValue(strCountries, ",")) {
                    data.Add(c);
                }
            }

            data.SetSid(strSid);
            data.SetDomid(strDomid);
            data.SetPhoneCountry(strPhoneCountry);

            YandexDomains_.Insert(data);

            TLog::Info("Inserted main domain: '%s' for countries: %s", name.c_str(), strCountries.c_str());
        }

        TString otherCountryDefault = config.AsString(xpathPrefix +
                                                      "/mail_domains/default_domain/@any_country");

        if (otherCountryDefault.empty()) {
            throw yexception() << "Failed to load mail domains list: no default domain for unconfigured countries";
        }

        YandexDomains_.SetOtherCountryDefault(otherCountryDefault);

        for (const TString& en : config.SubKeys(xpathPrefix + "/mail_domains/default_domain/entry")) {
            TString country = config.AsString(en + "/@country");
            TString domain = config.AsString(en + "/@domain");

            if (country.empty() || domain.empty()) {
                TLog::Info("Invalid default domain setting: country='%s', domain='%s'",
                           country.c_str(),
                           domain.c_str());
                continue;
            }

            YandexDomains_.AddDefault(country, domain);

            TLog::Info("Added default domain '%s' for country %s", domain.c_str(), country.c_str());
        }

        TLog::Info("For other countries default domain is '%s'", otherCountryDefault.c_str());
    }

    void TBlackboxImpl::InitAnonymiser(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/anonymiser")) {
            Anonymiser_.reset();
            TLog::Info("Anonymiser: OFF");
            return;
        }

        Anonymiser_ = std::make_unique<TAnonymiser>();

        // init attributes mapping
        for (const TString& at : config.SubKeys(xpathPrefix + "/anonymiser/attribute")) {
            TString name = config.AsString(at + "/@id");
            TString value = config.AsString(at);

            Anonymiser_->AddAttribute(name, value);

            TLog::Info("Anonymiser: Attribute <%s> value is mapped to '%s'", name.c_str(), value.c_str());
        }

        // init email address mapping
        if (config.Contains(xpathPrefix + "/anonymiser/emails")) {
            TString emailDomainPrefix = config.AsString(xpathPrefix + "/anonymiser/emails/@domain_prefix");
            TString emailDomainBody = config.AsString(xpathPrefix + "/anonymiser/emails/@domain_body");

            Anonymiser_->SetEmailDomain(emailDomainPrefix, emailDomainBody);

            TLog::Info("Anonymiser: Emails will be mapped to <login>@%s<N>%s<domain>", emailDomainPrefix.c_str(), emailDomainBody.c_str());
        } else {
            TLog::Info("Anonymiser: Emails mapping OFF");
        }

        // init nickname mapping
        if (config.Contains(xpathPrefix + "/anonymiser/nickname")) {
            TString nicknameRu = config.AsString(xpathPrefix + "/anonymiser/nickname/@ru");
            TString nicknameEn = config.AsString(xpathPrefix + "/anonymiser/nickname/@en");

            Anonymiser_->SetNickname(nicknameRu, nicknameEn);

            TLog::Info("Anonymiser: Nickname and display_name will be mapped to %s/%s", nicknameRu.c_str(), nicknameEn.c_str());
        } else {
            TLog::Info("Anonymiser: nickname and display_name address mapping OFF");
        }
    }

    void TBlackboxImpl::InitPerimeter(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/perimeter")) {
            Perimeter_.reset();
            TLog::Info("Perimeter: OFF");
            return;
        }

        // perimeter is critical, don't start if configured but broken
        std::shared_ptr<NDbPool::TDbPool> db = DbCtx_->CreateDb(config, xpathPrefix + "/perimeter");
        db->TryPing();

        Perimeter_ = std::make_unique<TPerimeterWrapper>(std::move(db));

        TLog::Info("Perimeter: ON");
    }

    void TBlackboxImpl::InitStaffInfo(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/staff_fetcher")) {
            StaffInfo_.reset();
            TLog::Info("StaffFetcher: OFF");
            return;
        }

        TStaffFetcher::TSettings staffSettings = {
            .CachePath = config.As<TString>(xpathPrefix + "/staff_fetcher/disk_cache_path"),
            .UpdatePeriod = TDuration::Seconds(config.As<ui64>(xpathPrefix + "/staff_fetcher/update_period", 3600)),
            .ClientSettings = {
                .Host = config.As<TString>(xpathPrefix + "/staff_fetcher/host"),
                .Port = config.As<ui16>(xpathPrefix + "/staff_fetcher/port"),
                .ConnectionTimeout = TDuration::MilliSeconds(config.As<ui64>(xpathPrefix + "/staff_fetcher/connection_timeout", 5000)),
                .QueryTimeout = TDuration::MilliSeconds(config.As<ui64>(xpathPrefix + "/staff_fetcher/query_timeout", 5000)),
                .Retries = config.As<ui32>(xpathPrefix + "/staff_fetcher/retries_per_request", 3),
                .Limit = config.As<ui32>(xpathPrefix + "/staff_fetcher/limit_per_request", 1000),
                .TvmClient = TvmClient_,
            },
        };

        StaffInfo_ = std::make_unique<TStaffInfo>(std::move(staffSettings));

        TLog::Info("StaffFetcher: ON");

        TString externalRobotsWhiteList = NUtils::ReadFile(config.AsString(xpathPrefix + "/external_robots_white_list"));
        for (auto& uidStr : NUtils::ToVector(externalRobotsWhiteList, '\n', 104)) {
            ui64 uid;
            NUtils::Trim(uidStr);
            if (uidStr.empty() || uidStr.StartsWith("#")) {
                continue;
            }
            Y_ENSURE(TryIntFromString<10>(uidStr, uid),
                     "invalid uid in external robots white list: " << uidStr << " " << uidStr.size());

            AllowedExternalRobotUids_.insert(uid);
        }

        TLog::Info("BlackBox: got %lu allowed external robots", AllowedExternalRobotUids_.size());
    }

    void TBlackboxImpl::InitOAuth(const NXml::TConfig& config, const TString& xpathPrefix) {
        OauthConfig_ = std::make_unique<TOAuthConfig>(*OauthLogger_);
        OauthConfig_->Init(config, xpathPrefix, DbCtx_);
    }

    // SOX: читаем из конфигурационных файлов настройки строгой парольной политики:
    // - количество разрешенных ошибочных попыток логина (по-умолчанию: 5)
    // - срок действия пароля (по-умолчанию: 90 дней)
    // Записываем полученные настройки в лог-файл
    void TBlackboxImpl::InitStrongPwdLimits(const NXml::TConfig& config, const TString& xpathPrefix) {
        StrongpwdExpireTime_ = config.AsInt(xpathPrefix + "/strongpwd_limits/expire_time", 0);

        if (!StrongpwdExpireTime_) {
            TLog::Info("BlackBox: strongpwd expire_time limit unconfigured! Using 90 days by default");
            StrongpwdExpireTime_ = 90 * 24 * 3600;
        }

        if (StrongpwdExpireTime_ > 0) {
            TLog::Info("BlackBox: strongpwd expire_time limit is set to %d seconds (%d days)", StrongpwdExpireTime_, StrongpwdExpireTime_ / 24 / 3600);
        } else {
            TLog::Info("BlackBox: Warning: strongpwd expire_time limit is OFF");
        }
    }

    void TBlackboxImpl::InitBadauth(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.AsBool(xpathPrefix + "/badauth/enabled", true)) {
            return;
        }

        BadauthFacade_ = std::make_unique<TBadauthFacade>(TvmClient_);
        BadauthFacade_->Init(config, xpathPrefix, DbCtx_);
    }

    void TBlackboxImpl::InitSessKillWrapper(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/sess_kill")) {
            SessKill_.reset();
            TLog::Info("SessKill wrapper: OFF");
            return;
        }

        std::shared_ptr<NDbPool::TDbPool> db;
        try {
            db = DbCtx_->CreateDb(config, xpathPrefix + "/sess_kill");
            db->TryPing();
        } catch (const std::exception& e) {
            TLog::Info() << "SessKill wrapper: unable to initialize pool on start, ignoring: " << e.what();
        }

        SessKill_ = std::make_unique<TSessKillWrapper>(StatboxLogger_.get(), std::move(db));
        TLog::Info("SessKill wrapper: initialized");
    }

    void TBlackboxImpl::InitPassportWrapper(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/passport_pool")) {
            Passport_.reset();
            TLog::Info("Passport wrapper: OFF");
            return;
        }

        NDbPool::TQueryOpts opts = {NTvmDbPool::CreateServiceTicketOpt(
            TvmClient_,
            NTvmCommon::TServiceTickets::PASSPORTAPI_)};

        std::shared_ptr<NDbPool::TDbPool> db;
        try {
            db = DbCtx_->CreateDb(config, xpathPrefix + "/passport_pool", std::move(opts));
            db->TryPing();
        } catch (const std::exception& e) {
            TLog::Info() << "Passport wrapper: unable to initialize passport pool on start, ignoring: " << e.what();
        }

        TString consumer = config.AsString(xpathPrefix + "/passport_consumer", "blackbox");
        Passport_ = std::make_unique<TPassportWrapper>(std::move(db), consumer);
        TLog::Info("Passport wrapper: initialized");
    }

    void TBlackboxImpl::InitStopWords(const NXml::TConfig& config, const TString& xpathPrefix) {
        long reload = config.AsInt(xpathPrefix + "/stopwords_update_period", 7200);
        TString sid66Logins = config.AsString(xpathPrefix + "/sid_66_logins", TStrings::EMPTY);

        StopWords_ = NUtils::TRegularTaskDecorator<TStopWordsList>::CreateUnique(
            {
                TDuration::Seconds(reload),
                "bb_ref_stopw",
            },
            *CentralSlowDb_,
            sid66Logins);

        TLog::Info("Stopwords list reload period set to %ld sec", reload);
        TLog::Info("Stopwords list: added logins from sid 66: '%s'", sid66Logins.c_str());
    }

    void TBlackboxImpl::InitMail4Domains(const NXml::TConfig& config, const TString& xpathPrefix) {
        ui32 cachePeriod = config.AsInt(xpathPrefix + "/mail_for_domains/domains_update_period", 60);
        TLog::Info("PDD domains: hosted domains list load period: %u sec", cachePeriod);

        ui32 pageSize = config.AsInt(
            xpathPrefix + "/mail_for_domains/page_size", 100000);
        TLog::Info("PDD domains: page list size: <%u>", pageSize);

        ui32 pageRetryCount = config.AsInt(
            xpathPrefix + "/mail_for_domains/page_retry_count", 3);
        TLog::Info("PDD domains: page retry count: <%u>", pageRetryCount);

        TString cacheFile = config.AsString(
            xpathPrefix + "/mail_for_domains/cache_file", "");
        TLog::Info() << "PDD domains: cache file: " << cacheFile;

        TDuration cacheFileWritePeriod = TDuration::Seconds(config.AsInt(
            xpathPrefix + "/mail_for_domains/cache_file_write_period__sec", 3600));
        TLog::Info() << "PDD domains: cache file update period: " << cacheFileWritePeriod;

        HostedDomains_ = NUtils::TRegularTaskDecorator<TDomainFetcher>::CreateUnique(
            {
                TDuration::Seconds(cachePeriod),
                "bb_ref_domain",
            },
            *CentralSlowDb_,
            TDomainFetcherSettings{
                .PageSize = pageSize,
                .PageRetryCount = pageRetryCount,
                .CacheFile = cacheFile,
                .CacheFileWritePeriod = cacheFileWritePeriod,
            });
    }

    void TBlackboxImpl::InitHosts(const NXml::TConfig& config, const TString& xpathPrefix) {
        long reload = config.AsInt(xpathPrefix + "/hosts_update_period", 3600);

        Hosts_ = NUtils::TRegularTaskDecorator<THostsList>::CreateUnique(
            {
                TDuration::Seconds(reload),
                "bb_ref_hosts",
            },
            *CentralSlowDb_);

        TLog::Info("Hosts list reload period set to %ld sec", reload);
    }

    void TBlackboxImpl::InitPartitionsSettings(const NXml::TConfig& config, const TString& xpathPrefix) {
        TString defaultPartition = config.AsString(xpathPrefix + "/partitions/@default");

        TPartitionsSettings::TPartitionIdByName partitions;
        for (const TString& xpath : config.SubKeys(xpathPrefix + "/partitions/partition")) {
            ui64 id = config.AsNum<ui64>(xpath + "/@id");
            TString name = config.AsString(xpath + "/@name");

            bool inserted = partitions.emplace(name, id).second;
            Y_ENSURE(inserted, "Duplicated partition '" << name << "'");
        }

        PartitionsSettings_ = std::make_unique<TPartitionsSettings>(std::move(partitions), defaultPartition);
    }

    void TBlackboxImpl::InitAttrSettings(const NXml::TConfig& config, const TString& xpathPrefix) {
        TString requireGrants = config.AsString(xpathPrefix + "/attributes/require_grants", TStrings::EMPTY);
        TString binAttrs = config.AsString(xpathPrefix + "/attributes/binary", TStrings::EMPTY);
        TString deleted = config.AsString(xpathPrefix + "/attributes/deleted", TStrings::EMPTY);
        TString phoneGrants = config.AsString(xpathPrefix + "/phone_attributes/require_grants", TStrings::EMPTY);

        AttrSettings_ = std::make_unique<TAttributesSettings>(requireGrants, binAttrs, phoneGrants, deleted);

        TLog::Info("Attributes that require access grants: %s", requireGrants.c_str());
        TLog::Info("Attributes considered binary: %s", binAttrs.c_str());
        TLog::Info("Attributes considered deleted: %s", deleted.c_str());
        TLog::Info("Phone attributes that require access grants: %s", phoneGrants.c_str());
    }

    void TBlackboxImpl::InitDbFieldsSettings(const NXml::TConfig& config, const TString& xpathPrefix) {
        TString reqGrants = config.AsString(xpathPrefix + "/dbfields/require_grants", TStrings::EMPTY);
        DbfieldsSettings_ = std::make_unique<TDbFieldsSettings>(reqGrants);
        TLog::Info("DbFields that require access grants: %s", reqGrants.c_str());
    }

    void TBlackboxImpl::InitTotpSettings(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/totp")) {
            TotpEncryptor_.reset();
            TLog::Info("TOTP: not configured");
            return;
        }

        const int digitsLen = config.AsInt(xpathPrefix + "/totp/digits_len",
                                           NRfc4226::THotpFactory::maxHotpDigitsLength);
        const int lettersLen = config.AsInt(xpathPrefix + "/totp/letters_len",
                                            NRfc4226::THotpFactory::defaultHotpLettersLength);

        const int totpWindow = config.AsInt(xpathPrefix + "/totp/window", 2);

        TotpService_ = std::make_unique<NTotp::TService>(digitsLen, lettersLen, totpWindow);

        TString aesKeysPath = config.AsString(xpathPrefix + "/totp/aes_keys_path");
        unsigned aesDefaultKeyId = config.AsNum<unsigned>(xpathPrefix + "/totp/aes_default_key_id");
        TString hmacKeysPath = config.AsString(xpathPrefix + "/totp/hmac_keys_path");
        unsigned hmacDefaultKeyId = config.AsNum<unsigned>(xpathPrefix + "/totp/hmac_default_key_id");

        TLog::Info("TOTP: AES keys file (default_id=%d): %s", aesDefaultKeyId, aesKeysPath.c_str());
        TLog::Info("TOTP: MAC keys file (default_id=%d): %s", hmacDefaultKeyId, hmacKeysPath.c_str());

        try {
            TotpEncryptor_ = std::make_unique<TTotpEncryptor>(
                NAuth::TKeyMap::CreateFromFile(aesKeysPath, aesDefaultKeyId),
                NAuth::TKeyMap::CreateFromFile(hmacKeysPath, hmacDefaultKeyId));
        } catch (const std::exception& e) {
            TLog::Error("TOTP: FATAL ERROR: couldn't read keys: %s", e.what());
            throw yexception() << "TOTP encryption is not initialized";
        }

        TLog::Info("TOTP: Initialized with digits_len=%d, letters_len=%d, window=%d", digitsLen, lettersLen, totpWindow);
    }

    void TBlackboxImpl::InitRfcTotpChecker(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/totp_rfc")) {
            TLog::Info("TOTP RFC: not configured");
            return;
        }

        const int totpWindow = config.AsInt(xpathPrefix + "/totp_rfc/window", 1);
        TotpServiceRfc_ = std::make_unique<NTotp::TServiceRfc>(totpWindow);
        TLog::Info("TOTP RFC: configured");
    }

    void TBlackboxImpl::InitTvm(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/tvm")) {
            TLog::Info("BlackBox: TVM ticket signer is OFF");
            return;
        }

        Env_ = NTvmCommon::TPrivateKey::TranslateEnv(config.AsString(xpathPrefix + "/tvm/env"));
        TvmSelfId_ = config.AsInt(xpathPrefix + "/tvm/self_tvmid");

        NTvmAuth::NTvmApi::TClientSettings settings;

        TString secretFile = config.AsString(xpathPrefix + "/tvm/secret_path");
        settings.SetDiskCacheDir(config.AsString(xpathPrefix + "/tvm/disk_cache_dir"));
        settings.SetSelfTvmId(TvmSelfId_);
        settings.EnableServiceTicketChecking();
        settings.EnableUserTicketChecking(Env_);

        NTvmAuth::NTvmApi::TClientSettings::TDstMap dsts;
        for (const TString& alias : {
                 NTvmCommon::TServiceTickets::KOLMOGOR_,
                 NTvmCommon::TServiceTickets::TVMAPI_,
                 NTvmCommon::TServiceTickets::ISSNKMS_,
                 NTvmCommon::TServiceTickets::PASSPORTAPI_,
                 NTvmCommon::TServiceTickets::STAFFAPI_,
             }) {
            dsts.insert({
                alias,
                config.AsInt(xpathPrefix + "/tvm/destinations/destination[@name='" + alias + "']"),
            });
        }

        settings.EnableServiceTicketsFetchOptions(NUtils::ReadFile(secretFile), std::move(dsts));

        TString tvmHost = config.AsString(xpathPrefix + "/tvm/host", settings.TvmHost);
        ui16 tvmPort = config.AsInt(xpathPrefix + "/tvm/port", settings.TvmPort);
        settings.SetTvmHostPort(tvmHost, tvmPort);

        TvmClient_ = std::make_shared<NTvmAuth::TTvmClient>(
            settings,
            NTvmLogger::TLogger::Create());

        InitTicketSigner2(config, xpathPrefix, settings);
    }

    void TBlackboxImpl::InitTicketSigner2(const NXml::TConfig& config, const TString& xpathPrefix, const NTvmAuth::NTvmApi::TClientSettings& settings) {
        if (!config.AsBool(xpathPrefix + "/ticket_signer/v2_enabled", false)) {
            TLog::Info("BlackBox: Ticket signer 2 is OFF");
            return;
        }

        TLog::Info("BlackBox: Ticket signer 2 is starting to initialize");

        if (config.AsBool(xpathPrefix + "/ticket_signer/secure_heap_enabled", false)) {
            const size_t secureHeapSize = config.AsNum<size_t>(
                xpathPrefix + "/ticket_signer/secure_heap_size",
                16 * 1024 * 1024);
            Y_ENSURE(std::bitset<64>(secureHeapSize).count() == 1,
                     "secure_heap_size must be > 0 and be power of 2");
            const int secureHeapMinSize = config.AsInt(
                xpathPrefix + "/ticket_signer/secure_heap_minsize",
                16);
            Y_ENSURE(std::bitset<32>(secureHeapMinSize).count() <= 1,
                     "secure_heap_minsize must be power of 2");
            NTvmAuth::NRw::TSecureHeap::Init(secureHeapSize, secureHeapMinSize);
        }

        const int period = config.AsInt(xpathPrefix + "/ticket_signer/refresh_period", 3600);
        const size_t preferedKeyIdx = config.AsInt(xpathPrefix + "/ticket_signer/prefered_key_idx", 7);
        UserTicketTtl_ = config.AsInt(xpathPrefix + "/ticket_signer/user_ticket_ttl", 300);

        ui64 cacheTtl = config.AsNum<ui64>("/ticket_signer/user_ticket_ttl", 432000);
        ui64 jugglerWarningTimeout = config.AsNum<ui64>("/ticket_signer/juggler/warning_timeout", 7200);
        ui64 jugglerCriticalTimeout = config.AsNum<ui64>("/ticket_signer/juggler/critical_timeout", 86400);

        try {
            NTvmCommon::TPrivateKeySettings pkSettings{
                .TvmHost = settings.TvmHost,
                .TvmPort = settings.TvmPort,
                .TvmCacheDir = settings.DiskCacheDir,
                .CacheTtl = TDuration::Seconds(cacheTtl),
                .JugglerWarningTimeout = TDuration::Seconds(jugglerWarningTimeout),
                .JugglerCriticalTimeout = TDuration::Seconds(jugglerCriticalTimeout),
                .PreferredKeyIdx = preferedKeyIdx,
                .Period = TDuration::Seconds(period),
                .Env = Env_,
            };

            TvmPrivateKey_ = NUtils::TRegularTaskDecorator<NTvmCommon::TPrivateKey>::CreateUnique(
                {
                    .Period = pkSettings.RetryPeriod,
                    .Name = "private_key_fetcher",
                },
                std::move(pkSettings),
                TvmClient_);

            TvmPrivateKey_->SelfCheck();
        } catch (const std::exception& e) {
            throw yexception() << "config is broken: ticket signer 2 can't work with this config: " << e.what();
        }

        TLog::Info("BlackBox: Ticket signer 2 is ON");
    }

    void TBlackboxImpl::InitPddRanges(const NXml::TConfig& config, const TString& xpathPrefix) {
        Mail4domainsFirstUid_ = config.AsNum<ui64>(xpathPrefix + "/mail_for_domains/first_uid", 1130000000000000ull);
        Mail4domainsLastUid_ = config.AsNum<ui64>(xpathPrefix + "/mail_for_domains/last_uid", 2260000000000000ull);

        TLog::Info("mail4domains: uid space: %ld..%ld", Mail4domainsFirstUid_, Mail4domainsLastUid_);
    }

    void TBlackboxImpl::InitPasswordChecker(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.AsBool(xpathPrefix + "/password/argon/enabled", false)) {
            PasswdChecker_ = std::make_unique<TPasswordChecker>();
            TLog::Info("Blackbox: Argon disabled");
        } else {
            const TString secretFile = config.AsString(xpathPrefix + "/password/argon/secrets_path");
            unsigned secretId = config.AsNum<unsigned>(xpathPrefix + "/password/argon/default_secret_id");
            NAuth::TKeyMap secrets = NAuth::TKeyMap::CreateFromFile(secretFile, secretId);
            size_t secretsCount = secrets.GetSize();

            ui32 hashlen = config.AsInt(xpathPrefix + "/password/argon/hashlen", 32);
            ui32 tcost = config.AsInt(xpathPrefix + "/password/argon/tcost", 1);
            ui32 mcost = config.AsInt(xpathPrefix + "/password/argon/mcost", 4096);

            PasswdChecker_ = std::make_unique<TPasswordChecker>(std::move(secrets), hashlen, tcost, mcost);
            TLog::Info("Blackbox: Argon enabled. Hashlen=%u. tcost=%u. mcost=%u. secrets_count=%lu. default_id=%d. file=%s.",
                       hashlen,
                       tcost,
                       mcost,
                       secretsCount,
                       secretId,
                       secretFile.c_str());
        }
    }

    void TBlackboxImpl::InitPublicIdSettings(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/public_id")) {
            PublicIdEncryptor_.reset();
            TLog::Info("PublicId settings not configured");
            return;
        }

        TString aesKeysPath = config.AsString(xpathPrefix + "/public_id/aes_keys_path");
        unsigned aesDefaultKeyId = config.AsNum<unsigned>(xpathPrefix + "/public_id/aes_default_key_id");

        TLog::Info("PublicId: AES keys file (default_id=%d): %s", aesDefaultKeyId, aesKeysPath.c_str());

        try {
            PublicIdEncryptor_ = std::make_unique<NAuth::TPublicIdEncryptor>(
                NAuth::TKeyMap::CreateFromFile(aesKeysPath, aesDefaultKeyId));
        } catch (const std::exception& e) {
            TLog::Error("PublicId: FATAL ERROR: couldn't read keys: %s", e.what());
            throw yexception() << "PublicId encryption is not initialized (no keys)";
        }

        TLog::Info("PublicId: Initialized");
    }

    void TBlackboxImpl::InitExperiments(const NXml::TConfig& config, const TString& xpathPrefix) {
        TExperiment::GetMutable().IsPddPartnerTokenEnabled = config.AsBool(xpathPrefix + "/experiment/pdd_token", true);
        TExperiment::GetMutable().IsHavePasswordEnabled = config.AsBool(xpathPrefix + "/experiment/have_password", true);
        TExperiment::GetMutable().IsHaveHintEnabled = config.AsBool(xpathPrefix + "/experiment/have_hint", true);

        TExperiment::GetMutable().CookieSignMethod = config.AsString(xpathPrefix + "/experiment/cookie_sign_method", "");
        TExperiment::GetMutable().TokenSignMethod = config.AsString(xpathPrefix + "/experiment/token_sign_method", "");
        TExperiment::GetMutable().AliasSignMethod = config.AsString(xpathPrefix + "/experiment/alias_sign_method", "");
        TExperiment::GetMutable().MainAttrsSignMethod = config.AsString(xpathPrefix + "/experiment/main_attrs_sign_method", "");
        TExperiment::GetMutable().FullAttrsSignMethod = config.AsString(xpathPrefix + "/experiment/full_attrs_sign_method", "");
        TExperiment::GetMutable().BadauthUseHalfIpv6AddressRate = config.AsNum<size_t>(xpathPrefix + "/experiment/badauth_use_half_ipv6_address", 0);
        TExperiment::GetMutable().RestrictFederalUsers = config.AsBool(xpathPrefix + "/experiment/restrict_federal_users", false);

        TExperiment::GetMutable().InvalidSessionidRate = config.AsNum<size_t>(xpathPrefix + "/experiment/invalid_sessionid_rate", 0);

        TLog::Info() << "Sign experiment settings: cookies='" << TExperiment::Get().CookieSignMethod
                     << "', tokens='" << TExperiment::Get().TokenSignMethod
                     << "', aliases='" << TExperiment::Get().AliasSignMethod
                     << "', main attributes='" << TExperiment::Get().MainAttrsSignMethod
                     << "', all attributes='" << TExperiment::Get().FullAttrsSignMethod << "'";
    }

    void TBlackboxImpl::InitMiscOptions(const NXml::TConfig& config, const TString& xpathPrefix) {
        SupportBanTime_ = config.AsBool(xpathPrefix + "/support_ban_time", false);

        FamilyKidsLimit_ = config.AsNum<ui16>(xpathPrefix + "/family_kids_limit", 4);

        AllowStressCookies_ = config.AsBool(xpathPrefix + "/allow_stress_test_cookies", false);

        TLog::Info() << "Blackbox: Stress test session cookies allowed: " << AllowStressCookies_;

        MultisessionUserLimit_ = config.AsInt(xpathPrefix + "/multisession_user_limit", 5);

        TLog::Info("Blackbox: maximum user count allowed for multisession cookie: %d", MultisessionUserLimit_);

        YakeyBackupExpireTime_ = config.AsInt(xpathPrefix + "/yakey_backup_expire_time", 90 * 24 * 3600);

        TLog::Info("BlackBox: yakey_backup_expire_time limit is set to %d seconds (%d days)", YakeyBackupExpireTime_, YakeyBackupExpireTime_ / 24 / 3600);

        MultisecretLimit_ = config.AsInt(xpathPrefix + "/totp/multisecret_limit", 20);

        TLog::Info("Blackbox: maximum TOTP secrets allowed for user: %d", MultisecretLimit_);

        MultiuserWarnLimit_ = config.AsInt(xpathPrefix + "/multiuser_warning_limit", 0);
        MultiuserLimit_ = config.AsInt(xpathPrefix + "/multiuser_limit", 0);

        TLog::Info("Blackbox: userinfo multiuser limits: warning=%s, error=%s",
                   MultiuserWarnLimit_ ? IntToString<10>(MultiuserWarnLimit_).c_str() : "OFF",
                   MultiuserLimit_ ? IntToString<10>(MultiuserLimit_).c_str() : "OFF");

        FindByPhoneNumbersLimit_ = config.AsInt(xpathPrefix + "/find_by_phone_numbers_limit", 200);

        TLog::Info("Blackbox: maximum phone numbers allowed in method=find_by_phone_numbers: %d", FindByPhoneNumbersLimit_);

        TString digitLogins = config.AsString(xpathPrefix + "/digit_logins", TStrings::EMPTY);

        DigitLogins_ = std::make_unique<TStringSet>(digitLogins);

        TLog::Info("BlackBox: reserved list of digit logins: '%s'", digitLogins.c_str());

        TString authtype = config.AsString(xpathPrefix + "/allowed_authtypes", TStrings::EMPTY);

        std::vector<TString> typesVec = NUtils::ToVectorWithEmpty(authtype, ',');
        AllowedAuthtypes_.insert(typesVec.begin(), typesVec.end());

        TLog::Info("BlackBox: allowed authtype values: '%s'", authtype.c_str());

        EnablePhoneAliases_ = config.AsBool(xpathPrefix + "/enable_phone_aliases", true);

        TLog::Info("BlackBox: phone aliases %s", EnablePhoneAliases_ ? "ENABLED" : "DISABLED");

        // init timezone, use MSK if not configured else
        TString oldtz = GetEnv("TZ");
        if (!oldtz) { // no TZ
            SetEnv("TZ", "Europe/Moscow");
            tzset();
        }
        TLog::Info("Blackbox: timezone is '%s'", getenv("TZ"));

        TDbFieldsConverter::TSidsToAttrMap sidsmap;

        for (const TString& en : config.SubKeys(xpathPrefix + "/attributes/sid")) {
            TString sid = config.AsString(en + "/@id");
            TString attr = config.AsString(en + "/@attr");
            sidsmap.insert(std::make_pair(sid, attr));
        }

        TDbFieldsConverter::InitMaps(std::move(sidsmap));

        DeviceSignatureMaxAge_ = config.AsNum<ui64>(xpathPrefix + "/device_signature/max_age", 86400);

        FullFioInPublicNameStartTime_ = config.AsInt(xpathPrefix + "/full_fio_in_public_name_start_time", 1609500000);

        TLog::Info("BlackBox: full FIO in public name for users registered since %ld", FullFioInPublicNameStartTime_);

        TSyntheticAttributes::TProxyAttrsMap proxyAttributes;
        for (const TString& xpath : config.SubKeys(xpathPrefix + "/attributes/proxy")) {
            TString synt = config.As<TString>(xpath + "/@synt");
            TString real = config.As<TString>(xpath + "/@real");
            proxyAttributes.emplace(synt, real);
        }

        SyntheticAttrs_ = std::make_unique<TSyntheticAttributes>(FullFioInPublicNameStartTime_, std::move(proxyAttributes));

        PublicProfileProtectionStartTime_ = config.AsInt(xpathPrefix + "/public_profile_protection_start_time", 1600000000);

        TLog::Info("BlackBox: need user confirmation for public data access for users registered since %ld", PublicProfileProtectionStartTime_);

        MailHostId_ = config.AsString(xpathPrefix + "/mail_host_id");
        TLog::Info("Blackbox: using mail host id: '%s'", MailHostId_.c_str());

        BlockCalendarByPassword_ = config.AsBool(xpathPrefix + "/block_calendar_by_password", false);

        TLog::Info("BlackBox: block calendar with portal password %s", BlockCalendarByPassword_ ? "ENABLED" : "DISABLED");

        TString hashSecretPath = config.AsString(xpathPrefix + "/password_hash_secret_path");
        PasswordHashSecret_ = NUtils::ReadFile(hashSecretPath);

        LrcsTtl_ = config.AsInt(xpathPrefix + "/reg_completion_recommended/lrcs_ttl", 30 * 24 * 60 * 60);
    }

    void TBlackboxImpl::InitLRandomsDumper(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/lrandoms_txt")) {
            TLog::Warning() << "BlackBox: dumping of lrandoms.txt was disabled";
            return;
        }

        TString filename = config.AsString(xpathPrefix + "/lrandoms_txt/file");
        TDuration writePeriod = TDuration::Seconds(
            config.AsInt(xpathPrefix + "/lrandoms_txt/write_period__sec"));

        TLog::Info() << "BlackBox: lrandoms would be dumped every " << writePeriod
                     << " to file '" << filename << "'";

        LrandomsDumper_ = NUtils::TRegularTaskDecorator<TLRandomsDumper>::CreateUnique(
            {
                .Period = writePeriod,
                .Name = "bb_lrandoms_txt",
            },
            filename,
            SessionSigner());
    }

    void TBlackboxImpl::InitSigner(const NXml::TConfig& config, const TString& xpathPrefix) {
        TString spacesStr = config.AsString(xpathPrefix + "/signer/long_sign_spaces", TStrings::EMPTY);
        std::vector<TString> spacesVec = NUtils::ToVector(spacesStr, ",;");

        std::unordered_set<TString> signSpaces;
        signSpaces.insert(spacesVec.begin(), spacesVec.end());

        Signer_ = std::make_unique<TSigner>(*SessSigner_, std::move(signSpaces));

        if (spacesStr) {
            TLog::Info() << "BlackBox: long sign spaces: " << spacesStr;
        }
    }

    void TBlackboxImpl::InitUserTicketCache(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!config.Contains(xpathPrefix + "/ticket_signer/user_ticket_cache")) {
            TLog::Info() << "User ticket cache is OFF";
            return;
        }

        double limit = 0;
        const TString limitKey = xpathPrefix + "/ticket_signer/user_ticket_cache/total_memory_limit_gb";
        Y_ENSURE(TryFromString(config.AsString(limitKey), limit),
                 "Failed to get cache memory limit: it is not int or double: " << limitKey);
        TLog::Info() << "User ticket cache is ON. Total memory limit: " << limit << " Gb";

        limit *= 1024 * 1024 * 1024;
        UserTicketCacheCtx_ = std::make_shared<NCache::TContext<NTicketSigner::TUserSigner, TString>>(limit, "userticket");

        int timeBucketCount = config.AsInt(xpathPrefix + "/ticket_signer/user_ticket_cache/time_bucket_count", 3);
        int keyBucketCount = config.AsInt(xpathPrefix + "/ticket_signer/user_ticket_cache/key_bucket_count", 128);
        UserTicketCacheTtl_ = config.AsNum<ui32>(xpathPrefix + "/ticket_signer/user_ticket_cache/ttl_sec", 300);

        if (config.AsBool(xpathPrefix + "/ticket_signer/user_ticket_cache/oauth/enabled")) {
            const bool dryRun = config.AsBool(xpathPrefix + "/ticket_signer/user_ticket_cache/oauth/dry_run", true);
            UserTicketCacheForOAuth_ = std::make_unique<TUserTicketCache>(
                UserTicketCacheCtx_->CreateCache(timeBucketCount, keyBucketCount, "userticket_oauth"),
                TDuration::Seconds(UserTicketCacheTtl_),
                dryRun);
            TLog::Info() << "User ticket cache for oauth is ON. Dry-run is " << (dryRun ? "ON" : "OFF");
        } else {
            TLog::Info() << "User ticket cache for oauth is OFF";
        }

        if (config.AsBool(xpathPrefix + "/ticket_signer/user_ticket_cache/sessionid/enabled")) {
            const bool dryRun = config.AsBool(xpathPrefix + "/ticket_signer/user_ticket_cache/sessionid/dry_run", true);
            UserTicketCacheForSessionid_ = std::make_unique<TUserTicketCache>(
                UserTicketCacheCtx_->CreateCache(timeBucketCount, keyBucketCount, "userticket_sessionid"),
                TDuration::Seconds(UserTicketCacheTtl_),
                dryRun);
            TLog::Info() << "User ticket cache for sessionid is ON. Dry-run is " << (dryRun ? "ON" : "OFF");
        } else {
            TLog::Info() << "User ticket cache for sessionid is OFF";
        }

        if (config.AsBool(xpathPrefix + "/ticket_signer/user_ticket_cache/user_ticket/enabled", true)) {
            const bool dryRun = config.AsBool(xpathPrefix + "/ticket_signer/user_ticket_cache/user_ticket/dry_run", false);
            UserTicketCacheForUserTicket_ = std::make_unique<TUserTicketCache>(
                UserTicketCacheCtx_->CreateCache(timeBucketCount, keyBucketCount, "userticket_userticket"),
                TDuration::Seconds(UserTicketCacheTtl_),
                dryRun);
            TLog::Info() << "User ticket cache for user_ticket is ON. Dry-run is " << (dryRun ? "ON" : "OFF");
        } else {
            TLog::Info() << "User ticket cache for user_ticket is OFF";
        }

        TDuration cleaningPeriod = TDuration::Seconds(config.AsInt(
            xpathPrefix + "/ticket_signer/user_ticket_cache/cleaning_period",
            UserTicketCacheTtl_ / (timeBucketCount + 1)));
        UserTicketCacheCtx_->StartCleaning(cleaningPeriod);
    }

    void TBlackboxImpl::InitRateLimiter(const NXml::TConfig& config, const TString& xpathPrefix) {
        if (!Badauth() || !Badauth()->GetKolmogor()) {
            RateLimiter_ = std::make_unique<TNoopRateLimiter>();
            TLog::Warning() << "RateLimiter is disabled because of Kolmogor absence";
            return;
        }

        TRateLimiterSettings settings{
            .Period = config.AsNum<ui64>(xpathPrefix + "/rate_limiting/period"),
            .Keyspace = config.AsString(xpathPrefix + "/rate_limiting/kolmogor_space"),
        };
        Y_ENSURE(settings.Period > 0, "'/rate_limiting/period' must be > 0");

        settings.Limits[TRateLimiterSettings::GetDebugUserTicket] =
            config.AsNum<ui64>(xpathPrefix + "/rate_limiting/limits/get_debug_user_ticket");

        RateLimiter_ = std::make_unique<TKolmogorBasedRateLimiter>(
            *Badauth()->GetKolmogor(),
            settings);
        TLog::Info() << "RateLimiter is enabled for 'get_debug_user_ticket'";
    }

    void TBlackboxImpl::SetHostId(const TString& id) {
        ui32 tmp;
        Y_ENSURE(TryIntFromString<16>(TStringBuf(id), tmp), "invalid hostid: '" << id << "'");
        HostId_ = id;
    }

    void TBlackboxImpl::CheckGrants(const NCommon::TRequest& req,
                                    const TConsumer& consumer,
                                    TBlackboxMethods::EMethod method) {
        if (!consumer.IsAllowed(method)) {
            const TString& ip = req.GetRemoteAddr();
            throw TBlackboxError(TBlackboxError::EType::AccessDenied)
                << AccessDeniedMessage(consumer, ip, TBlackboxMethods::Name(method));
        }
    }

    TConsumerInfo
    TBlackboxImpl::GetConsumer(NCommon::TRequest& req) const {
        // Determine consumer IP address and look it up in the ACL. Don't proceed
        // if this consumer is not found in the ACL
        NUtils::TIpAddr addr;

        const TString& ip = req.GetRemoteAddr();
        if (!addr.Parse(ip)) {
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "Internal error: invalid peer IP: " << InvalidValue(ip);
        }

        const TString& serviceTicket = req.GetHeader(TStrings::X_YA_SERVICE_TICKET);
        if (!serviceTicket.empty()) {
            const NTvmAuth::TCheckedServiceTicket ticket = TvmClient().CheckServiceTicket(serviceTicket);
            if (!ticket) {
                TString err = NUtils::CreateStr(
                    "failed to check service ticket: ",
                    NTvmAuth::StatusToString(ticket.GetStatus()),
                    (ticket.GetStatus() == NTvmAuth::ETicketStatus::InvalidDst
                         ? NUtils::CreateStr(". Expected dst=", TvmSelfId_)
                         : TString()),
                    ". ", ticket.DebugInfo(),
                    ": ", TString(InvalidValue(NTvmAuth::NUtils::RemoveTicketSignature(serviceTicket))));

                return TConsumerInfo({},
                                     std::move(err),
                                     TConsumerInfo::EAuthType::Tvm2,
                                     TConsumerInfo::EServiceTicketStatus::Bad);
            }

            TTvmGrants::TFindResult result = TvmGrants_->FindConsumer(ticket.GetSrc(), addr);
            if (result.Consumer) {
                req.SetConsumerFormattedName("3:" + result.Consumer->GetName());
            } else {
                req.SetConsumerFormattedName(NUtils::CreateStr("3:_unknown_tvmid=", ticket.GetSrc()));
            }
            return TConsumerInfo(std::move(result.Consumer),
                                 std::move(result.Err),
                                 TConsumerInfo::EAuthType::Tvm2,
                                 TConsumerInfo::EServiceTicketStatus::Unspecified,
                                 ticket.GetSrc());
        }

        TIpAclMap<TConsumer>::EStorage storage;
        std::shared_ptr<TConsumer> consumer = IpGrants_->Find(addr, &storage);
        if (!consumer) {
            return TConsumerInfo(
                {},
                NUtils::CreateStr("request without header '", TStrings::X_YA_SERVICE_TICKET, "': no grants for IP=",
                                  ip,
                                  " without TVM"),
                TConsumerInfo::EAuthType::Ip);
        }

        TString name;
        name.reserve(4 + consumer->GetName().size());
        name.append("1:").append(consumer->GetName()).push_back(':');
        switch (storage) {
            case TIpAclMap<TConsumer>::EStorage::Range:
                name.push_back('1');
                break;
            case TIpAclMap<TConsumer>::EStorage::Trypo:
                name.push_back('2');
                break;
            case TIpAclMap<TConsumer>::EStorage::Single:
                name.push_back('3');
                break;
        }
        req.SetConsumerFormattedName(std::move(name));

        return TConsumerInfo(consumer, {}, TConsumerInfo::EAuthType::Ip);
    }

    ENetworkKind TBlackboxImpl::CheckYandexIp(const TString& ip,
                                              const NCommon::TRequest& req,
                                              const TStringBuf description) const {
        // Determine peer IP address and look it up in the allowed hosts list.
        NUtils::TIpAddr addr;

        if (!addr.Parse(ip)) {
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "Internal error: invalid peer IP";
        }

        return UsernetsList_->FindNetwork(addr, req, description);
    }

    TString TBlackboxImpl::AccessDeniedMessage(const TConsumer& consumer, const TString& ip, const TString& method) {
        return NUtils::CreateStr("no grants for ", method, ". consumer: ", consumer.PrintId(ip));
    }

    time_t TBlackboxImpl::GetIpGrantsMTime() const {
        return IpGrants_ ? IpGrants_->GetMTime() : 0;
    }

    time_t TBlackboxImpl::GetTvmGrantsMTime() const {
        return TvmGrants_ ? TvmGrants_->GetMTime() : 0;
    }

    bool TBlackboxImpl::DbOk(TString& msgaccum) const {
        TString msg;

        // Check centraldb: critical, affects user aliases fetching
        bool centraldbOk = CentralDb_->IsOk(&msg);
        NUtils::AddMessage(msgaccum, msg);

        // Check shardeddb: critical, affects user attributes fetching
        bool shardeddbOk = ShardsMap_->ShardsOk(msg);
        NUtils::AddMessage(msgaccum, msg);

        // Check oauth: critical
        bool oauthdbOk = OauthConfig_->DbOk(msg);
        NUtils::AddMessage(msgaccum, msg);

        bool perimeterOk = true;
        if (Perimeter_) {
            perimeterOk = Perimeter_->DbOk(msg);
            NUtils::AddMessage(msgaccum, msg);
        }

        bool centralSlowDbOk = CentralSlowDb_->IsOk(&msg);
        NUtils::AddMessage(msgaccum, msg);

        // return true if all DB are OK
        // for optional DBs: ok if not critical or critical and really ok
        return centraldbOk && shardeddbOk && oauthdbOk && perimeterOk && centralSlowDbOk;
    }

    TString TBlackboxImpl::DbpoolStats() const {
        return DbCtx_->GetLegacyStats();
    }

    TString TBlackboxImpl::DbpoolExtendedStats() const {
        return DbCtx_->GetExtendedStats();
    }

    NJuggler::TStatus TBlackboxImpl::GetJugglerStatus() const {
        NJuggler::TStatus status;
        status.Update(TvmClient().GetStatus());
        status.Update(TvmPrivateKeys().GetJugglerStatus());
        status.Update(SessionSigner().GetJugglerStatus());
        status.Update(OauthParser().GetJugglerStatus());
        if (StaffInfo_) {
            status.Update(StaffInfo_->GetJugglerStatus());
        }

        return status;
    }

    std::unique_ptr<TValidator>
    TBlackboxImpl::CreateValidator(const NCommon::TRequest& request, TDbFieldsConverter& conv) const {
        if (!request.HasArg(TStrings::EMAILS)) {
            return std::unique_ptr<TValidator>();
        }

        std::unique_ptr<TValidator> validator = std::make_unique<TValidator>(
            conv, YandexDomains_);

        const TString& email = request.GetArg(TStrings::EMAILS);
        if (email == TStrings::TEST_ONE) {
            validator->SetAddrToTest(TUtils::GetCheckedArg(request, TStrings::TESTED_EMAIL));
        } else if (email == TStrings::JUST_DEFAULT) {
            validator->SetJustDefaultScope();
        } else if (email == TStrings::JUST_YANDEX) {
            validator->SetYandexScope();
        } else if (email != TStrings::GET_ALL) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid " << TStrings::EMAILS << " parameter value: " << InvalidValue(email);
        }

        if (Anonymiser_) {
            validator->SetAnonymiser(Anonymiser_.get());
        }

        return validator;
    }

    TDbFetcher TBlackboxImpl::CreateDbFetcher() const {
        return TDbFetcher(
            CentralDb(),
            ShardsMap(),
            &YandexDomains(),
            HostedDomains(),
            TotpEncryptor(),
            *SyntheticAttrs_,
            Anonymiser());
    }

    void TBlackboxImpl::AddUnistat(NUnistat::TBuilder& builder) {
        DbCtx_->AddUnistat(builder);
        builder.AddRow("grants.ip.parsing_errors_dmmm", IpGrants_->GetParsingErrors());
        builder.AddRow("grants.tvm.parsing_errors_dmmm", TvmGrants_->GetParsingErrors());
        if (StaffInfo_) {
            StaffInfo_->AddUnistat(builder);
        }
        if (BadauthFacade_) {
            BadauthFacade_->AddUnistat(builder);
        }
        if (UserTicketCacheCtx_) {
            UserTicketCacheCtx_->AddUnistat(builder);
        }
        if (UserTicketCacheForOAuth_) {
            UserTicketCacheForOAuth_->Cache->AddUnistat(builder);
        }
        if (UserTicketCacheForSessionid_) {
            UserTicketCacheForSessionid_->Cache->AddUnistat(builder);
        }
    }

    void TBlackboxImpl::AddExtendedUnistatForDbPool(NUnistat::TBuilder& builder) {
        DbCtx_->AddUnistatExtended(builder);
    }

    // check if login looks like phone alias and normalize it
    bool TBlackboxImpl::IsPhoneAlias(TString& login,
                                     const TString& domain,
                                     const TString& country,
                                     bool allowUTF) const {
        if (TUtils::HasLetter(login, allowUTF)) { // has letters - does not look like phone alias
            return false;
        }

        TString normLogin(login);
        TUtils::DotsToHyphens(normLogin, false);
        if (DigitLogins_->Contains(normLogin)) { // lecacy digit login - not a phone alias
            return false;
        }

        login = TUtils::NormalizePhone(login, domain, YandexDomains_, country);
        return true;
    }

    //
    // REFACTORING: need redesign using table-driven approach
    //
    ELoginType TBlackboxImpl::TranslateLoginSids(TString& login, TString& sid, TDomain& domain, const TString& country) const {
        if (sid == TStrings::FEDERAL_SID) {
            // no login lowercase and no other logic for sid=federal
            return ELoginType::Yandex;
        }

        login = TUtils::TolowerUtf(login);
        if (sid == TStrings::UBER_SID || sid == TStrings::MAILISH_SID || sid == TStrings::BANK_SID) {
            // no other logic (i.e. checking for @ or phone number) for sids: uber, mailish, bank
            return ELoginType::Yandex;
        }

        // Does this login contains domain part?
        TString::size_type atPos = login.find('@');
        bool allowPhoneAliases = (sid == RESTORE_MACRO) ? false : EnablePhoneAliases_;

        if (atPos == TString::npos) {
            if (login.StartsWith(TStrings::SOCIAL_LOGIN_PREFIX)) { // social user
                sid = TStrings::SID_58;
                return ELoginType::Yandex;
            }

            if (login.StartsWith(TStrings::PHONISH_LOGIN_PREFIX)) { // phne- user
                sid = TStrings::SID_68;
                return ELoginType::Yandex;
            }

            if (login.StartsWith(TStrings::NEOPHONISH_LOGIN_PREFIX)) { // nphne- user
                sid = TStrings::NEOPHONISH_SID;
                return ELoginType::Yandex;
            }

            if (allowPhoneAliases && IsPhoneAlias(login, TStrings::EMPTY, country, true)) { // phone alias
                sid = TStrings::SID_65;
                return ELoginType::Yandex;
            }
        }

        // If sids is one of the mail macros, we may need to strip off domain
        // part of login and we may also find out that we shall head for
        // ma-4-domains. Otherwise, this is a "normal" request and we're done.
        // 'smtp' and 'pop' need to have almost the same behavour:
        // find pdd by aliases for account and aliases for domain.
        // 'smtp' works with defaultUid for pdd, 'pop' - does not
        bool nosids = false;
        bool smtp = false;
        bool passport = false;
        bool mk = false;
        if (sid == MAIL_MACRO) {
            nosids = true;
        } else if (sid == SMTP_MACRO) {
            smtp = true;
        } else if (sid == PASSPORT_MACRO || sid == RESTORE_MACRO || sid.empty()) {
            passport = true;
        } else if (sid == MK_MACRO) {
            mk = true;
        } else if (sid != POP_MACRO) {
            return ELoginType::Yandex;
        }
        // Here we have only these cases for sid: mail, smtp, passport, restore, <empty>, mk, pop

        // No domain part:
        //
        // Assume lookup based on the 'accounts' table, need substitute '-' in dbfields with '2'
        // (we enforce this by returning appropriate code).
        if (atPos == TString::npos) {
            sid.clear();
            return ELoginType::Yandex_Force_Sid2;
        }
        // Nothing after the '@' sign - shall not happen! Cannot help this,
        // assume 'accounts' and no sids
        if (atPos == login.size() - 1) {
            sid.clear();
            return ELoginType::Yandex;
        }

        // Yes, domain part found; now make a decision based on what it is: narod.ru,
        // ya.ru, yandex.* or otherwise; in the latter case we're dealing with a
        // mail-for-domains user or a lite-user (either full-registered or otheriwse)
        TString domainStr(login, atPos + 1, login.size() - atPos - 1);
        NUtils::Tolower(domainStr);

        const TYandexDomains::TDomainData* nativeDomain;
        if ((nativeDomain = YandexDomains().Find(domainStr)) != nullptr) {
            ELoginType ret = ELoginType::Yandex;

            if (domainStr == NAROD_DOMAIN) {
                sid = TStrings::MAIL_NAROD_SID;
                ret = ELoginType::Narod;
            } else if (!nativeDomain->GetDomid().empty()) {
                // check if alternate domain login is actually a phone alias
                TString loginPart(login);
                loginPart.erase(atPos);
                if (allowPhoneAliases && IsPhoneAlias(loginPart, domainStr, country)) {
                    login.assign(loginPart); // store normalized phone alias
                    sid = TStrings::SID_65;
                    return ELoginType::Yandex;
                }
                if (nosids) {
                    sid.clear();
                    return ELoginType::Yandex_Force_Sid2;
                }

                sid = passport ? TStrings::ALTDOMAIN_SID : TStrings::MAIL_SID;
                domain.SetId(nativeDomain->GetDomid());
                // Be careful: here we don't cut off @domain part if sid is found
                // To be able to find user@galatasaray.net aliases later on in db results
                // But since it is a email it will not be normalized later, we need to normalize it here
                TUtils::DotsToHyphens(login, true);
                return ELoginType::AltDomain;
            } else {
                sid = TStrings::MAIL_SID;
            }

            // It is a regular Yandex user, so strip off domain part of login and clear the domain itself
            login.erase(atPos);

            if (allowPhoneAliases && IsPhoneAlias(login, domainStr, country)) {
                sid = TStrings::SID_65;
                return ELoginType::Yandex;
            }
            if (nosids) {
                sid.clear();
                return ELoginType::Yandex_Force_Sid2;
            }

            return ret;
        }

        // If request came from MoiKrug give preference to MK-users
        if (mk) {
            sid = TStrings::MK_SID;
            return ELoginType::Yandex;
        }

        // At this point we have one of the three possibilities:
        //  - this is a PDD domain
        //  - this is not a PDD domain but request originates at Passport so give MoiKrug a chance
        //  - this is an alien domain which we refuse to process
        //
        //  Note that we may actually find an alias for a PDD domain. For smtp
        //  requests we substitute master domain for the alias, but in all
        //  other cases we ignore alias completely. In case it is an alias
        //  replace domain part of the email.
        // We must search in aliases for smtp, passport or pop. It's equivalent to (!nosids && !mk)
        TDomainFetcher::TResult hostedFound = HostedDomains().Find(
            domainStr,
            nosids || mk
                ? TDomainFetcher::EPolicy::MasterOnly
                : TDomainFetcher::EPolicy::ReplaceAliasWithMaster);
        if (hostedFound) {
            domain = *hostedFound.Domain;

            login.erase(atPos + 1).append(domain.UtfName());
            // It is a mail-for-domains user
            if (nosids) {
                sid.clear();
                domain.ClearDefaultUid();
            } else {
                // Here possible: smtp, passport, pop
                sid = SMTP_HOSTED_SIDS;
                if (!smtp) {
                    domain.ClearDefaultUid();
                }
            }
            return ELoginType::Hosted; // domain found, PDD user
        }

        if (passport) {
            sid = TStrings::MK_SID;
            return ELoginType::Yandex; // domain unknown, maybe MK
        }

        sid.clear();
        return ELoginType::Alien; // domain unknown, alien
    }

    bool TBlackboxImpl::IsYaTeam() const {
        return bool(Perimeter_);
    }

    bool TBlackboxImpl::IsStopWord(const TString& word) const {
        if (StopWords_) {
            return StopWords_->IsStopWord(word) || StopWords_->IsStopSubstr(word);
        }

        return false;
    }

    bool TBlackboxImpl::IsMail4domainsUid(const TString& uid) const {
        unsigned long long uidNum = TUtils::ToUInt(uid, TStrings::UID);
        return (uidNum >= Mail4domainsFirstUid_ &&
                uidNum <= Mail4domainsLastUid_);
    }

    bool TBlackboxImpl::IsMdaDomain(const TString& domain) const {
        return MdaDomains_ && MdaDomains_->Contains(domain);
    }

    const TSessGuardOptions& TBlackboxImpl::GetGuardSpaceOptions(const TString& guardspace) const {
        auto it = GuardSpaceOptions_.find(guardspace);
        return it == GuardSpaceOptions_.end()
                   ? DefaultGuardSpaceOptions_
                   : it->second;
    }

    bool TBlackboxImpl::IsAllowedAuthtype(const TString& type) const {
        return AllowedAuthtypes_.contains(type);
    }

    bool TBlackboxImpl::IsExternalRobotAllowed(ui64 uid) const {
        return AllowedExternalRobotUids_.contains(uid);
    }

    const NTvmAuth::TTvmClient& TBlackboxImpl::TvmClient() const {
        if (!TvmClient_) {
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "TVM client was not initialized";
        }
        return *TvmClient_;
    }

    const IStaffInfo* TBlackboxImpl::StaffInfo() const {
        return StaffInfo_.get();
    }

    void TBlackboxImpl::InitLoggers(const NXml::TConfig& config, const TString& xpathPrefix) {
        LoginLogger_ = config.CreateLogger(xpathPrefix + "/logger_login");

        AuthLogger_ = std::make_unique<TAuthLog>(
            config.CreateLogger(xpathPrefix + "/logger_auth"), HostId_);

        StatboxLogger_ = std::make_unique<TTskvLog>(
            config.CreateLogger(xpathPrefix + "/logger_statbox"));
        OauthLogger_ = std::make_unique<TTskvLog>(
            config.CreateLogger(xpathPrefix + "/logger_oauth"));

        DbCtx_->InitLogger(config.AsString(xpathPrefix + "/log_dbpool"));
    }
}
