#include "domain.h"

#include <passport/infra/daemons/blackbox/src/protobuf/domain_lists.pb.h>

#include <passport/infra/libs/cpp/idn/idn.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    TDomain::TDomain(TDomainParams&& params)
        : Id_(std::move(params.Id))
        , Master_(std::move(params.Master))
        , UtfName_(NIdn::PunycodeToUtf(params.Name))
        , AsciiName_(std::move(params.Name))
        , Ena_(params.Ena)
        , Mx_(params.Mx)
        , AdminUid_(std::move(params.AdminUid))
        , DefaultUid_(std::move(params.DefaultUid))
        , BornDate_(std::move(params.BornDate))
        , Options_(std::move(params.Options))
    {
        // in case of broken punycode it's better to use ascii name than empty string
        if (UtfName_.empty()) {
            UtfName_ = AsciiName_;
            TLog::Warning() << "TDomain: broken punycode domain found. Domain '" << AsciiName_
                            << "' id=" << Id_ << " does not convert to utf.";
        }
    }

    TDomain::TDomain(const domainlists_proto::Domain& d)
        : TDomain(TDomainParams{
              .Id = IntToString<10>(d.id()),
              .Master = IntToString<10>(d.masterid()),
              .Name = d.asciiname(),
              .Ena = d.ena(),
              .Mx = d.mx(),
              .AdminUid = IntToString<10>(d.adminuid()),
              .DefaultUid = IntToString<10>(d.defaultuid()),
              .BornDate = d.borndate(),
              .Options = TOptions{
                  .OrganizationName = d.organizationname(),
                  .OrganizationId = d.organizationid(),
                  .RawOptions = d.options(),
              },
          })
    {
        Slaves_.reserve(d.slaves().size());
        for (auto s : d.slaves()) {
            Slaves_.push_back(IntToString<10>(s));
        }
    }

    void TDomain::Serialize(domainlists_proto::Domain& out) const {
        out.set_id(IntFromString<ui64, 10>(Id_));
        out.set_masterid(IntFromString<ui64, 10>(Master_));
        out.set_asciiname(AsciiName_);
        out.set_ena(Ena_);
        out.set_mx(Mx_);
        out.set_adminuid(IntFromString<ui64, 10>(AdminUid_));
        out.set_defaultuid(IntFromString<ui64, 10>(DefaultUid_));
        out.set_borndate(BornDate_);
        // TODO: do not serialize `organizationname` and `organizationid`
        // They are in `options` already
        out.set_organizationname(Options_.OrganizationName);
        out.set_organizationid(Options_.OrganizationId);
        out.set_options(Options_.RawOptions);

        auto& sl = *out.mutable_slaves();
        sl.Reserve(Slaves_.size());
        for (const TString& s : Slaves_) {
            sl.Add(IntFromString<ui64, 10>(s));
        }
    }

    void TDomain::AddSlave(const TString& slave) {
        // Assume average slaves_.size() < 10 in production
        if (std::find(Slaves_.begin(), Slaves_.end(), slave) == Slaves_.end()) {
            Slaves_.push_back(slave);
        }
    }

    void TDomain::DropSlave(const TString& slave) {
        auto it = std::find(Slaves_.begin(), Slaves_.end(), slave);
        if (it != Slaves_.end()) {
            Slaves_.erase(it);
        }
    }

    void TDomain::UpdateDomain(TDomain&& d) {
        Master_ = std::move(d.Master_);
        AsciiName_ = std::move(d.AsciiName_);
        UtfName_ = std::move(d.UtfName_);
        Ena_ = d.Ena_;
        Mx_ = d.Mx_;
        AdminUid_ = std::move(d.AdminUid_);
        DefaultUid_ = std::move(d.DefaultUid_);
        BornDate_ = std::move(d.BornDate_);
        Options_ = std::move(d.Options_);
    }

    bool TDomain::operator==(const TDomain& o) const {
        return Id_ == o.Id_ &&
               Master_ == o.Master_ &&
               UtfName_ == o.UtfName_ &&
               AsciiName_ == o.AsciiName_ &&
               Ena_ == o.Ena_ &&
               Mx_ == o.Mx_ &&
               AdminUid_ == o.AdminUid_ &&
               DefaultUid_ == o.DefaultUid_ &&
               BornDate_ == o.BornDate_ &&
               Options_.OrganizationName == o.Options_.OrganizationName &&
               Options_.OrganizationId == o.Options_.OrganizationId &&
               Options_.RawOptions == o.Options_.RawOptions &&
               Slaves_ == o.Slaves_;
    }
}
