#include "domain_builder.h"

#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/protobuf/domain_lists.pb.h>

#include <passport/infra/libs/cpp/dbpool/value.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    TDomainListBuilderBase::TDomainListBuilderBase(TDomainCache& cache)
        : Out_(cache)
    {
    }

    void TDomainListBuilderBase::Finish() {
        // Some straw for case when master_domain_id is bigger than domain_id of slave
        for (TDomain& d : Slaves_) {
            Out_.List.AddOrUpdateDomain(std::move(d));
        }
    }

    static const char* const ORGANIZATION_NAME_SHORT = "2";
    static const char* const ORGANIZATION_NAME_LONG = "organization_name";
    static const char* const GLOGOUT = "4";
    TDomain::TOptions TDomainListBuilderBase::ReadDomainOptions(const TString& options) {
        if (options.empty()) {
            return {};
        }

        rapidjson::Document doc;
        if (!NJson::TReader::DocumentAsObject(options, doc)) {
            return {};
        }

        TDomain::TOptions res;
        TString name;
        if (!NJson::TReader::MemberAsString(doc, ORGANIZATION_NAME_LONG, name)) {
            NJson::TReader::MemberAsString(doc, ORGANIZATION_NAME_SHORT, name);
        }

        if (name.StartsWith("org_id:")) {
            TStringBuf orgId = TStringBuf(name).Skip(7);
            if (NUtils::DigitsOnly(orgId)) {
                res.OrganizationId = orgId;
            } else {
                TLog::Warning("Bad organization id for domain: %s", name.c_str());
            }
        }
        res.OrganizationName = std::move(name);

        time_t glogout = 0;
        if (NJson::TReader::MemberAsInt64(doc, GLOGOUT, glogout)) {
            res.Glogout = TInstant::Seconds(glogout);
        }

        res.RawOptions = options;
        return res;
    }

    TString TDomainListBuilder::MakeQuery(const TString& domId,
                                          const size_t limit) {
        return NUtils::CreateStr(
            "SELECT domain_id,master_domain_id,name,enabled,mx,admin_uid,default_uid,ts,options "
            "FROM domains WHERE domain_id>",
            domId,
            " ORDER BY domain_id LIMIT ",
            limit);
    }

    void TDomainListBuilder::Reserve(size_t size) {
        Out_.List.Reserve(size);
    }

    TString TDomainListBuilder::AddRowAndGetDomId(NDbPool::TRow&& row) {
        const size_t DOMAIN_ID = 0;
        const size_t MASTER_DOMAIN_ID = 1;
        const size_t DOMAIN_NAME = 2;
        const size_t DOMAIN_ENA = 3;
        const size_t DOMAIN_MX = 4;
        const size_t DOMAIN_ADMIN_UID = 5;
        const size_t DOMAIN_DEFAULT_UID = 6;
        const size_t DOMAIN_BORN_DATE = 7;
        const size_t DOMAIN_OPTIONS = 8;

        if (row.size() != 9) {
            throw yexception() << "DomainListBuilder: response from db has invalid number of fields: "
                               << row.size();
        }

        TDomain::TOptions options = ReadDomainOptions(row[DOMAIN_OPTIONS].AsString());

        TDomain d({
            .Id = row[DOMAIN_ID].AsString(),
            .Master = row[MASTER_DOMAIN_ID].AsString(),
            .Name = row[DOMAIN_NAME].AsString(),
            .Ena = TDbValue::AsBoolean(row[DOMAIN_ENA].AsString()),
            .Mx = TDbValue::AsBoolean(row[DOMAIN_MX].AsString()),
            .AdminUid = row[DOMAIN_ADMIN_UID].AsString(),
            .DefaultUid = row[DOMAIN_DEFAULT_UID].AsString(),
            .BornDate = row[DOMAIN_BORN_DATE].AsString(),
            .Options = std::move(options),
        });

        if (d.IsSlave()) {
            Slaves_.push_back(std::move(d));
        } else {
            Out_.List.AddOrUpdateDomain(std::move(d));
        }

        return row[DOMAIN_ID].AsString();
    }

    void TDomainListBuilder::SetLastEventId(const TString& id) {
        Out_.LastEventId = id;
    }

    void TDomainListUpdater::SetLogging(bool val) {
        Out_.List.SetLogger(val);
    }

    TString TDomainListUpdater::MakeQuery(const TString& eventId) {
        return NUtils::CreateStr(
            "SELECT"
            " MAX(e.id),"
            " e.domain_id,"
            " d.master_domain_id,"
            " d.name,"
            " d.enabled,"
            " d.mx,"
            " d.admin_uid,"
            " d.default_uid,"
            " d.ts,"
            " d.options "
            "FROM domains_events e LEFT JOIN domains d USING (domain_id) "
            "WHERE e.id>",
            eventId,
            " GROUP BY domain_id");
    }

    // We don't use domain event type anymore
    // but here are the list of currently used types, for reference
    // 2: 'add'
    // 3: 'delete'
    // 4: 'mx'
    // 5: 'ena'
    // 6: 'default_uid'
    // 7: 'options'
    // 8: 'swap'
    // 9: 'update'

    TDomainListUpdater::Event TDomainListUpdater::AddRow(const NDbPool::TRow& row) {
        const size_t EVENT_ID = 0;
        const size_t DOMAIN_ID = 1;
        const size_t MASTER_DOMAIN_ID = 2;
        const size_t DOMAIN_NAME = 3;
        const size_t DOMAIN_ENA = 4;
        const size_t DOMAIN_MX = 5;
        const size_t DOMAIN_ADMIN_UID = 6;
        const size_t DOMAIN_DEFAULT_UID = 7;
        const size_t DOMAIN_BORN_DATE = 8;
        const size_t DOMAIN_OPTIONS = 9;

        if (row.size() != 10) {
            throw yexception() << "DomainListUpdater: response from db has invalid number of fields: "
                               << row.size();
        }

        const TString& id = row[DOMAIN_ID].AsString();
        if (id == "0") {
            return Event::Reload;
        }

        const TString& eventId = row[EVENT_ID].AsString();
        ui32 eventIdInt = IntFromString<ui32, 10>(eventId);
        if (LastEventIdInt_ < eventIdInt) {
            LastEventIdInt_ = eventIdInt;
            Out_.LastEventId = eventId;
        }

        if (row[DOMAIN_NAME].IsNull()) {
            TLog::Info("DomainListUpdater: delete domain: %s", id.c_str());
            Out_.List.DeleteDomain(id);
            return Event::GoNext;
        }

        TDomain::TOptions options = ReadDomainOptions(row[DOMAIN_OPTIONS].AsString());

        const TString& master = row[MASTER_DOMAIN_ID].AsString();
        TString name = row[DOMAIN_NAME].AsString();
        TLog::Info("DomainListUpdater: set domain: id='%s' master_id='%s' name='%s'",
                   id.c_str(),
                   master.c_str(),
                   name.c_str());

        TDomain d({
            .Id = id,
            .Master = master,
            .Name = name,
            .Ena = TDbValue::AsBoolean(row[DOMAIN_ENA].AsString()),
            .Mx = TDbValue::AsBoolean(row[DOMAIN_MX].AsString()),
            .AdminUid = row[DOMAIN_ADMIN_UID].AsString(),
            .DefaultUid = row[DOMAIN_DEFAULT_UID].AsString(),
            .BornDate = row[DOMAIN_BORN_DATE].AsString(),
            .Options = std::move(options),
        });

        if (d.IsSlave()) {
            Slaves_.push_back(std::move(d));
        } else {
            Out_.List.AddOrUpdateDomain(std::move(d));
        }

        return Event::GoNext;
    }
}
