#include "domain_fetcher.h"

#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/protobuf/domain_lists.pb.h>

#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/util.h>
#include <passport/infra/libs/cpp/dbpool/value.h>
#include <passport/infra/libs/cpp/idn/idn.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/stream/file.h>
#include <util/string/join.h>

namespace NPassport::NBb {
    static const float RESERVE_RATIO = 1.1;

    static TString&
    Cell2Str(const NDbPool::TRow& r, int i, TString& s) {
        if (!r[i].IsNull()) {
            s.assign(r[i].AsString());
        } else {
            s.clear();
        }
        return s;
    }

    //
    // DomainFetcher
    //
    TDomainFetcher::TDomainFetcher(
        NDbPool::TDbPool& dbp,
        const TDomainFetcherSettings& settings)
        : Dbp_(dbp)
        , Settings_(settings)
    {
        // First try load the list for the very first time.
        InitFromFile();

        if (Cache_.Get() && SecondaryCache_) {
            return;
        }

        InitFromDb();
    }

    TDomainFetcher::~TDomainFetcher() {
        try {
            SaveToFile(*Cache_.Get(), Settings_.CacheFile);
        } catch (const std::exception& e) {
            TLog::Error() << "PDD domains: saveToFile failed: " << e.what();
        } catch (...) {
            TLog::Error() << "PDD domains: saveToFile failed: unknown exception";
        }
    }

    const TDomain* TDomainFetcher::FindDomItem(const TDomain* domain, const TDomainList& list, EPolicy policy) {
        if (domain == nullptr) {
            return nullptr;
        }

        switch (policy) {
            case EPolicy::MasterOnly:
                return domain->IsSlave() ? nullptr : domain;
            case EPolicy::MasterOrAlias:
                return domain;
            case EPolicy::ReplaceAliasWithMaster: {
                if (!domain->IsSlave()) {
                    return domain;
                }
                const TDomain* master = list.FindById(domain->Master());
                if (master == nullptr) {
                    TLog::Debug("PDD domains: couldn't find master '%s' for PDD domain alias %s",
                                domain->Master().c_str(),
                                domain->AsciiName().c_str());
                }
                return master;
            }
        }
    }

    TDomainFetcher::TResult TDomainFetcher::Find(const TString& domain, EPolicy policy) const {
        TDomainCachePtr cache = Cache_.Get();
        const TDomainList& list = cache->List;

        TString punycoded = NIdn::UtfToPunycode(domain);
        if (punycoded.empty()) {
            return {};
        }

        const TDomain* d = FindDomItem(list.FindByName(punycoded), list, policy);
        return {std::move(cache), d};
    }

    TDomainFetcher::TResult TDomainFetcher::FindById(const TString& domid, EPolicy policy) const {
        TDomainCachePtr cache = Cache_.Get();
        const TDomainList& list = cache->List;

        const TDomain* d = FindDomItem(list.FindById(domid), list, policy);
        return {std::move(cache), d};
    }

    static const TString DOMAIN_QUERY(
        "SELECT m.name,m.domain_id,m.default_uid,m.enabled,m.mx,s.name,m.options,s.domain_id,m.admin_uid,m.ts FROM domains d"
        " LEFT JOIN domains m ON m.domain_id=IF(d.master_domain_id,d.master_domain_id,d.domain_id)"
        " LEFT JOIN domains s ON s.master_domain_id=m.domain_id"
        " WHERE d.");

    TDomainFetcher::TResult TDomainFetcher::FindInDb(const TString& domain, bool useDomid, EPolicy policy) const {
        TLog::Warning("BlackBox performance warning: PDD domains: looking for %s '%s' in db",
                      useDomid ? "domain_id" : "domain",
                      domain.c_str());
        NDbPool::TTable result;
        try {
            NDbPool::TBlockingHandle sqlh(Dbp_);

            TString query = DOMAIN_QUERY;
            if (useDomid) {
                query.append("domain_id=").append(domain);
            } else {
                query.append("name='")
                    .append(NUtils::TolowerCopy(sqlh.EscapeQueryParam(domain)))
                    .push_back('\'');
            }

            // TLog::Error("QUERY: '%s'", query.c_str());

            result = sqlh.Query(query)->ExctractTable();
        } catch (const NDbPool::TException& e) {
            TLog::Debug("PDD domains: failed to read domain '%s' from db: %s", domain.c_str(), e.what());
            return {};
        }

        if (result.empty() || result[0][0].IsNull()) {
            return {};
        }

        /* Query result looks the same for slave and master domain from one group: for domaind=2 or 4640 or 4642 or 4643
         * +-----------------+-----------+-------------+---------+------+------------------------+---------+-----------+-----------+----------------------+
         * | name            | domain_id | default_uid | enabled | mx   | name                   | options | domain_id | admin_uid | born_date            |
         * +-----------------+-----------+-------------+---------+------+------------------------+---------+-----------+-----------+----------------------+
         * | imap1.yandex.ru |         2 |           0 |       1 |    0 | yaaa.ru                |         |      4640 |     70500 |  2018-06-21 10:01:03 |
         * | imap1.yandex.ru |         2 |           0 |       1 |    0 | zinbistroamericana.com |         |      4642 |     70500 |  2018-06-21 10:01:03 |
         * | imap1.yandex.ru |         2 |           0 |       1 |    0 | potapov.com            |         |      4643 |     70500 |  2018-06-21 10:01:03 |
         * +-----------------+-----------+-------------+---------+------+------------------------+---------+-----------+-----------+----------------------+
         */

        const int MASTER_NAME = 0;
        const int MASTER_DOMID = 1;
        const int DEFAULT_UID = 2;
        const int ENABLED = 3;
        const int MX = 4;
        const int SLAVE_NAME = 5;
        const int MASTER_OPTIONS = 6;
        const int SLAVE_ID = 7;
        const int ADMIN_UID = 8;
        const int BORN_DATE = 9;

        const NDbPool::TRow& firstRow = result[0];
        // Cheap deny
        bool isAlias = useDomid ? (firstRow[MASTER_DOMID].AsString() != domain)
                                : (firstRow[MASTER_NAME].AsString() != domain);
        if (policy == EPolicy::MasterOnly && isAlias) {
            return {};
        }

        TString ena = firstRow[ENABLED].AsString();
        bool isEna = !ena.empty() && ena[0] != '0';
        TString mx = firstRow[MX].AsString();
        bool isMx = !mx.empty() && mx[0] != '0';

        // Read only master 'options'. Slave 'options' may have only useless for us values
        TDomain::TOptions options =
            TDomainListBuilderBase::ReadDomainOptions(firstRow[MASTER_OPTIONS].AsString());

        TDomainList list;
        // master
        list.AddOrUpdateDomain(TDomain({
            .Id = firstRow[MASTER_DOMID].AsString(),
            .Master = "0",
            .Name = firstRow[MASTER_NAME].AsString(),
            .Ena = isEna,
            .Mx = isMx,
            .AdminUid = firstRow[ADMIN_UID].AsString(),
            .DefaultUid = firstRow[DEFAULT_UID].AsString(),
            .BornDate = firstRow[BORN_DATE].AsString(),
            .Options = std::move(options),
        }));

        const TString masterId = firstRow[MASTER_DOMID].AsString();

        // slaves
        for (const NDbPool::TRow& row : result) {
            if (row[SLAVE_NAME].IsNull()) {
                break;
            }
            list.AddOrUpdateDomain(TDomain({
                .Id = row[SLAVE_ID].AsString(),
                .Master = row[MASTER_DOMID].AsString(),
                .Name = row[MASTER_NAME].AsString(),
                .Ena = isEna,
                .Mx = isMx,
                .AdminUid = row[ADMIN_UID].AsString(),
                .DefaultUid = row[DEFAULT_UID].AsString(),
                .BornDate = row[BORN_DATE].AsString(),
                .Options = {},
            }));
        }

        TDomainCachePtr fakeCache = std::make_shared<TDomainCache>();
        fakeCache->List = std::move(list);

        const TDomain* item = policy == TDomainFetcher::EPolicy::ReplaceAliasWithMaster
                                  ? fakeCache->List.FindById(masterId)
                                  : (useDomid ? fakeCache->List.FindById(domain)
                                              : fakeCache->List.FindByName(domain));

        return {std::move(fakeCache), item};
    }

    void TDomainFetcher::InitFromFile() {
        if (Settings_.CacheFile.empty()) {
            return;
        }

        try {
            TLog::Info() << "PDD domains: loading hosted domains list from file: " << Settings_.CacheFile;

            SaveLoadedCache(LoadFromFile(Settings_.CacheFile));
            TLog::Info() << "PDD domains: loaded hosted domains list: "
                         << Cache_.Get()->List.size() << " entries. Starting update with events from DB";

            Update();
            TLog::Info() << "PDD domains: cache was updated with events from DB";
        } catch (const std::exception& e) {
            TLog::Warning() << "PDD domains: Failed to load hosted domains list at startup from file '"
                            << Settings_.CacheFile << "': " << e.what();
        }
    }

    void TDomainFetcher::InitFromDb() {
        try {
            TLog::Info("PDD domains: loading hosted domains list from DB...");

            SaveLoadedCache(Load());
            // to avoid second call of `load()` on start in `run()`
            NextCacheFileWrite_ = TInstant::Now();

            TLog::Info("PDD domains: loaded hosted domains list: %ld entries",
                       Cache_.Get()->List.size());
        } catch (const std::exception& e) {
            // We've already logged any details inside Load()
            TLog::Error("PDD domains: Failed to load hosted domains list at startup. <%s>", e.what());
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "Failed to load hosted domains list at startup";
        }
    }

    TDomainCachePtr TDomainFetcher::LoadFromFile(const TString& filename) {
        const TInstant start = TInstant::Now();
        const TString body = NUtils::ReadFile(filename);

        TLog::Debug() << "PDD domains: reading cache from disk took " << (TInstant::Now() - start)
                      << ". File size: " << body.size()
                      << ". md5:" << NUtils::Bin2hex(NUtils::TCrypto::Md5(body));

        TDomainCachePtr res = std::make_shared<TDomainCache>(LoadFromProto(body));

        TLog::Debug() << "PDD domains: loading from disk (full time) took " << (TInstant::Now() - start);

        return res;
    }

    TDomainCache TDomainFetcher::LoadFromProto(const TString& body) {
        const TInstant start = TInstant::Now();
        domainlists_proto::Cache protoCache;
        Y_ENSURE(protoCache.ParseFromString(body), "Failed to parse proto");

        TLog::Debug() << "PDD domains: parsing protobuf from disk took " << (TInstant::Now() - start);

        const TInstant startLoading = TInstant::Now();
        TDomainCache res{
            .List = TDomainList(protoCache),
            .LastEventId = IntToString<10>(protoCache.GetlastEventId()),
        };

        TLog::Debug() << "PDD domains: restoring domain list from cache took " << (TInstant::Now() - startLoading);

        return res;
    }

    void TDomainFetcher::SaveToFile(const TDomainCache& cache, const TString& filename) {
        if (filename.empty()) {
            TLog::Warning() << "PDD domains: skip writing to disk: filename was not configured";
            return;
        }

        TInstant start = TInstant::Now();
        const TString body = SerializeToProto(cache);

        TLog::Debug() << "PDD domains: protobuf serialization took " << (TInstant::Now() - start)
                      << ". File size: " << body.size()
                      << ". md5:" << NUtils::Bin2hex(NUtils::TCrypto::Md5(body));

        start = TInstant::Now();
        {
            TUnbufferedFileOutput file(filename);
            file.Write(body);
        }

        TLog::Debug() << "PDD domains: wrote cache to disk: " << filename
                      << ". Took " << (TInstant::Now() - start);
    }

    TString TDomainFetcher::SerializeToProto(const TDomainCache& cache) {
        google::protobuf::Arena arena;
        domainlists_proto::Cache& protoCache =
            *google::protobuf::Arena::CreateMessage<domainlists_proto::Cache>(&arena);

        if (cache.LastEventId) {
            protoCache.SetlastEventId(IntFromString<ui64, 10>(cache.LastEventId));
            cache.List.Serialize(protoCache);
        }

        return protoCache.SerializeAsString();
    }

    static const TString MAX_EVENT_ID_QUERY(
        "SELECT max(id) as num FROM domains_events UNION ALL SELECT count(domain_id) as num FROM domains");

    TDomainCachePtr TDomainFetcher::Load() const {
        const TInstant startLoading = TInstant::Now();

        TDomainCache cache;
        TDomainListBuilder builder(cache);
        TString domainsListQuery;

        try {
            const TInstant startMaxId = TInstant::Now();
            // Create DB handle
            NDbPool::TTable eventResult;
            {
                NDbPool::TBlockingHandle sqlh(Dbp_);

                // Remember max event id in the domains_events table before
                // reading the entire hosted_domains tble
                // Do the query
                eventResult = sqlh.Query(MAX_EVENT_ID_QUERY)->ExctractTable();
            }
            if (eventResult.size() != 2) {
                throw yexception() << "Error: DomainFetcher::Load(): internal error: "
                                   << "unexpected result set for the max event id query. "
                                   << "Got " << eventResult.size() << " rows instead of 2";
            }

            TString maxId;
            Cell2Str(eventResult[0], 0, maxId);
            builder.SetLastEventId(maxId);
            // Save last processed event id
            TLog::Info() << "PDD domains: max domains_events id=" << maxId
                         << ". Request took " << (TInstant::Now() - startMaxId);

            size_t reserveSize = RESERVE_RATIO * eventResult[1][0].As<ui64>();
            builder.Reserve(reserveSize);

            // Now the main query
            TString maxDomainId("0");
            while (true) {
                const TInstant startPage = TInstant::Now();

                domainsListQuery = TDomainListBuilder::MakeQuery(maxDomainId, Settings_.PageSize);
                NDbPool::NUtils::TResponseWithRetries result = NDbPool::NUtils::DoQueryTries(Dbp_, domainsListQuery, Settings_.PageRetryCount);
                const TInstant readyPage = TInstant::Now();

                if (result.Result->size() <= 0) {
                    break;
                }

                for (NDbPool::TRow& row : result.Result->MutateTable()) {
                    maxDomainId = builder.AddRowAndGetDomId(std::move(row));
                }

                TLog::Info() << "Page of PDD domains list loaded in " << (TInstant::Now() - startPage)
                             << ". Request took " << (readyPage - startPage);

                if (result.Result->size() < Settings_.PageSize) {
                    break;
                }
            }
        } catch (const NDbPool::TException& e) {
            throw yexception() << "Error: mail4domains: dbpool exception in the query: " << e.what()
                               << ", query: <" << domainsListQuery << ">, " << Dbp_.GetDbInfo();
        }

        builder.Finish();
        TLog::Info() << "PDD domains list loaded in " << (TInstant::Now() - startLoading)
                     << ". Total count: " << cache.List.size();
        return std::make_shared<TDomainCache>(std::move(cache));
    }

    void TDomainFetcher::SaveLoadedCache(TDomainCachePtr cache) {
        SecondaryCache_ = std::make_shared<TDomainCache>(*cache);
        Cache_.Set(cache);
    }

    void TDomainFetcher::Update() {
        Y_VERIFY(!SecondaryCache_->LastEventId.empty(), "update() has not last event_id. That is a bug");

        const TString query = TDomainListUpdater::MakeQuery(SecondaryCache_->LastEventId);
        const NDbPool::TTable table = GetUpdatingResult(query);
        if (table.empty()) {
            return;
        }

        TLog::Info() << "PDD domains - updating: cache is going to be updated with: " << table.size() << " rows";

        if (!UpdateWithRows(table)) {
            TLog::Warning("PDD domains - updating: forced immediate reloading of PDD domains cache");
            SaveLoadedCache(Load());
        }
    }

    bool TDomainFetcher::UpdateWithRows(const NDbPool::TTable& table) {
        // First of all we need to patch secondary cache - it is not being used now
        {
            TDomainListUpdater updater(*SecondaryCache_);
            updater.SetLogging(true);
            if (ProcessUpdatingResult(table, updater) == status_Need_Reload) {
                return false;
            }
        }
        TLog::Debug() << "PDD domains - updating: post-proccess finished #1";

        // Now secondary cache has actual data
        // We can apply it for workers
        SecondaryCache_ = Cache_.Swap(std::move(SecondaryCache_));
        TLog::Info("PDD domains - updating: new cache was applied");

        // swap() guarantees that secondary cache is owned with only one instance of smart prt
        // So we can safely patch the second instance of cache
        {
            TDomainListUpdater updater(*SecondaryCache_);
            updater.SetLogging(false);
            ProcessUpdatingResult(table, updater);
        }
        TLog::Debug() << "PDD domains - updating: post-proccess finished #2";

        return true;
    }

    NDbPool::TTable TDomainFetcher::GetUpdatingResult(const TString& query) const {
        try {
            // Create DB handle
            NDbPool::TBlockingHandle sqlh(Dbp_);

            // Do the query. This query joins the domains_events and
            // hosted_domains table thus providing us with all the required
            // info at once. Presumably, events are rare and most of the time
            // result set ius empty and occasionally it consists of few rows.
            return sqlh.Query(query)->ExctractTable();
        } catch (const NDbPool::TException& e) {
            throw yexception() << "Error: mail4domains: dbpool exception in the query: " << e.what()
                               << ", query: <" << query << ">, " << Dbp_.GetDbInfo();
        }
    }

    TDomainFetcher::EStatus TDomainFetcher::ProcessUpdatingResult(const NDbPool::TTable& table,
                                                                  TDomainListUpdater& updater) {
        // Go through rows and act upon each event:
        //  - remove domain from the cache if name is NULL
        //  - set new value for domain from DB if name is not NULL (add or replace)
        //  - reload skip all rows and return value indicating
        //    that complete cache reloading is necessary if domain_id == "0"
        //
        //  Every domain exists in response only once.
        //  Every row is consistent data for domain on query executing moment.
        //  Swaping of two domains looks like two rows in response.
        //  Because of swaping don't try interpret row as transaction.
        //
        //  Read more: PASSP-16174
        for (const NDbPool::TRow& row : table) {
            if (TDomainListUpdater::Event::Reload == updater.AddRow(row)) {
                return status_Need_Reload;
            }
        }

        updater.Finish();
        return status_Current;
    }

    void TDomainFetcher::Run() {
        try {
            if (NextCacheFileWrite_ == TInstant()) {
                // Full reload is required to avoid incremental errors in cache
                SaveLoadedCache(Load());
            } else {
                Update();
            }

            const TInstant cacheWritingStart = TInstant::Now();
            if (NextCacheFileWrite_ < cacheWritingStart) {
                NextCacheFileWrite_ = cacheWritingStart + Settings_.CacheFileWritePeriod;
                TLog::Debug() << "PDD domains: starting to save cache to file: " << Settings_.CacheFile;
                SaveToFile(*Cache_.Get(), Settings_.CacheFile);
                TLog::Debug() << "PDD domains: finished to save cache to file: " << Settings_.CacheFile;
            }
        } catch (const std::exception& e) {
            TLog::Warning("BlackBox: PDD domains: leaving hosted domains list intact: %s", e.what());
        }
    }
}
