#include "domain_list.h"

#include <passport/infra/daemons/blackbox/src/protobuf/domain_lists.pb.h>

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    static const float MAX_LOAD_FACTOR = 0.5;
    static const float RESERVE_RATIO = 1.1;

    TDomainList::TDomainList() {
        Reserve(0);
    }

    TDomainList::TDomainList(const domainlists_proto::Cache& c) {
        Reserve(c.domains().size());

        for (const domainlists_proto::Domain& d : c.domains()) {
            const TString id = IntToString<10>(d.id());

            IndexByName_.emplace(d.asciiname(), id);
            ListById_.emplace(id, d);
        }
    }

    TDomainList::TDomainList(const TDomainList& o)
        : Logging_(o.Logging_)
    {
        Reserve(o.ListById_.size());
        ListById_ = o.ListById_;
        IndexByName_ = o.IndexByName_;
    }

    void TDomainList::Serialize(domainlists_proto::Cache& out) const {
        auto& domains = *out.mutable_domains();
        domains.Reserve(ListById_.size());

        for (const auto& [id, domain] : ListById_) {
            domain.Serialize(*domains.Add());
        }
    }

    void TDomainList::DeleteDomain(const TString& id) {
        auto it = ListById_.find(id);
        if (it == ListById_.end()) {
            // In this case indexByName_ may have names with incorrect domain id
            // Assume it is rare case
            if (Logging_) {
                TLog::Info("DomainList: cannot delete domain %s", id.c_str());
            }
            return;
        }

        if (it->second.IsSlave()) {
            // Erase from master's slave list
            auto itMaster = ListById_.find(it->second.Master());
            if (itMaster == ListById_.end()) {
                if (Logging_) {
                    TLog::Warning("DomainList: failed to drop slave %s from master %s: master is absent",
                                  id.c_str(),
                                  it->second.Master().c_str());
                }
            } else {
                if (Logging_) {
                    TLog::Info("DomainList: droped slave %s from master %s",
                               id.c_str(),
                               it->second.Master().c_str());
                }
                itMaster->second.DropSlave(id);
            }
        } else {
            // Master: erase slaves too
            for (const TString& idSlave : it->second.Slaves()) {
                auto itSlave = ListById_.find(idSlave);
                if (itSlave == ListById_.end()) {
                    if (Logging_) {
                        TLog::Info("DomainList: slave %s already deleted for master %s",
                                   idSlave.c_str(),
                                   id.c_str());
                    }
                } else {
                    ListById_.erase(itSlave);
                    if (Logging_) {
                        TLog::Info("DomainList: delete slave %s for master %s",
                                   idSlave.c_str(),
                                   id.c_str());
                    }
                }
            }
        }

        ListById_.erase(it);

        // do not delete anything from indexByName_ because it is buggy proccess:
        // name can already point to another domain
    }

    void TDomainList::AddOrUpdateDomain(TDomain&& newDomain) {
        // Assume master_domain_id cannot be changed
        // There is no public API for this
        auto oldDomain = ListById_.find(newDomain.Id());
        if (oldDomain != ListById_.end() && oldDomain->second.Master() != newDomain.Master()) {
            if (Logging_) {
                TLog::Warning("DomainList: master domain id of domain %s changed from %s to %s, this is not supported, operation ignored",
                              newDomain.Id().c_str(),
                              oldDomain->second.Master().c_str(),
                              newDomain.Master().c_str());
            }
            return;
        }

        // Process master-slave relationships
        if (newDomain.IsSlave()) {
            auto itMaster = ListById_.find(newDomain.Master());
            if (itMaster == ListById_.end()) {
                if (Logging_) {
                    TLog::Warning("DomainList: master domain %s is not found for slave domain %s, domain not added",
                                  newDomain.Master().c_str(),
                                  newDomain.Id().c_str());
                }
                return;
            }
            itMaster->second.AddSlave(newDomain.Id());
        }

        // newDomain object is moved later, so we need a copy of name and id
        const TString newAsciiName = newDomain.AsciiName();
        const TString newId = newDomain.Id();

        if (oldDomain == ListById_.end()) {
            // ID not found - we are adding a domain with completely new ID
            if (Logging_) {
                TLog::Info("DomainList: new domain %s is added with name '%s'", newId.c_str(), newAsciiName.c_str());
            }
            ListById_.emplace(newId, std::move(newDomain));
            // it is possible that we had the same domain NAME with different ID
            // so here (below) the name will start point to new domain but old domain_id can still remain in the cache
        } else {
            // we already have the same ID in cache, need to update cache consistently
            auto itName = IndexByName_.find(oldDomain->second.AsciiName());
            if (itName == IndexByName_.end()) {
                // old domain can not be found by name
                // that is strange, but it is safe to ignore
                // we'll add new domain data instead of the old one and put new name to index
                if (Logging_) {
                    TLog::Info("DomainList: old name '%s' not found before setting new '%s' for id=%s",
                               oldDomain->second.AsciiName().c_str(),
                               newAsciiName.c_str(),
                               newId.c_str());
                }
            } else if (itName->second != newId) {
                // old domain name already points to some other id, don't touch the mapping
                // this could be in the middle of swap operation
                if (Logging_) {
                    TLog::Info("DomainList: old name '%s' points to id=%s (new id is %s)",
                               oldDomain->second.AsciiName().c_str(),
                               itName->second.c_str(),
                               newId.c_str());
                }
            } else {
                // old domain name points to the same domain that we are updating
                // we need to delete this link, it does not show to correct domain anymore
                IndexByName_.erase(itName);
                if (Logging_) {
                    TLog::Info("DomainList: deleted old name '%s' before setting new '%s' for id=%s",
                               oldDomain->second.AsciiName().c_str(),
                               newAsciiName.c_str(),
                               newId.c_str());
                }
            }

            // Updating domain value
            oldDomain->second.UpdateDomain(std::move(newDomain));
            if (Logging_) {
                TLog::Info("DomainList: domain %s with name '%s' is updated", newId.c_str(), newAsciiName.c_str());
            }
        }

        // Set new value for name
        IndexByName_[newAsciiName] = newId;
    }

    const TDomain* TDomainList::FindById(const TString& id) const {
        auto it = ListById_.find(id);
        return it == ListById_.end() ? nullptr : &it->second;
    }

    const TDomain* TDomainList::FindByName(const TString& name) const {
        auto itName = IndexByName_.find(name);
        if (itName == IndexByName_.end()) {
            return nullptr;
        }
        auto it = ListById_.find(itName->second);
        return it == ListById_.end() ? nullptr : &it->second;
    }

    void TDomainList::Reserve(size_t res) {
        ListById_.max_load_factor(MAX_LOAD_FACTOR);
        IndexByName_.max_load_factor(MAX_LOAD_FACTOR);

        if (0 == res) {
            return;
        }

        ListById_.reserve(RESERVE_RATIO * res);
        IndexByName_.reserve(RESERVE_RATIO * res);
    }

    size_t TDomainList::size() const {
        return ListById_.size();
    }

    TDomainList::TIterator TDomainList::GetIterator() const {
        return TIterator(*this);
    }

    void TDomainList::SetLogger(bool logging) {
        Logging_ = logging;
    }

    bool TDomainList::operator==(const TDomainList& o) const {
        return ListById_ == o.ListById_ &&
               IndexByName_ == o.IndexByName_ &&
               Logging_ == o.Logging_;
    }

    TDomainList::TIterator::TIterator(const TDomainList& parent)
        : Parent_(parent)
        , It_(Parent_.ListById_.begin())
    {
    }

    const TDomain* TDomainList::TIterator::Next() {
        if (It_ == Parent_.ListById_.end()) {
            return nullptr;
        }
        return &(It_++)->second;
    }
}
