#include "fetcher.h"

#include <passport/infra/libs/cpp/auth_core/oauth_token_parser.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/gamma_fetcher/fetcher.h>
#include <passport/infra/libs/cpp/keyutils/keys.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <unordered_map>

namespace NPassport::NBb::NGammas {
    NAuth::TGammaKeeperPtr TFetcher::CreateKeeper(NGammaFetcher::TIssPartFetcherPtr issPartsFetcher,
                                                  const NXml::TConfig& config,
                                                  const TString& xpath) {
        NGammaFetcher::TFetcher::TGammas gammas =
            NGammaFetcher::TFetcher::FetchGammas(std::move(issPartsFetcher), config, xpath);

        if (gammas.Gammas.empty()) {
            return NAuth::TGammaKeeper::Create();
        }

        NAuth::TGammaKeeperSettings settings{
            .Gammas = std::move(gammas.Gammas),
            .SignedTypes = InitSignedTypes(config, xpath),
            .SigningGamma = gammas.SigningGamma,
            .KeyspaceToType = InitKeyspaces(config, xpath),
            .AllowToCheckWithoutGammaCreatedBefore = InitTimeBounds(config, xpath),
        };

        return NAuth::TGammaKeeper::Create(std::move(settings));
    }

    static const std::unordered_map<TString, NAuth::EEnitityType> TYPE_MAPPING = {
        {"session", NAuth::EEnitityType::Session},
        {"smth", NAuth::EEnitityType::Smth},
        {"oauth", NAuth::EEnitityType::OAuth},
        {"other", NAuth::EEnitityType::Other},
    };

    NAuth::EEnitityType TFetcher::GetType(const TString& type, const TString& path) {
        auto it = TYPE_MAPPING.find(type);

        Y_ENSURE(it != TYPE_MAPPING.end(),
                 "type '" << type << "' is unknown in " << path);

        return it->second;
    }

    NAuth::TGammaKeeperSettings::TKeyspaces TFetcher::InitKeyspaces(const NXml::TConfig& config,
                                                                    const TString& xpath) {
        NAuth::TGammaKeeperSettings::TKeyspaces res;

        for (const TString& xkeyspace : config.SubKeys(xpath + "/serve_kspaces/entry")) {
            const TString keyspace = config.AsString(xkeyspace + "/@id");
            const TString xtype = xkeyspace + "/@type";

            res.emplace(
                NAuth::TSessionUtils::Dots2under(keyspace),
                GetType(config.AsString(xtype), xtype));
        }

        res.emplace(NAuth::TOAuthTokenParser::KEYSPACE,
                    NAuth::EEnitityType::OAuth);

        return res;
    }

    NAuth::TGammaKeeperSettings::TTimeBound TFetcher::InitTimeBounds(const NXml::TConfig& config,
                                                                     const TString& xpath) {
        NAuth::TGammaKeeperSettings::TTimeBound res;

        for (const TString& xbefore : config.SubKeys(
                 xpath + "/gamma_fetcher/allow_to_check_without_gamma/created_before")) {
            const TString xtype = xbefore + "/@type";

            res.emplace(
                GetType(config.AsString(xtype), xtype),
                TInstant::Seconds(config.AsNum<time_t>(xbefore)));
        }

        return res;
    }

    NAuth::TGammaKeeperSettings::TSignedTypes TFetcher::InitSignedTypes(const NXml::TConfig& config,
                                                                        const TString& xpath) {
        NAuth::TGammaKeeperSettings::TSignedTypes res;

        for (const TString& item : config.SubKeys(xpath + "/gamma_fetcher/signed_types/item")) {
            const TString xtype = item + "/@type";
            if (!config.AsBool(item)) {
                continue;
            }

            const TString type = config.AsString(xtype);

            res.emplace(GetType(type, xtype));
            TLog::Debug() << "GammaFetcher: signing for " << type << " enabled";
        }

        return res;
    }
}
