#include "consumer.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/stream/str.h>

namespace std {
    template <>
    struct hash<NPassport::NBb::TBlackboxMethods::EMethod> {
        std::size_t operator()(const NPassport::NBb::TBlackboxMethods::EMethod& t) const {
            return hash<ui32>()(static_cast<ui32>(t));
        }
    };
}

namespace NPassport::NBb {
    static const TString EMPTY_;

    const std::unordered_map<TBlackboxMethods::EMethod, TString> TBlackboxMethods::namesMap_ = {
        {TBlackboxMethods::Login, "Login"},
        {TBlackboxMethods::Cookie, "Cookie"},
        {TBlackboxMethods::OAuth, "OAuth"},
        {TBlackboxMethods::UserInfo, "UserInfo"},
        {TBlackboxMethods::CheckIp, "CheckIp"},
        {TBlackboxMethods::LoginOccupation, "LoginOccupation"},
        {TBlackboxMethods::PwdHistory, "PwdHistory"},
        {TBlackboxMethods::CreateSession, "CreateSession"},
        {TBlackboxMethods::HostedDomains, "HostedDomains"},
        {TBlackboxMethods::FindPddAccounts, "FindPddAccounts"},
        {TBlackboxMethods::LCookie, "LCookie"},
        {TBlackboxMethods::PhoneBindings, "PhoneBindings"},
        {TBlackboxMethods::PhoneOperations, "PhoneOperations"},
        {TBlackboxMethods::TestPwdHashes, "TestPwdHashes"},
        {TBlackboxMethods::GetTrack, "GetTrack"},
        {TBlackboxMethods::ProveKeyDiag, "ProveKeyDiag"},
        {TBlackboxMethods::EditTotp, "EditTotp"},
        {TBlackboxMethods::EmailBindings, "EmailBindings"},
        {TBlackboxMethods::GetAllTracks, "GetAllTracks"},
        {TBlackboxMethods::YakeyBackup, "YakeyBackup"},
        {TBlackboxMethods::CreatePwdHash, "CreatePwdHash"},
        {TBlackboxMethods::DeletionOperations, "DeletionOperations"},
        {TBlackboxMethods::CreateOAuthToken, "CreateOAuthToken"},
        {TBlackboxMethods::GetRecoveryKeys, "GetRecoveryKeys"},
        {TBlackboxMethods::CheckRfcTotp, "CheckRfcTotp"},
        {TBlackboxMethods::UserTicket, "UserTicket"},
        {TBlackboxMethods::Sign, "Sign"},
        {TBlackboxMethods::CheckSign, "CheckSign"},
        {TBlackboxMethods::CheckHasPlus, "CheckHasPlus"},
        {TBlackboxMethods::CheckDeviceSignature, "CheckDeviceSignature"},
        {TBlackboxMethods::GetDevicePublicKey, "GetDevicePublicKey"},
        {TBlackboxMethods::FamilyInfo, "FamilyInfo"},
        {TBlackboxMethods::FindByPhoneNumbers, "FindByPhoneNumbers"},
        {TBlackboxMethods::GeneratePublicId, "GeneratePublicId"},
        {TBlackboxMethods::GetMaxUid, "GetMaxUid"},
        {TBlackboxMethods::WebauthnCredentials, "WebauthnCredentials"},
        {TBlackboxMethods::GetOAuthTokens, "GetOAuthTokens"},
    };

    const TString& TBlackboxMethods::Name(TBlackboxMethods::EMethod m) {
        auto p = TBlackboxMethods::namesMap_.find(m);
        if (p != TBlackboxMethods::namesMap_.end()) {
            return p->second;
        }

        return EMPTY_;
    }

    const std::unordered_map<TBlackboxFlags::EFlag, TString> TBlackboxFlags::namesMap_ = {
        {TBlackboxFlags::OAuthAttributes, "OAuthAttributes"},
        {TBlackboxFlags::LoginByUid, "LoginByUid"},
        {TBlackboxFlags::ResignCookie, "ResignCookie"},
        {TBlackboxFlags::CreateStressSession, "CreateStressSession"},
        {TBlackboxFlags::FindByPhoneAlias, "FindByPhoneAlias"},
        {TBlackboxFlags::GetHiddenAliases, "GetHiddenAliases"},
        {TBlackboxFlags::GetPublicName, "GetPublicName"},
        {TBlackboxFlags::ResignForDomains, "ResignForDomains"},
        {TBlackboxFlags::CreateGuard, "CreateGuard"},
        {TBlackboxFlags::GetBillingFeatures, "GetBillingFeatures"},
        {TBlackboxFlags::CheckDeviceSignatureWithPublicKey, "CheckDeviceSignatureWithPublicKey"},
        {TBlackboxFlags::CheckDeviceSignatureWithDebugMode, "CheckDeviceSignatureWithDebugMode"},
        {TBlackboxFlags::FamilyInfoPlace, "FamilyInfoPlace"},
        {TBlackboxFlags::ForceShowMailSubscription, "ForceShowMailSubscription"},
        {TBlackboxFlags::GetWebauthnCredentials, "GetWebauthnCredentials"},
        {TBlackboxFlags::ScholarLogin, "ScholarLogin"},
        {TBlackboxFlags::ScholarSession, "ScholarSession"},
        {TBlackboxFlags::FederalAccounts, "FederalAccounts"},
        {TBlackboxFlags::FullInfo, "FullInfo"},
    };

    const TString& TBlackboxFlags::Name(TBlackboxFlags::EFlag m) {
        auto p = TBlackboxFlags::namesMap_.find(m);
        if (p != TBlackboxFlags::namesMap_.end()) {
            return p->second;
        }

        return EMPTY_;
    }

    TConsumer::TConsumer(NTvmAuth::TTvmId clientId)
        : ClientId_(clientId)
    {
        AllowedMethods_.resize(TBlackboxMethods::TOTAL_COUNT, false);
        AllowedFlags_.resize(TBlackboxFlags::TOTAL_COUNT, false);
    }

    TConsumer::~TConsumer() = default;

    void TConsumer::SetName(const TString& n) {
        Name_ = n;
    }

    const TString& TConsumer::GetName() const {
        return Name_;
    }

    void TConsumer::SetAllow(TBlackboxMethods::EMethod m, bool f) {
        AllowedMethods_[static_cast<ui32>(m)] = f;
    }

    bool TConsumer::IsAllowed(TBlackboxMethods::EMethod m) const {
        return AllowedMethods_[static_cast<ui32>(m)];
    }

    void TConsumer::SetAllow(TBlackboxFlags::EFlag m, bool f) {
        AllowedFlags_[static_cast<ui32>(m)] = f;
    }

    bool TConsumer::IsAllowed(TBlackboxFlags::EFlag m) const {
        return AllowedFlags_[static_cast<ui32>(m)];
    }

    namespace {
        TString PrintFieldSet(const TConsumer::TFieldSet& container) {
            TString result;

            for (const auto& [elem, rank] : container) {
                NUtils::AppendSeparated(result, ", ", elem);
            }

            return result;
        }

        template <typename Set>
        TString PrintSet(const Set& container) {
            TString result;

            for (const TString& elem : container) {
                NUtils::AppendSeparated(result, ", ", elem);
            }

            return result;
        }
    }

    TString TConsumer::ToString() const {
        TStringStream fmt;

        fmt << "Service=" << Name_
            << ", CAPTCHA=" << (CanCaptcha_ ? 'Y' : 'N')
            << ", DELAY=" << (CanDelay_ ? 'Y' : 'N')
            << ", PIN_TEST=" << (AllowPinTest_ ? 'Y' : 'N')
            << ", LOGIN=" << (IsAllowed(TBlackboxMethods::Login) ? (LoginSafety_ == TConsumer::login_Weak ? 'U' : 'Y') : 'N');

        for (size_t idx = 0; idx < TBlackboxMethods::TOTAL_COUNT; ++idx) {
            fmt << ", " << TBlackboxMethods::Name(static_cast<TBlackboxMethods::EMethod>(idx)) << '=' << (AllowedMethods_[idx] ? 'Y' : 'N');
        }
        for (size_t idx = 0; idx < TBlackboxFlags::TOTAL_COUNT; ++idx) {
            fmt << ", " << TBlackboxFlags::Name(static_cast<TBlackboxFlags::EFlag>(idx)) << '=' << (AllowedFlags_[idx] ? 'Y' : 'N');
        }

        if (!Fields_.empty()) {
            fmt << ", DB FIELDS: " << PrintFieldSet(Fields_);
        } else {
            fmt << ", no DB FIELDS";
        }

        if (!Attrs_.empty()) {
            fmt << ", ATTRIBUTES: " << PrintFieldSet(Attrs_);
        } else {
            fmt << ", no ATTRIBUTES";
        }

        if (!PhoneAttrs_.empty()) {
            fmt << ", PHONE_ATTRIBUTES: " << PrintFieldSet(PhoneAttrs_);
        } else {
            fmt << ", no PHONE_ATTRIBUTES";
        }

        if (!AllowedPartitions_.empty()) {
            fmt << ", PARTITIONS: " << PrintSet(AllowedPartitions_);
        } else {
            fmt << ", no PARTITIONS";
        }

        if (!CanSign_.empty()) {
            fmt << ", SIGN signspaces: " << PrintSet(CanSign_);
        } else {
            fmt << ", no SIGN signspaces";
        }

        if (!CanChecksign_.empty()) {
            fmt << ", CHECK_SIGN signspaces: " << PrintSet(CanChecksign_);
        } else {
            fmt << ", no CHECK_SIGN signspaces";
        }

        return fmt.Str();
    }

    void TConsumer::SetCaptchaCap(bool f) {
        CanCaptcha_ = f;
    }

    bool TConsumer::CanCaptcha() const {
        return CanCaptcha_;
    }

    void TConsumer::SetDelayCap(bool f) {
        CanDelay_ = f;
    }

    bool TConsumer::CanDelay() const {
        return CanDelay_;
    }

    void TConsumer::SetAllowPinTest(bool f) {
        AllowPinTest_ = f;
    }

    bool TConsumer::IsPinTestAllowed() const {
        return AllowPinTest_;
    }

    static bool IsAllowed(const TConsumer::TFieldSet& map, const TString& field, TConsumer::ERank rankFromReq) {
        auto it = map.find(field);
        if (it == map.end()) {
            return false;
        }
        TConsumer::ERank permittedRank = it->second;
        return permittedRank <= rankFromReq;
    }

    bool TConsumer::IsFieldAllowed(const TString& field, TConsumer::ERank rankFromReq) const {
        return NBb::IsAllowed(Fields_, field, rankFromReq);
    }

    bool TConsumer::IsAttrAllowed(const TString& attr, TConsumer::ERank rankFromReq) const {
        return NBb::IsAllowed(Attrs_, attr, rankFromReq);
    }

    bool TConsumer::IsPhoneAttrAllowed(const TString& attr, TConsumer::ERank rankFromReq) const {
        return NBb::IsAllowed(PhoneAttrs_, attr, rankFromReq);
    }

    NTvmAuth::TTvmId TConsumer::GetClientId() const {
        return ClientId_;
    }

    void TConsumer::SetLoginSafety(ELoginMode mode) {
        LoginSafety_ = mode;
    }

    TConsumer::ELoginMode TConsumer::LoginSafety() const {
        return LoginSafety_;
    }

    static void InsertRow(TConsumer::TFieldSet& map, const TString& s, TConsumer::ERank rank) {
        auto res = map.insert({s, rank});
        if (res.second) {
            return;
        }
        if (res.first->second > rank) {
            res.first->second = rank;
        }
    }

    void TConsumer::AddField(const TString& s, TConsumer::ERank rank) {
        InsertRow(Fields_, s, rank);
    }

    void TConsumer::AddAttr(const TString& s, TConsumer::ERank rank) {
        InsertRow(Attrs_, s, rank);
    }

    void TConsumer::AddPhoneAttr(const TString& s, TConsumer::ERank rank) {
        InsertRow(PhoneAttrs_, s, rank);
    }

    void TConsumer::AddSignSignspace(const TString& signSpace) {
        CanSign_.insert(signSpace);
    }

    void TConsumer::AddChecksignSignspace(const TString& signSpace) {
        CanChecksign_.insert(signSpace);
    }

    void TConsumer::AddCheckDeviceSignatureSignspace(const TString& space) {
        CheckDeviceSignature_.insert(space);
    }

    bool TConsumer::IsSignAllowed(const TString& signSpace) const {
        return CanSign_.find(signSpace) != CanSign_.end();
    }

    bool TConsumer::IsChecksignAllowed(const TString& signSpace) const {
        return CanChecksign_.find(signSpace) != CanChecksign_.end();
    }

    bool TConsumer::IsCheckDeviceSignatureAllowed(const TString& space) const {
        return CheckDeviceSignature_.find(space) != CheckDeviceSignature_.end();
    }

    TString TConsumer::PrintId(const TString& ip) const {
        TStringStream str;
        str << Name_ << " (tvm_id=";
        if (ClientId_ > 0) {
            str << ClientId_;
        } else {
            str << "NULL";
        }
        str << ";IP=" << ip << ")";
        return str.Str();
    }

    void TConsumer::AddCachableMethod(const TString& method) {
        CachableMethods_.insert(method);
    }

    bool TConsumer::IsMethodCacheable(const TString& method) const {
        return CachableMethods_.find(method) != CachableMethods_.end();
    }

    void TConsumer::AddPartition(const TString& partition) {
        AllowedPartitions_.insert(partition);
    }

    void TConsumer::AddPartition(TString&& partition) {
        AllowedPartitions_.emplace(std::move(partition));
    }

    bool TConsumer::IsPartitionAllowed(TStringBuf partition) const {
        return AllowedPartitions_.contains(partition);
    }

    static TConsumer CreateUnknownConsumer() {
        TConsumer res;
        res.SetName("_unknown_");
        return res;
    }

    static const TConsumer UNKNOWN_CONSUMER = CreateUnknownConsumer();

    const TConsumer& TConsumer::Unknown() {
        return UNKNOWN_CONSUMER;
    }
}
