#pragma once

#include <library/cpp/tvmauth/type.h>

#include <util/generic/hash_set.h>
#include <util/generic/string.h>

#include <map>
#include <memory>
#include <unordered_map>
#include <unordered_set>

namespace NPassport::NBb {
    struct TBlackboxMethods {
        enum EMethod: ui32 {
            Login = 0,
            Cookie,
            OAuth,
            UserInfo,
            CheckIp,
            LoginOccupation,
            PwdHistory,
            CreateSession,
            HostedDomains,
            FindPddAccounts,
            LCookie,
            PhoneBindings,
            PhoneOperations,
            TestPwdHashes,
            GetTrack,
            ProveKeyDiag,
            EditTotp,
            EmailBindings,
            GetAllTracks,
            YakeyBackup,
            CreatePwdHash,
            DeletionOperations,
            CreateOAuthToken,
            GetRecoveryKeys,
            CheckRfcTotp,
            UserTicket,
            Sign,
            CheckSign,
            CheckHasPlus,
            CheckDeviceSignature,
            GetDevicePublicKey,
            FamilyInfo,
            FindByPhoneNumbers,
            GeneratePublicId,
            GetMaxUid,
            WebauthnCredentials,
            GetOAuthTokens,

            TOTAL_COUNT
        };

        static const TString& Name(EMethod m);

    private:
        static const std::unordered_map<EMethod, TString> namesMap_;
    };

    struct TBlackboxFlags {
        enum EFlag {
            LoginByUid,
            OAuthAttributes,
            ResignCookie,
            CreateStressSession,
            FindByPhoneAlias,
            GetHiddenAliases,
            GetPublicName,
            ResignForDomains,
            CreateGuard,
            GetBillingFeatures,
            CheckDeviceSignatureWithPublicKey,
            CheckDeviceSignatureWithDebugMode,
            FamilyInfoPlace,
            ForceShowMailSubscription,
            GetWebauthnCredentials,
            ScholarLogin,
            ScholarSession,
            FederalAccounts,
            FullInfo,

            TOTAL_COUNT
        };

        static const TString& Name(EFlag m);

    private:
        static const std::unordered_map<EFlag, TString> namesMap_;
    };

    //
    // Class representing a consumer - a BlackBox client. Incapsulates
    // properties associated with a specific IP or range of IP, such
    // as description, allowed DB fields and ability to show CAPTCHA.
    //
    class TConsumer {
    public:
        // Historically, this enumeration defined degree of how agressively we detect/prevent
        // login/password attacks. Currently, this is defined flexibly by individual members
        // of LoginLimits. So, we keep this just as a boolean: ON or OFF.
        enum ELoginMode {
            login_Strict,
            login_Weak
        };

        enum class ERank {
            NoCred,
            HasCred
        };

        TConsumer(NTvmAuth::TTvmId clientId = 0);
        ~TConsumer();

        void SetName(const TString& n);
        const TString& GetName() const;

        void SetAllow(TBlackboxMethods::EMethod m, bool f);
        bool IsAllowed(TBlackboxMethods::EMethod m) const;

        void SetAllow(TBlackboxFlags::EFlag m, bool f);
        bool IsAllowed(TBlackboxFlags::EFlag m) const;

        TString ToString() const;

        void SetCaptchaCap(bool f);
        bool CanCaptcha() const;

        void SetDelayCap(bool f);
        bool CanDelay() const;

        void SetAllowPinTest(bool f);
        bool IsPinTestAllowed() const;

        void SetLoginSafety(ELoginMode mode);
        ELoginMode LoginSafety() const;

        using TFieldSet = std::unordered_map<TString, ERank>;
        void AddField(const TString& s, TConsumer::ERank rank);
        void AddAttr(const TString& s, TConsumer::ERank rank);
        void AddPhoneAttr(const TString& s, TConsumer::ERank rank);

        bool IsFieldAllowed(const TString& field, TConsumer::ERank rank) const;
        bool IsAttrAllowed(const TString& attr, TConsumer::ERank rank) const;
        bool IsPhoneAttrAllowed(const TString& attr, TConsumer::ERank rank) const;

        NTvmAuth::TTvmId GetClientId() const;

        // sign/checksign signspaces
        using TSignspaceSet = std::unordered_set<TString>;
        void AddSignSignspace(const TString& signSpace);
        void AddChecksignSignspace(const TString& signSpace);
        void AddCheckDeviceSignatureSignspace(const TString& space);

        bool IsSignAllowed(const TString& signSpace) const;
        bool IsChecksignAllowed(const TString& signSpace) const;
        bool IsCheckDeviceSignatureAllowed(const TString& space) const;

        TString PrintId(const TString& ip) const;

        void AddCachableMethod(const TString& method);
        bool IsMethodCacheable(const TString& method) const;

        void AddPartition(const TString& partition);
        void AddPartition(TString&& partition);
        bool IsPartitionAllowed(TStringBuf partition) const;

        static const TConsumer& Unknown();

    private:
        // allowed methods map
        std::vector<bool> AllowedMethods_;
        std::vector<bool> AllowedFlags_;
        std::unordered_set<TString> CachableMethods_;

        bool CanCaptcha_ = false;
        bool CanDelay_ = false;
        bool AllowPinTest_ = false;
        ELoginMode LoginSafety_ = login_Strict;
        TString Name_;
        TFieldSet Fields_;
        TFieldSet Attrs_;
        TFieldSet PhoneAttrs_;
        const NTvmAuth::TTvmId ClientId_;
        TSignspaceSet CanSign_;
        TSignspaceSet CanChecksign_;
        TSignspaceSet CheckDeviceSignature_;
        THashSet<TString> AllowedPartitions_;
    };
}
