#include "grants_checker.h"

#include "consumer.h"

#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>

#include <passport/infra/libs/cpp/utils/string/split.h>

namespace NPassport::NBb {
    TGrantsChecker::TGrantsChecker(const NCommon::TRequest& request,
                                   const TConsumer& consumer,
                                   bool throwOnError)
        : Request_(request)
        , Consumer_(consumer)
        , ThrowOnError_(throwOnError)
    {
    }

    void TGrantsChecker::CheckMethodAllowed(TBlackboxMethods::EMethod method) {
        if (Consumer_.IsAllowed(method)) {
            return;
        }

        ProcessError(NUtils::CreateStr("no grants for method=", Request_.GetArg(TStrings::METHOD)));
    }

    void TGrantsChecker::CheckNotEmptyArgAllowed(const TString& arg, TBlackboxFlags::EFlag flag) {
        if (Request_.GetArg(arg).empty() || Consumer_.IsAllowed(flag)) {
            return;
        }

        ProcessError(NUtils::CreateStr("no grants for arg '", arg, "'"));
    }

    void TGrantsChecker::CheckHasArgAllowed(const TString& arg, TBlackboxFlags::EFlag flag) {
        if (!Request_.HasArg(arg) || Consumer_.IsAllowed(flag)) {
            return;
        }

        ProcessError(NUtils::CreateStr("no grants for arg '", arg, "'"));
    }

    void TGrantsChecker::CheckArgIsTrue(const TString& arg, TBlackboxFlags::EFlag flag) {
        if (!TUtils::GetBoolArg(Request_, arg) || Consumer_.IsAllowed(flag)) {
            return;
        }

        ProcessError(NUtils::CreateStr("no grants for ", arg, "=", Request_.GetArg(arg)));
    }

    void TGrantsChecker::Add(TString&& error) {
        ProcessError(std::move(error));
    }

    const NCommon::TRequest& TGrantsChecker::GetRequest() {
        return Request_;
    }

    const TConsumer& TGrantsChecker::GetConsumer() {
        return Consumer_;
    }

    const TCheckGrantsResult& TGrantsChecker::GetResult() const {
        return Result_;
    }

    TCheckGrantsResult TGrantsChecker::Extract() {
        return std::move(Result_);
    }

    void TGrantsChecker::ProcessArgWithValue(const TString& key, const TString& value) {
        ProcessError(NUtils::CreateStr("no grants for ", key, "=", value));
    }

    void TGrantsChecker::ProcessError(TString&& error) {
        if (ThrowOnError_) {
            throw TBlackboxError(TBlackboxError::EType::AccessDenied)
                << error << ". consumer: " << Consumer_.PrintId(Request_.GetRemoteAddr());
        }

        Result_.Errors.insert(std::move(error));
    }
}
