#include "ip_grants.h"

#include "consumer.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/regular_task.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <util/stream/file.h>
#include <util/system/fstat.h>

#include <regex>

namespace NPassport::NBb {
    static const std::regex ILLEGAL_DBFIELD_REGEX("[^a-zA-Z._0-9-]");

    TIpGrants::TIpGrants(const TString& path, TDuration timeout)
        : TIpAclLoader(path, "BlackBox: new grants are in effect:", "BlackBox: Peer added:")
    {
        Init(timeout);
    }

    std::shared_ptr<TIpAclMap<TConsumer>> TIpGrants::Parse(const TStringBuf fileBody, const time_t mtime) {
        // Try XML-parse the entire config file at once, then pick pieces we
        // need using XPath
        auto acl = std::make_shared<TIpAclMap<TConsumer>>();
        acl->SetMTime(mtime);
        try {
            NXml::TConfig config = NXml::TConfig::ReadFromMemory(fileBody);

            for (const TString& entry : config.SubKeys("/peers/entry")) {
                TString idstr;
                try {
                    // id attribute (containing IP(s)) is mandatory, keep it at hand
                    idstr = config.AsString(entry + "/@id");
                } catch (const std::exception& e) {
                    TLog::Error("Grants parser: missing 'id' attribute in a 'peers' entry, skipping this entry");
                    continue;
                }

                //
                // Create Peer and fill it with configured values.
                //
                std::shared_ptr<TConsumer> consumer = std::make_shared<TConsumer>();

                // Name
                consumer->SetName(config.AsString(entry + "/name", "NONAME"));

                // Add cache options
                for (const TString& method : {"userinfo", "oauth", "sessionid", "hosted_domains"}) {
                    if (config.Contains(entry + "/__allow_cache_for_" + method)) {
                        consumer->AddCachableMethod(method);
                        TLog::Info() << "Grants parser: enable cache for method=" << method
                                     << ": " << consumer->GetName();
                    }
                }

                // login/password checking policy
                consumer->SetCaptchaCap(config.Contains(entry + "/can_captcha"));
                consumer->SetDelayCap(config.Contains(entry + "/can_delay"));
                consumer->SetAllowPinTest(config.Contains(entry + "/allow_pin_test"));
                if (config.Contains(entry + "/allow_login")) {
                    TString allow = config.AsString(entry + "/allow_login");
                    consumer->SetAllow(TBlackboxMethods::Login, true);
                    consumer->SetLoginSafety(allow == "weak" ? TConsumer::login_Weak : TConsumer::login_Strict);
                }

                // methods
                consumer->SetAllow(TBlackboxMethods::Cookie, config.Contains(entry + "/allow_parse_cookie"));
                consumer->SetAllow(TBlackboxMethods::OAuth, config.Contains(entry + "/allow_oauth"));
                consumer->SetAllow(TBlackboxMethods::UserInfo, config.Contains(entry + "/allow_user_info"));
                consumer->SetAllow(TBlackboxMethods::CheckIp, config.Contains(entry + "/allow_check_ip"));
                consumer->SetAllow(TBlackboxMethods::LoginOccupation, config.Contains(entry + "/allow_login_occupation"));
                consumer->SetAllow(TBlackboxMethods::PwdHistory, config.Contains(entry + "/allow_pwd_history"));
                consumer->SetAllow(TBlackboxMethods::CreateSession, config.Contains(entry + "/allow_create_session"));
                consumer->SetAllow(TBlackboxMethods::HostedDomains, config.Contains(entry + "/allow_hosted_domains"));
                consumer->SetAllow(TBlackboxMethods::FindPddAccounts, config.Contains(entry + "/allow_find_pdd_accounts"));
                consumer->SetAllow(TBlackboxMethods::LCookie, config.Contains(entry + "/allow_l_cookie"));
                consumer->SetAllow(TBlackboxMethods::PhoneBindings, config.Contains(entry + "/allow_phone_bindings"));
                consumer->SetAllow(TBlackboxMethods::PhoneOperations, config.Contains(entry + "/allow_phone_operations"));
                consumer->SetAllow(TBlackboxMethods::TestPwdHashes, config.Contains(entry + "/allow_test_pwd_hashes"));
                consumer->SetAllow(TBlackboxMethods::GetTrack, config.Contains(entry + "/allow_get_track"));
                consumer->SetAllow(TBlackboxMethods::ProveKeyDiag, config.Contains(entry + "/allow_prove_key_diag"));
                consumer->SetAllow(TBlackboxMethods::EditTotp, config.Contains(entry + "/allow_edit_totp"));
                consumer->SetAllow(TBlackboxMethods::EmailBindings, config.Contains(entry + "/allow_email_bindings"));
                consumer->SetAllow(TBlackboxMethods::GetAllTracks, config.Contains(entry + "/allow_get_all_tracks"));
                consumer->SetAllow(TBlackboxMethods::YakeyBackup, config.Contains(entry + "/allow_yakey_backup"));
                consumer->SetAllow(TBlackboxMethods::CreatePwdHash, config.Contains(entry + "/allow_create_pwd_hash"));
                consumer->SetAllow(TBlackboxMethods::DeletionOperations, config.Contains(entry + "/allow_deletion_operations"));
                consumer->SetAllow(TBlackboxMethods::CreateOAuthToken, config.Contains(entry + "/allow_create_oauth_token"));
                consumer->SetAllow(TBlackboxMethods::GetRecoveryKeys, config.Contains(entry + "/allow_get_recovery_keys"));
                consumer->SetAllow(TBlackboxMethods::CheckRfcTotp, config.Contains(entry + "/allow_check_rfc_totp"));
                consumer->SetAllow(TBlackboxMethods::UserTicket, config.Contains(entry + "/allow_user_ticket"));
                consumer->SetAllow(TBlackboxMethods::CheckHasPlus, config.Contains(entry + "/allow_check_has_plus"));
                consumer->SetAllow(TBlackboxMethods::GetDevicePublicKey, config.Contains(entry + "/allow_get_device_public_key"));
                consumer->SetAllow(TBlackboxMethods::FamilyInfo, config.Contains(entry + "/allow_family_info"));
                consumer->SetAllow(TBlackboxMethods::FindByPhoneNumbers, config.Contains(entry + "/allow_find_by_phone_numbers"));
                consumer->SetAllow(TBlackboxMethods::GeneratePublicId, config.Contains(entry + "/allow_generate_public_id"));
                consumer->SetAllow(TBlackboxMethods::GetMaxUid, config.Contains(entry + "/allow_get_max_uid"));
                consumer->SetAllow(TBlackboxMethods::WebauthnCredentials, config.Contains(entry + "/allow_method_webauthn_credentials"));
                consumer->SetAllow(TBlackboxMethods::GetOAuthTokens, config.Contains(entry + "/allow_get_oauth_tokens"));

                // flags
                consumer->SetAllow(TBlackboxFlags::LoginByUid, config.Contains(entry + "/allow_login_by_uid"));
                consumer->SetAllow(TBlackboxFlags::OAuthAttributes, config.Contains(entry + "/allow_oauth_attributes"));
                consumer->SetAllow(TBlackboxFlags::ResignCookie, config.Contains(entry + "/allow_resign_cookie"));
                consumer->SetAllow(TBlackboxFlags::CreateStressSession, config.Contains(entry + "/allow_create_stress_session"));
                consumer->SetAllow(TBlackboxFlags::FindByPhoneAlias, config.Contains(entry + "/allow_find_by_phone_alias"));
                consumer->SetAllow(TBlackboxFlags::GetHiddenAliases, config.Contains(entry + "/allow_to_get_hidden_aliases"));
                consumer->SetAllow(TBlackboxFlags::GetPublicName, config.Contains(entry + "/allow_get_public_name"));
                consumer->SetAllow(TBlackboxFlags::ResignForDomains, config.Contains(entry + "/allow_resign_for_domains"));
                consumer->SetAllow(TBlackboxFlags::CreateGuard, config.Contains(entry + "/allow_create_guard"));
                consumer->SetAllow(TBlackboxFlags::GetBillingFeatures, config.Contains(entry + "/allow_get_billing_features"));
                consumer->SetAllow(TBlackboxFlags::CheckDeviceSignatureWithPublicKey, config.Contains(entry + "/allow_check_device_signature_with_public_key"));
                consumer->SetAllow(TBlackboxFlags::CheckDeviceSignatureWithDebugMode, config.Contains(entry + "/allow_check_device_signature_with_debug_mode"));
                consumer->SetAllow(TBlackboxFlags::FamilyInfoPlace, config.Contains(entry + "/allow_family_info_get_place"));
                consumer->SetAllow(TBlackboxFlags::ForceShowMailSubscription, config.Contains(entry + "/allow_force_show_mail_subscription"));
                consumer->SetAllow(TBlackboxFlags::GetWebauthnCredentials, config.Contains(entry + "/allow_get_webauthn_credentials"));
                consumer->SetAllow(TBlackboxFlags::ScholarLogin, config.Contains(entry + "/allow_scholar_login"));
                consumer->SetAllow(TBlackboxFlags::ScholarSession, config.Contains(entry + "/allow_scholar_session"));
                consumer->SetAllow(TBlackboxFlags::FederalAccounts, config.Contains(entry + "/allow_federal_accounts"));
                consumer->SetAllow(TBlackboxFlags::FullInfo, config.Contains(entry + "/allow_fullinfo"));

                // Allowed DB fields
                for (const TString& f : config.SubKeys(entry + "/dbfield")) {
                    TString field;
                    try {
                        field = config.AsString(f + "/@id");
                    } catch (const std::exception& e) {
                        TLog::Error("Grants parser: empty 'id' attribute in a dbfield in peer with id='%s', ignoring this dbfield", idstr.c_str());
                        continue;
                    }

                    if (field.empty() || std::regex_search(field.cbegin(), field.cend(), ILLEGAL_DBFIELD_REGEX)) {
                        TLog::Error("Grants parser: illegal dbfield '%s' in peer with id='%s', ignoring this field", field.c_str(), idstr.c_str());
                        continue;
                    }
                    consumer->AddField(field, TConsumer::ERank::NoCred);
                }

                // Allowed attributes
                std::vector<TString> attrs = NUtils::NormalizeListValue(config.AsString(entry + "/allowed_attributes", ""), ",;");
                for (const TString& attr : attrs) {
                    consumer->AddAttr(attr, TConsumer::ERank::NoCred);
                }

                // Allowed phone attributes
                attrs = NUtils::NormalizeListValue(config.AsString(entry + "/allowed_phone_attributes", ""), ",;");
                for (const TString& attr : attrs) {
                    consumer->AddPhoneAttr(attr, TConsumer::ERank::NoCred);
                }

                // Allowed sign/checksign signspaces
                if (config.Contains(entry + "/allow_sign")) {
                    std::vector<TString> allowed = NUtils::NormalizeListValue(config.AsString(entry + "/allow_sign"), ",; ");
                    consumer->SetAllow(TBlackboxMethods::Sign, true);
                    for (const TString& signSpace : allowed) {
                        consumer->AddSignSignspace(signSpace);
                    }
                }

                if (config.Contains(entry + "/allow_check_sign")) {
                    std::vector<TString> allowed = NUtils::NormalizeListValue(config.AsString(entry + "/allow_check_sign"), ",; ");
                    consumer->SetAllow(TBlackboxMethods::CheckSign, true);
                    for (const TString& signSpace : allowed) {
                        consumer->AddChecksignSignspace(signSpace);
                    }
                }

                if (config.Contains(entry + "/allow_check_device_signature")) {
                    std::vector<TString> allowed = NUtils::NormalizeListValue(config.AsString(entry + "/allow_check_device_signature"), ",; ");
                    consumer->SetAllow(TBlackboxMethods::CheckDeviceSignature, true);
                    for (const TString& signSpace : allowed) {
                        consumer->AddCheckDeviceSignatureSignspace(signSpace);
                    }
                }

                // Allowed partitions
                std::vector<TString> partitions = NUtils::NormalizeListValue(config.AsString(entry + "/allowed_partitions", ""), ",;");
                for (const TString& partition : partitions) {
                    consumer->AddPartition(partition);
                }

                // Now loop through IP addresses and/or ranges to which this grants set
                // belongs. Split the id attribute which is a ';' - separated list of ranges
                // into individual entries.
                TStringBuf idview(idstr);
                while (TStringBuf ip = idview.NextTok(';')) {
                    try {
                        acl->ParseAndAddEntry(ip, consumer);
                    } catch (const std::exception& e) {
                        TLog::Error("Grants parser: error: %s in peer '%s'; ignoring this range", e.what(), consumer->GetName().c_str());
                    } catch (...) {
                        TLog::Error("Grants parser: unknown error in peer '%s'; ignoring this range", consumer->GetName().c_str());
                    }
                }
            }
        } catch (const std::exception& e) {
            ++ParsingErrors_;
            TLog::Error("BlackBox: error loading grants config: %s (file <%s>)", e.what(), Path_.c_str());
            throw;
        }

        return acl;
    }

    TString TIpGrants::PrintEntryInfo(const TConsumer& consumer) const {
        return NUtils::CreateStr(", name: ", consumer.GetName());
        ;
    }

    ui64 TIpGrants::GetParsingErrors() const {
        return ParsingErrors_.GetValue();
    }
}
