#include "account_helper.h"

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/experiment.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/account_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/split.h>

namespace NPassport::NBb {
    const std::vector<TString> TAccountHelper::ALL_ALIASES =
        {"1", "2", "3", "5", "6", "7", "8", "9", "10", "11", "12", "13", "15", "16", "17", "18", "19", "20", "21", "22", "23", "24", "25"};

    TAccountHelper::TAccountHelper(TDbFieldsConverter& conv, const NCommon::TRequest& request) {
        // add all aliases for constructing login, in their priority order, order is important here!
        LoginAliases_ = {
            conv.Fetcher().AddAlias(TAlias::PORTAL_LOGIN),
            conv.Fetcher().AddAlias(TAlias::ALT_DOMAIN_LOGIN),
            conv.Fetcher().AddAlias(TAlias::LITE_LOGIN),
            conv.Fetcher().AddAlias(TAlias::PDD_MASTER_LOGIN),
            conv.Fetcher().AddAlias(TAlias::SCHOLAR),
            conv.Fetcher().AddAlias(TAlias::MAILISH_LOGIN),
            conv.Fetcher().AddAlias(TAlias::UBER_ID),
            conv.Fetcher().AddAlias(TAlias::KINOPOISK_ID),
        };

        Regname_ = conv.Add(TDbFieldsConverter::REGNAME);
        HavePwd_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_HAVE_PASSWORD);
        HaveHint_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_HAVE_HINT);
        EnablePhoneAlias_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_ENABLE_SEARCH_BY_PHONE_ALIAS);

        AreAliasesRequired_ = request.HasArg(TStrings::ALIASES);
        const TString& aliases = request.GetArg(TStrings::ALIASES);
        if (aliases.empty()) {
            return;
        }

        if (aliases == TStrings::ALL_WITH_HIDDEN) {
            HiddenAliasesAsked_ = true;
        }

        if (aliases == TStrings::GET_SOCIAL) {
            Aliases_.push_back(TAlias::SOCIAL_LOGIN);
        } else if (aliases == TStrings::GET_PHONENUMBER) {
            Aliases_.push_back(TAlias::PHONE_NUMBER);
        } else if (aliases == TStrings::ALL || aliases == TStrings::ALL_WITH_HIDDEN) {
            Aliases_ = ALL_ALIASES;
        } else {
            Aliases_ = NUtils::NormalizeListValue(aliases, ", ");
        }

        for (const TString& alias : Aliases_) {
            unsigned val = TUtils::ToUInt(alias, TStrings::ALIASES);
            if (val < 1 || val > 25 || val == 14) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "Unsupported alias type: " << InvalidValue(alias);
            }
            conv.Fetcher().AddAlias(alias);
        }
    }

    void TAccountHelper::CheckGrants(TGrantsChecker& checker) {
        bool needGrant = checker.GetRequest().GetArg(TStrings::ALIASES) == TStrings::ALL_WITH_HIDDEN;
        if (needGrant && !checker.GetConsumer().IsAllowed(TBlackboxFlags::GetHiddenAliases)) {
            checker.Add(NUtils::CreateStr("no grants to use aliases=", TStrings::ALL_WITH_HIDDEN));
        }
    }

    std::unique_ptr<TAccountChunk>
    TAccountHelper::Result(const TDbProfile* profile) const {
        if (nullptr == profile) {
            return std::unique_ptr<TAccountChunk>();
        }

        const TDbProfile::TAliases& aliases = profile->Aliases();
        bool phoneAliasEnabled = profile->Get(EnablePhoneAlias_)->AsBoolean();

        TAccountChunk::TAliasesType data;
        for (const TString& aliasType : Aliases_) {
            if (TAlias::PDD_ALIAS_LOGIN == aliasType) {
                for (const TString& aliasPdd : profile->AliasesPdd()) {
                    data.insert(std::make_pair(TAlias::PDD_ALIAS_LOGIN, aliasPdd));
                }
            } else if (TAlias::OLDPUBLICID == aliasType) {
                if (!HiddenAliasesAsked_) {
                    continue;
                }
                for (const TString& alias : profile->AliasesOldPublic()) {
                    data.insert(std::make_pair(TAlias::OLDPUBLICID, alias));
                }
            } else {
                // check if we allowed to show phone alias (skip if not enabled by user AND not all_with_hidden option given)
                if (aliasType == TAlias::PHONE_NUMBER && !phoneAliasEnabled && !HiddenAliasesAsked_) {
                    continue;
                }

                // skip public_id aliases if not allowed to show hidden aliases
                if (aliasType == TAlias::PUBLICID && !HiddenAliasesAsked_) {
                    continue;
                }

                // skip bank_phone_number aliases if not allowed to show hidden aliases
                if (aliasType == TAlias::BANK_PHONE_NUMBER && !HiddenAliasesAsked_) {
                    continue;
                }

                TDbProfile::TAliases::const_iterator aliasValue = aliases.find(aliasType);
                if (aliasValue != aliases.end() && aliasValue->second.Exists) {
                    data.insert(std::make_pair(aliasType, aliasValue->second.Value));
                }
            }
        }

        std::unique_ptr<TAccountChunk> accountChunk = std::make_unique<TAccountChunk>();
        accountChunk->Aliases = std::move(data);
        accountChunk->HavePassword = !profile->Get(HavePwd_)->Value.empty();
        accountChunk->HaveHint = !profile->Get(HaveHint_)->Value.empty();

        TString login = GetLoginField(profile);
        if (!login.empty()) {
            accountChunk->Login = TUtils::SelectGreeting(login, Regname_->Value(profile));
        }
        accountChunk->AreAliasesRequired = AreAliasesRequired_;

        return accountChunk;
    }

    const TString& TAccountHelper::GetLoginField(const TDbProfile* profile) const {
        // select value for <login> fields looking at non-synthetic aliases
        for (TDbIndex alias : LoginAliases_) {
            const TDbValue* login = profile->Get(alias);
            if (login->Exists) {
                return login->Value;
            }
        }

        return TStrings::EMPTY;
    }

}
