#include "attributes_helper.h"

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/attributes_chunk.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>

namespace NPassport::NBb {
    TAttributesHelper::TAttributesHelper(TDbFetcher& fetcher,
                                         const TAttributesSettings& settings,
                                         const NCommon::TRequest& request)
        : AttrSettings_(settings)
    {
        std::vector<TString> attrs = TUtils::GetNumbersArgChecked(request.GetArg(TStrings::ATTRIBUTES), "attribute");
        for (const TString& a : attrs) {
            if (AttrSettings_.IsDeleted(a)) {
                continue;
            }
            Attrs_[a] = fetcher.AddAttr(a);
        }
    }

    void TAttributesHelper::CheckGrants(TConsumer::ERank rank,
                                        const TAttributesSettings& settings,
                                        TGrantsChecker& checker) {
        const TString& attrs = checker.GetRequest().GetArg(TStrings::ATTRIBUTES);

        NUtils::Transform(attrs, ',', [&](TStringBuf buf) {
            NUtils::Trim(buf);
            if (buf.empty()) {
                return;
            }

            const TString attr(buf);
            const bool needGrant = settings.RequiresGrant(attr);
            if (needGrant && !checker.GetConsumer().IsAttrAllowed(attr, rank)) {
                checker.Add(NUtils::CreateStr("no grants for attribute '", attr, "'"));
            }
        });
    }

    std::unique_ptr<TAttributesChunk> TAttributesHelper::Result(const TDbProfile* profile) const {
        if (nullptr == profile) {
            return std::unique_ptr<TAttributesChunk>();
        }

        TAttributesChunk::TAttributesType data;
        for (const auto& [type, index] : Attrs_) {
            const TDbValue* value = profile->Get(index);
            if (value && value->Exists) {
                data.insert(std::make_pair(
                    type,
                    AttrSettings_.IsBinary(type) ? NUtils::BinToBase64(value->Value) : value->Value));
            }
        }
        return std::make_unique<TAttributesChunk>(std::move(data));
    }

}
