#include "base_result_helper.h"

#include "attributes_helper.h"
#include "email_attrs_helper.h"
#include "phone_attrs_helper.h"
#include "webauthn_attrs_helper.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/misc/validator.h>
#include <passport/infra/daemons/blackbox/src/output/account_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/attributes_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/base_result.h>
#include <passport/infra/daemons/blackbox/src/output/billing_features_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/dbfields_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/display_name_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/emails_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/ext_attrs_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/karma_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/phone_bindings_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/phone_operations_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>

namespace NPassport::NBb {
    TBaseResultHelper::TBaseResultHelper(TDbFieldsConverter& conv,
                                         const TBlackboxImpl& bb,
                                         const NCommon::TRequest& req)
        : Karma_(conv.Fetcher(), bb.SupportBantime())
        , DbFields_(conv, req)
        , DisplayName_(conv, req, bb.FullFioInPublicNameStartTime(), bb.PublicProfileProtectionStartTime())
        , Account_(conv, req)
        , BillingFeatures_(conv.Fetcher(), req)
        , PlusSubscriberState_(conv.Fetcher(), req)
        , PublicId_(conv.Fetcher(), req, bb.PublicIdEncryptor())
        , FamilyInfo_(conv.Fetcher(), req)
    {
        conv.SetForceShowMailSubscription(TUtils::GetBoolArg(req, TStrings::FORCE_SHOW_MAIL_SUBSCRIPTION));

        if (req.HasArg(TStrings::ATTRIBUTES)) {
            Attrs_ = std::make_unique<TAttributesHelper>(conv.Fetcher(), bb.AttributeSettings(), req);
        }

        Validator_ = bb.CreateValidator(req, conv);

        if (req.HasArg(TStrings::GET_PHONES)) {
            PhoneAttrs_ = std::make_unique<TPhoneAttrsHelper>(conv.Fetcher(), req);
        }

        if (req.HasArg(TStrings::GET_EMAILS)) {
            EmailAttrs_ = std::make_unique<TEmailAttrsHelper>(conv.Fetcher(), req);
        }

        if (req.HasArg(TStrings::GET_WEBAUTHN_CREDENTIALS)) {
            WebauthnAttrs_ = std::make_unique<TWebauthnAttrsHelper>(conv.Fetcher(), req);
        }

        NeedPhoneOperations_ = TUtils::GetBoolArg(req, TStrings::GET_PHONEOPERATIONS);
        if (NeedPhoneOperations_) {
            conv.Fetcher().AddPhoneOperations();
        }

        const EPhoneBindingsType pbType = GetPhoneBindingsType(req);
        NeedPhoneBindings_ = pbType != EPhoneBindingsType::None;
        if (NeedPhoneBindings_) {
            conv.Fetcher().AddPhoneBindings(pbType);
        }
    }

    TBaseResultHelper::~TBaseResultHelper() = default;

    void TBaseResultHelper::CheckGrants(const TDbFieldsSettings& dbfieldSettings,
                                        const TAttributesSettings& attrSettings,
                                        TConsumer::ERank rank,
                                        TGrantsChecker& checker) {
        TDbFieldsHelper::CheckGrants(rank, dbfieldSettings, checker);
        TDisplayNameHelper::CheckGrants(checker);
        TAccountHelper::CheckGrants(checker); // aliases
        TBillingFeaturesHelper::CheckGrants(checker);

        checker.CheckHasArgAllowed(TStrings::FORCE_SHOW_MAIL_SUBSCRIPTION, TBlackboxFlags::ForceShowMailSubscription);

        TAttributesHelper::CheckGrants(rank, attrSettings, checker);
        TPhoneAttrsHelper::CheckGrants(rank, attrSettings, checker);
        TWebauthnAttrsHelper::CheckGrants(checker);
        CheckPhoneOperationsGrants(checker);
        CheckPhoneBindingsGrants(checker);
    }

    void TBaseResultHelper::CheckPhoneOperationsGrants(TGrantsChecker& checker) {
        const bool needGrant = TUtils::GetBoolArg(checker.GetRequest(), TStrings::GET_PHONEOPERATIONS);
        if (needGrant && !checker.GetConsumer().IsAllowed(TBlackboxMethods::PhoneOperations)) {
            checker.Add("no grants for phone operations");
        }
    }

    void TBaseResultHelper::CheckPhoneBindingsGrants(TGrantsChecker& checker) {
        const bool needGrant = GetPhoneBindingsType(checker.GetRequest()) != EPhoneBindingsType::None;
        if (needGrant && !checker.GetConsumer().IsAllowed(TBlackboxMethods::PhoneBindings)) {
            checker.Add("no grants for phone bindings");
        }
    }

    EPhoneBindingsType TBaseResultHelper::GetPhoneBindingsType(const NCommon::TRequest& request) {
        const TString& phoneBindings = request.GetArg(TStrings::GET_PHONEBINDINGS);

        if (phoneBindings == TStrings::ALL) {
            return EPhoneBindingsType::All;
        }
        if (phoneBindings == TStrings::UNBOUND) {
            return EPhoneBindingsType::Unbound;
        }
        if (phoneBindings == TStrings::CURRENT) {
            return EPhoneBindingsType::Current;
        }

        return EPhoneBindingsType::None;
    }

    void TBaseResultHelper::FillResults(TBaseResult& result, const TDbProfile* profile) const {
        result.Karma = Karma_.Result(profile);
        result.Dbfields = DbFields_.Result(profile);
        result.DisplayName = DisplayName_.Result(profile);
        result.Account = Account_.Result(profile);
        result.BillingFeatures = BillingFeatures_.Result(profile);
        result.PlusSubscriberState = PlusSubscriberState_.Result(profile);
        result.PublicId = PublicId_.Result(profile);
        result.FamilyInfo = FamilyInfo_.Result(profile);

        if (Attrs_) {
            result.Attributes = Attrs_->Result(profile);
        }

        if (Validator_) {
            result.Emails = std::make_unique<TEmailsChunk>(Validator_->MakeList(profile));
        }

        if (PhoneAttrs_) {
            result.PhoneAttrs = PhoneAttrs_->Result(profile);
        }

        if (EmailAttrs_) {
            result.EmailAttrs = EmailAttrs_->Result(profile);
        }

        if (WebauthnAttrs_) {
            result.WebauthnAttrs = WebauthnAttrs_->Result(profile);
        }

        if (NeedPhoneOperations_) {
            std::unique_ptr<TPhoneOperationsChunk> opsChunk = std::make_unique<TPhoneOperationsChunk>();
            if (profile) {
                opsChunk->Ops = profile->PhoneOperations(); // copy the whole map
            }
            result.PhoneOperations = std::move(opsChunk);
        }

        if (NeedPhoneBindings_) {
            std::unique_ptr<TPhoneBindingsChunk> bindingsChunk = std::make_unique<TPhoneBindingsChunk>();
            if (profile) {
                bindingsChunk->Bindings = profile->PhoneBindings(); // copy vector
            }
            result.PhoneBindings = std::move(bindingsChunk);
        }
    }
}
