#include "billing_features_helper.h"

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/output/billing_features_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>

#include <passport/protobuf/billing_features/billing_features.pb.h>

#include <util/string/cast.h>

namespace NPassport::NBb {
    TBillingFeaturesHelper::TBillingFeaturesHelper(TDbFetcher& fetcher,
                                                   const NCommon::TRequest& request) {
        if (!request.HasArg(TStrings::GET_BILLING_FEATURES)) {
            return;
        }

        const TString& features = request.GetArg(TStrings::GET_BILLING_FEATURES);
        if (features != TStrings::ALL) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Unsupported get_billing_features value: " << InvalidValue(features);
        }

        FeaturesAttr_ = fetcher.AddAttr(TAttr::ACCOUNT_BILLING_FEATURES);
    }

    void TBillingFeaturesHelper::CheckGrants(TGrantsChecker& checker) {
        checker.CheckHasArgAllowed(TStrings::GET_BILLING_FEATURES, TBlackboxFlags::GetBillingFeatures);
    }

    std::unique_ptr<TBillingFeaturesChunk> TBillingFeaturesHelper::Result(const TDbProfile* profile) const {
        if (!profile || FeaturesAttr_ < 0) {
            return {};
        }

        const TString& value = profile->Get(FeaturesAttr_)->Value;
        std::unique_ptr<TBillingFeaturesChunk> featuresChunk = std::make_unique<TBillingFeaturesChunk>();
        billing_features::BillingFeatures proto;

        if (!proto.ParseFromArray(value.data(), value.size())) {
            TLog::Error() << "Broken billing features protobuf value, uid=" << profile->Uid()
                          << ", attr value=" << NUtils::BinToBase64(value);
            return {};
        }

        for (const auto& [name, attributes] : proto.GetFeatures()) {
            TBillingFeaturesChunk::TFeatureAttributes attrs;
            if (attributes.HasInTrial()) {
                attrs.InTrial = attributes.GetInTrial();
            }
            if (attributes.HasTrialDuration()) {
                attrs.TrialDuration = attributes.GetTrialDuration();
            }
            if (attributes.HasPaidTrial()) {
                attrs.PaidTrial = attributes.GetPaidTrial();
            }
            if (attributes.HasRegionId()) {
                attrs.RegionId = attributes.GetRegionId();
            }
            if (attributes.HasBrand()) {
                attrs.Brand = attributes.GetBrand();
            }
            featuresChunk->Features.emplace(name, std::move(attrs));
        }

        return featuresChunk;
    }
}
