#include "display_name_helper.h"

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/display_name_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <ctime>

namespace NPassport::NBb {
    TDisplayNameHelper::TDisplayNameHelper(TDbFieldsConverter& conv,
                                           const NCommon::TRequest& request,
                                           time_t fullFioStartTime,
                                           time_t publicProfileProtectionStartTime)
        : FullFioStartTime_(fullFioStartTime)
        , PublicProfileProtectionStartTime_(publicProfileProtectionStartTime)
    {
        if (!TUtils::GetBoolArg(request, TStrings::REGNAME)) {
            return;
        }
        Regname_ = conv.Add(TDbFieldsConverter::REGNAME);
        Login_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_NORMALIZED_LOGIN);
        DisplayName_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_DISPLAY_NAME);
        Avatar_ = conv.Fetcher().AddAttr(TAttr::AVATAR_DEFAULT);
        FirstName_ = conv.Fetcher().AddAttr(TAttr::PERSON_FIRSTNAME);
        LastName_ = conv.Fetcher().AddAttr(TAttr::PERSON_LASTNAME);
        IsVerified_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_IS_VERIFIED);

        conv.Fetcher().AddAlias(TAlias::PORTAL_LOGIN);
        conv.Fetcher().AddAlias(TAlias::PHONY_LOGIN);
        conv.Fetcher().AddAlias(TAlias::NEOPHONISH);
        conv.Fetcher().AddExtendedPhoneAttr(TPhoneAttr::E164_NUMBER);
        conv.Fetcher().AddExtendedPhoneAttr(TPhoneAttr::MASKED_FORMATTED_NUMBER);

        IsDisplayNameEmptinessRequired_ = TUtils::GetBoolArg(request, TStrings::IS_DISPLAY_NAME_EMPTY);

        if (TUtils::GetBoolArg(request, TStrings::GET_PUBLIC_NAME)) {
            IsPublicNameRequired_ = true;
            Language_ = conv.Fetcher().AddAttr(TAttr::PERSON_LANGUAGE);
            DontUseDisplaynameAsPublicname_ = conv.Fetcher().AddAttr(TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME);
            ShowFioInPublicName_ = conv.Fetcher().AddAttr(TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME);
            RegistrationTime_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_REGISTRATION_DATETIME);
            PublicProfileAllowed_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_PERSONAL_DATA_PUBLIC_ACCESS_ALLOWED);
            PublicProfile3PartyCanUse_ = conv.Fetcher().AddAttr(TAttr::ACCOUNT_PERSONAL_DATA_THIRD_PARTY_PROCESSING_ALLOWED);
        }
    }

    void TDisplayNameHelper::CheckGrants(TGrantsChecker& checker) {
        if (TUtils::GetBoolArg(checker.GetRequest(), TStrings::REGNAME)) {
            checker.CheckArgIsTrue(TStrings::GET_PUBLIC_NAME, TBlackboxFlags::GetPublicName);
        }
    }

    static const TString PASSPORT_PREFIX("p:");
    static const TString SOCIAL_PREFIX("s:");
    static const TString PHONE_ALIAS_PREFIX("pna:");
    static const TString TEMPLATE_PREFIX("t:");
    static const TString DEFAULT_PDD_TEMPLATE("t:%pdd_username%@%display_domain%");

    bool TDisplayNameHelper::DisplayNameEmpty(const TString& value) {
        if (value.size() < 2) {
            return true;
        }

        return !value.StartsWith(SOCIAL_PREFIX) &&
               !value.StartsWith(PASSPORT_PREFIX) &&
               !value.StartsWith(PHONE_ALIAS_PREFIX) &&
               !value.StartsWith(TEMPLATE_PREFIX);
    }

    std::unique_ptr<TDisplayNameChunk>
    TDisplayNameHelper::Result(const TDbProfile* profile) const {
        if (DisplayName_ == -1 || nullptr == profile) {
            return std::unique_ptr<TDisplayNameChunk>();
        }

        const TString& avatar = profile->Get(Avatar_)->Value;
        bool verified = profile->Get(IsVerified_)->AsBoolean();
        std::unique_ptr<TDisplayNameChunk> displayChunk = std::make_unique<TDisplayNameChunk>(avatar, verified);

        { // here we build a regname and default displayname/publicname values
            TString regName = profile->Get(Login_)->Value;
            if (!regName.empty()) {
                regName = TUtils::SelectGreeting(regName, Regname_->Value(profile));
            }

            // by default displayname and publicname is regname (if not set below)
            displayChunk->SetDisplayName(regName, true, IsDisplayNameEmptinessRequired_);
            if (IsPublicNameRequired_) {
                displayChunk->PublicName = GetPublicName(profile, displayChunk->DisplayName);
            }
            displayChunk->Regname = std::move(regName);
        }

        // this is a phonish account, it has no display name, only masked phone number
        if (profile->HasAlias(TAlias::PHONY_LOGIN) && !profile->HasAlias(TAlias::PORTAL_LOGIN)) {
            // masked number is considered as set up phonish display name
            const TString& maskedFormattedNumber = GetPhoneAttr(profile, TPhoneAttr::MASKED_FORMATTED_NUMBER);
            if (!maskedFormattedNumber.empty()) {
                displayChunk->SetDisplayName(maskedFormattedNumber, false, IsDisplayNameEmptinessRequired_);
                if (IsPublicNameRequired_) {
                    displayChunk->PublicName = GetPublicName(profile, displayChunk->DisplayName);
                }
            }
            return displayChunk;
        }

        // this is not a phonish account, check if neophonish
        bool isNeophonish = profile->HasAlias(TAlias::NEOPHONISH) && !profile->HasAlias(TAlias::PORTAL_LOGIN);

        const TString& displayNameValue = profile->Get(DisplayName_)->Value;
        TString displayName;

        if (!displayNameValue.empty()) {
            // we have custom displayname value, need to generate displayname
            if (displayNameValue.StartsWith(SOCIAL_PREFIX)) {
                TString profileId;
                TString provider;

                if (ParseSocial(displayNameValue, profileId, provider, displayName)) {
                    displayChunk->SetSocial(std::move(profileId),
                                            std::move(provider),
                                            GenerateRedirectTarget(profile->Uid(), profileId));
                }
            } else if (displayNameValue.StartsWith(PASSPORT_PREFIX)) {
                TString::size_type namePos = PASSPORT_PREFIX.size();
                displayName = displayNameValue.substr(namePos);
            } else if (displayNameValue.StartsWith(PHONE_ALIAS_PREFIX)) {
                TString::size_type namePos = PHONE_ALIAS_PREFIX.size();
                displayName = displayNameValue.substr(namePos);
            } else if (displayNameValue.StartsWith(TEMPLATE_PREFIX)) {
                TDisplayNameBuilder builder(FirstName_, LastName_);
                displayName = builder.BuildDisplayName(profile, displayNameValue, displayChunk->Regname);
            }

            if (!displayName.empty()) { // we have valid custom displayname value
                displayChunk->SetDisplayName(displayName, false, IsDisplayNameEmptinessRequired_);
            }
        } else if (isNeophonish) {
            // for neophonish users with no displayname we show full fio
            TStringBuf first = profile->Get(FirstName_)->Value;
            TStringBuf last = profile->Get(LastName_)->Value;
            TString nphneDisplayName = NUtils::CreateStr(NUtils::Trim(first), " ", NUtils::Trim(last));
            NUtils::Trim(nphneDisplayName);

            if (nphneDisplayName.empty()) {
                nphneDisplayName = GetPhoneAttr(profile, TPhoneAttr::E164_NUMBER);
            }

            if (!nphneDisplayName.empty()) {
                displayChunk->SetDisplayName(nphneDisplayName, true, IsDisplayNameEmptinessRequired_);
            }
        } else if (profile->Pdd()) {
            // if no custom displayname but account is PDD - generate from default PDD template
            // note: this generated name is considered empty
            TDisplayNameBuilder builder(FirstName_, LastName_);
            displayChunk->SetDisplayName(builder.BuildDisplayName(profile, DEFAULT_PDD_TEMPLATE, displayChunk->Regname),
                                         true, IsDisplayNameEmptinessRequired_);
        }

        // and now generate public name if needed
        if (IsPublicNameRequired_) {
            bool dontUseDisplayname = profile->Get(DontUseDisplaynameAsPublicname_)->AsBoolean();
            if (dontUseDisplayname || displayName.empty()) {
                bool fioAllowed = profile->Get(ShowFioInPublicName_)->AsBoolean() ||
                                  profile->Get(RegistrationTime_)->AsTime() > FullFioStartTime_;

                // try to generate public name from fio
                TString publicName = BuildPublicName(profile->Get(FirstName_)->Value,
                                                     profile->Get(LastName_)->Value,
                                                     fioAllowed);
                if (publicName.empty() && isNeophonish) {
                    publicName = BuildIncognitoName(profile->Get(Language_)->Value, profile->Uid());
                }

                if (!publicName.empty()) {
                    displayChunk->PublicName = GetPublicName(profile, publicName);
                }
            } else { // use custom displayname as public name
                displayChunk->PublicName = GetPublicName(profile, displayName);
            }
        }

        return displayChunk;
    }

    TString TDisplayNameHelper::BuildPublicName(TStringBuf firstName, TStringBuf lastName, bool showFIO) {
        NUtils::Trim(firstName);
        NUtils::Trim(lastName);

        if (showFIO) {
            // if person agreed to show FIO he should have both fields not empty
            // but it somehow this changes we are ready not to show login
            if (firstName.empty()) {
                return TString(lastName);
            }

            if (lastName.empty()) {
                return TString(firstName);
            }

            return NUtils::CreateStr(firstName, " ", lastName);
        }

        if (firstName.empty()) {
            return {};
        }

        const size_t firstCharLength = NUtils::Utf8CharLength(lastName.cbegin(), lastName.cend());
        if (0 == firstCharLength || lastName.size() == firstCharLength) {
            return TString(firstName);
        }

        return NUtils::CreateStr(firstName, " ", lastName.Head(firstCharLength), ".");
    }

    static const TString INCOGNITO_RU = "Инкогнито";
    static const TString INCOGNITO_EN = "Incognito";

    TString TDisplayNameHelper::BuildIncognitoName(const TString& language, const TString& uid) {
        return NUtils::CreateStr(
            language == TStrings::LANG_RU ? INCOGNITO_RU : INCOGNITO_EN,
            " ",
            TStringBuf(uid).Last(4));
    }

    TString TDisplayNameHelper::SuggestPublicName(TStringBuf firstName, TStringBuf lastName) {
        NUtils::Trim(firstName);
        NUtils::Trim(lastName);

        if (lastName.empty()) {
            return TStrings::EMPTY; // no sense asking for FIO if no last name
        }

        if (firstName.empty()) {
            return TString(lastName);
        }

        return NUtils::CreateStr(firstName, " ", lastName);
    }

    bool TDisplayNameHelper::ParseSocial(const TString& data,
                                         TString& profileId,
                                         TString& provider,
                                         TString& displayName) {
        TString::size_type idPos = SOCIAL_PREFIX.size();
        TString::size_type providerColon = data.find(':', idPos);
        if (providerColon == TString::npos || providerColon + 1 == data.size()) {
            return false; // bad format
        }
        TString::size_type nameColon = data.find(':', providerColon + 1);
        if (nameColon == TString::npos || nameColon + 1 == data.size()) {
            return false; // bad format
        }

        profileId = data.substr(idPos, providerColon - idPos);
        provider = data.substr(providerColon + 1, nameColon - providerColon - 1);
        displayName = data.substr(nameColon + 1, data.size() - nameColon - 1);
        return true;
    }

    static const TString DELIMITER = ".";

    TString TDisplayNameHelper::GenerateRedirectTarget(const TString& uid,
                                                       const TString& id) {
        TString target = NUtils::CreateStrExt(
            52 + id.size(),
            time(nullptr),
            DELIMITER,
            rand() % 100000,
            DELIMITER,
            id,
            DELIMITER);

        TString privatePart(target);
        privatePart.append(uid);

        target.append(NUtils::Bin2hex(NUtils::TCrypto::Md5(privatePart)));

        return target;
    }

    const TString& TDisplayNameHelper::GetPhoneAttr(const TDbProfile* profile, const TString& attr) {
        const auto& ent = profile->ExtendedPhoneAttrs();
        if (!ent.empty()) {
            const auto it = ent.rbegin()->second.find(attr);
            if (it->second.Exists) {
                return it->second.Value;
            }
        }
        return TStrings::EMPTY;
    }

    std::optional<TPublicNameData> TDisplayNameHelper::GetPublicName(const TDbProfile* profile,
                                                                     const TString& name) const {
        return TPublicNameData{
            .Name = name,
            .HasPublicProfile = GetPublicDataFlag(profile, PublicProfileAllowed_),
            .ThirdPartyCanUse = GetPublicDataFlag(profile, PublicProfile3PartyCanUse_),
        };
    }

    bool TDisplayNameHelper::GetPublicDataFlag(const TDbProfile* profile, int attr) const {
        const TDbValue* value = profile->Get(attr);

        // empty value in db is not valid and should not be considered as user decision to hide the profile
        if (value && !value->Value.empty()) { // have explicit user decision flag in db
            return value->AsBoolean();
        }

        // user did not confirm explicitly, we consider old users agreed
        return profile->Get(RegistrationTime_)->AsTime() < PublicProfileProtectionStartTime_;
    }

    static const TString LOGIN = "login";
    static const TString PDD_USERNAME = "pdd_username";
    static const TString PDD_DOMAIN_ = "pdd_domain";
    static const TString DISPLAY_DOMAIN = "display_domain";
    static const TString FIRST_NAME = "firstname";
    static const TString LAST_NAME = "lastname";

    TDisplayNameBuilder::TDisplayNameBuilder(int firstName, int lastName) {
        FirstName_ = firstName;
        LastName_ = lastName;
    }

    TString TDisplayNameBuilder::BuildDisplayName(const TDbProfile* profile,
                                                  const TString& format,
                                                  const TString& regname) const {
        Y_ENSURE(format.size() >= 2, "format is too small '" << format << "'");

        TString result;
        const std::size_t size = format.size();
        result.reserve(size);

        TString::size_type prevEnd = TEMPLATE_PREFIX.size();
        std::size_t curPos;
        while ((curPos = format.find('%', prevEnd)) != TString::npos) {
            result.append(format, prevEnd, curPos - prevEnd);
            prevEnd = curPos;
            if (curPos + 1 == size) {
                break;
            }

            std::size_t startVarName = curPos + 1;
            std::size_t endVar = startVarName;
            while (endVar < size && (isalpha(format[endVar]) || format[endVar] == '_')) {
                ++endVar;
            }
            if (endVar == size) {
                break;
            }

            if (format[endVar] != '%') {
                result.append(format, curPos, endVar - curPos);
                prevEnd = endVar;
                continue;
            }

            prevEnd = endVar + 1;

            std::size_t varNameSize = endVar - startVarName;
            if (format.compare(startVarName, varNameSize, LOGIN) == 0) {
                result.append(regname);
            } else if (format.compare(startVarName, varNameSize, PDD_USERNAME) == 0) {
                AppendPddUsername(profile, regname, result);
            } else if (format.compare(startVarName, varNameSize, PDD_DOMAIN_) == 0 ||
                       format.compare(startVarName, varNameSize, DISPLAY_DOMAIN) == 0)
            {
                result.append(profile->PddDomainUtf());
            } else if (format.compare(startVarName, varNameSize, FIRST_NAME) == 0) {
                AppendDbValue(profile, FirstName_, result);
            } else if (format.compare(startVarName, varNameSize, LAST_NAME) == 0) {
                AppendDbValue(profile, LastName_, result);
            } else {
                result.append(format, curPos, endVar - curPos + 1);
            }
        }

        result.append(format, prevEnd, TString::npos);
        return result;
    }

    void TDisplayNameBuilder::AppendPddUsername(const TDbProfile* profile,
                                                const TString& regname,
                                                TString& result) {
        if (!profile->Pdd()) {
            return;
        }

        std::size_t atPos = regname.find('@');
        if (atPos == TString::npos) {
            return;
        }

        result.append(regname, 0, atPos);
    }

    void TDisplayNameBuilder::AppendDbValue(const TDbProfile* profile,
                                            TDbIndex idx,
                                            TString& result) {
        const TString& value = profile->Get(idx)->Value;
        if (!profile->Get(idx)->Exists || value.empty()) {
            return;
        }

        result.append(value);
    }
}
