#include "email_attrs_helper.h"

#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/ext_attrs_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    static const TString ALL_EMAIL_ATTRIBUTES = "1,2,3,4,5,6,7";

    TEmailAttrsHelper::TEmailAttrsHelper(TDbFetcher& fetcher, const NCommon::TRequest& request) {
        const TString& mode = request.GetArg(TStrings::GET_EMAILS);
        if (mode != TStrings::ALL) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "unknown " << TStrings::GET_EMAILS << " parameter value: " << InvalidValue(mode);
        }

        // add one hidden attr to fetch at least email ids
        fetcher.AddExtendedEmailAttr(TEmailAttr::ADDRESS);

        const TString& attrs = request.GetArg(TStrings::EMAIL_ATTRIBUTES);
        Attrs_ = TUtils::GetNumbersArgChecked(attrs == TStrings::ALL ? ALL_EMAIL_ATTRIBUTES : attrs,
                                              "email attribute");

        for (const TString& attr : Attrs_) {
            fetcher.AddExtendedEmailAttr(attr);
        }
    }

    std::unique_ptr<TExtAttrsChunk> TEmailAttrsHelper::Result(const TDbProfile* profile) const {
        if (nullptr == profile) {
            return std::unique_ptr<TExtAttrsChunk>();
        }
        TExtAttrsChunk::TExtAttrs resAttrs;
        resAttrs.reserve(profile->ExtendedEmailAttrs().size());

        for (const auto& [extId, extAttrs] : profile->ExtendedEmailAttrs()) {
            if (extId.empty()) {
                continue;
            }

            TExtAttrsChunk::TAttrs attrs;
            attrs.reserve(Attrs_.size());

            for (const TString& attr : Attrs_) {
                TDbProfile::TAttrs::const_iterator pval = extAttrs.find(attr);
                if (pval == extAttrs.cend() || !pval->second.Exists) {
                    continue;
                }
                attrs.push_back(std::make_pair(attr, pval->second.Value));
            }
            resAttrs.emplace_back(extId, std::move(attrs));
        }

        return std::make_unique<TExtAttrsChunk>(std::move(resAttrs));
    }

}
