#include "karma_helper.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/output/karma_chunk.h>

#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NBb {
    TKarmaHelper::TKarmaHelper(TDbFetcher& fetcher, bool supportBantime) {
        Karma_ = fetcher.AddAttr(TAttr::KARMA_VALUE);
        if (supportBantime) {
            Bantime_ = fetcher.AddAttr(TAttr::KARMA_ACTIVATION_DATETIME);
        }
    }

    std::unique_ptr<TKarmaChunk>
    TKarmaHelper::Result(const TDbProfile* profile) const {
        // See wiki-yandex-team.ru/passport/karma:
        //  - 0..100 -- original value by Spamoborona
        //  - 1000..1100 -- black-listed by a human
        //  - 3000..3100 -- black-listed based on spam activity on Mail or Narod
        //  - 4000..4100 -- white-listed automatically once user have complained to support@ via mode=userapprove
        //  - 2000..2100 -- white-listed by a human
        //  - 6000..6100 -- self-whitened
        //  - 7000..7100 -- karma 85 set when password changed
        //  - 8000..8100 -- black-listed when password changed
        //  - ...
        //
        //  banTime is a future-landed point where "bad guys" (those having karma=100)
        //  shall be "banned" by particular services. We just return them this point in time.
        //  However, we introduced this late, and older "bad" accounts already processed
        //  by other means and thus having "prefix" marks shall be left intact.
        //  (time_t) 0 indicates "no ban".
        //

        const TString& karmaVal = profile && Karma_ >= 0 ? profile->Get(Karma_)->Value : TStrings::EMPTY;
        const TString& bantime = profile && Bantime_ >= 0 ? profile->Get(Bantime_)->Value : TStrings::EMPTY;

        long rawValue = strtol(karmaVal.c_str(), nullptr, 10);
        long base = rawValue % 1000;
        int prefix = (int)rawValue / 1000;
        if (prefix == 1 || prefix == 3 || prefix == 8) {
            base = 100;
        } else if (prefix == 7) {
            base = 85;
        } else if (prefix > 0) {
            base = 0;
        }
        std::unique_ptr<TKarmaChunk> karmaChunk = std::make_unique<TKarmaChunk>(base, rawValue);
        if (0 == prefix && bantime.size() >= 10) {
            karmaChunk->Bantime = bantime;
        }

        return karmaChunk;
    }

}
