#include "oauth_attrs_helper.h"

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/oauth/fetcher.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>
#include <passport/infra/daemons/blackbox/src/output/attributes_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/split.h>

namespace NPassport::NBb {
    TOAuthAttrsHelper::TOAuthAttrsHelper(TOAuthBaseFetcher& fetcher, const NCommon::TRequest& request) {
        const TString& tokenAttrsArg = request.GetArg(TStrings::OAUTH_TOKEN_ATTRIBUTES);
        const TString& clientAttrsArg = request.GetArg(TStrings::OAUTH_CLIENT_ATTRIBUTES);

        if (tokenAttrsArg.empty() && clientAttrsArg.empty()) {
            return;
        }

        if (tokenAttrsArg == TStrings::ALL) {
            AllTokenAttrs_ = true;
        } else {
            TokenAttrs_ = NUtils::NormalizeListValue(tokenAttrsArg, ", ");

            for (auto& it : TokenAttrs_) {
                // sanitize values: throws on non-numeric attrs
                TUtils::ToUInt(it, TStrings::OAUTH_TOKEN_ATTRIBUTES);
            }
        }

        if (clientAttrsArg == TStrings::ALL) {
            AllClientAttrs_ = true;
            fetcher.AddAllClientAttrs();
        } else {
            ClientAttrs_ = NUtils::NormalizeListValue(clientAttrsArg, ", ");

            for (auto& it : ClientAttrs_) {
                // sanitize values: throws on non-numeric attrs
                TUtils::ToUInt(it, TStrings::OAUTH_CLIENT_ATTRIBUTES);

                fetcher.AddClientAttr(it);
            }
        }
    }

    void TOAuthAttrsHelper::CheckGrants(TGrantsChecker& checker) {
        const NCommon::TRequest& request = checker.GetRequest();
        const TString& tokenAttrsArg = request.GetArg(TStrings::OAUTH_TOKEN_ATTRIBUTES);
        const TString& clientAttrsArg = request.GetArg(TStrings::OAUTH_CLIENT_ATTRIBUTES);

        if ((clientAttrsArg || tokenAttrsArg) && !checker.GetConsumer().IsAllowed(TBlackboxFlags::OAuthAttributes)) {
            checker.Add("no grants for oauth attributes");
        }
    }

    std::unique_ptr<TAttributesChunk> TOAuthAttrsHelper::TokenAttrsChunk(const TOAuthTokenInfo* info) const {
        if (info == nullptr || (TokenAttrs_.empty() && !AllTokenAttrs_)) {
            return {};
        }

        TAttributesChunk::TAttributesType attrs;

        if (AllTokenAttrs_) { // fill with all token attrs
            for (const auto& it : info->TokenAttrs) {
                attrs.insert(it);
            }
        } else { // fill with requested attrs
            for (const TString& it : TokenAttrs_) {
                if (info->HasTokenAttr(it)) {
                    attrs.insert(std::make_pair(it, info->GetTokenAttr(it)));
                }
            }
        }

        return std::make_unique<TAttributesChunk>(std::move(attrs));
    }

    std::unique_ptr<TAttributesChunk> TOAuthAttrsHelper::ClientAttrsChunk(const TOAuthTokenInfo* info) const {
        if (info == nullptr || (ClientAttrs_.empty() && !AllClientAttrs_)) {
            return {};
        }

        TAttributesChunk::TAttributesType attrs;

        if (AllClientAttrs_) { // fill with all client attrs
            for (const auto& it : info->ClientAttrs) {
                attrs.insert(it);
            }
        } else { // fill with requested attrs
            for (const TString& it : ClientAttrs_) {
                if (info->HasClientAttr(it)) {
                    attrs.insert(std::make_pair(it, info->GetClientAttr(it)));
                }
            }
        }

        return std::make_unique<TAttributesChunk>(std::move(attrs));
    }

}
