#include "phone_attrs_helper.h"

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/ext_attrs_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    static const TString ALL_PHONE_ATTRIBUTES = "1,2,3,4,5,6,101,102,103,104,105,106,107,108,109";

    TPhoneAttrsHelper::TPhoneAttrsHelper(TDbFetcher& fetcher, const NCommon::TRequest& request) {
        const TString& mode = request.GetArg(TStrings::GET_PHONES);
        if (mode == TStrings::ALL) {
            Mode_ = EMode::ModeAll;
        } else if (mode == TStrings::BOUND) {
            Mode_ = EMode::ModeBound;
        } else {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "unknown " << TStrings::GET_PHONES << " parameter value: " << InvalidValue(mode);
        }

        // add one hidden attr to fetch at least phone ids
        fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
        if (Mode_ == EMode::ModeBound) {
            // nessesary for 'getphones=bound'
            fetcher.AddExtendedPhoneAttr(TPhoneAttr::BOUND);
        }

        const TString& attrs = request.GetArg(TStrings::PHONE_ATTRIBUTES);
        Attrs_ = TUtils::GetNumbersArgChecked(attrs == TStrings::ALL ? ALL_PHONE_ATTRIBUTES : attrs,
                                              "phone attribute");

        for (const TString& attr : Attrs_) {
            fetcher.AddExtendedPhoneAttr(attr);
        }
    }

    void TPhoneAttrsHelper::CheckGrants(TConsumer::ERank rank,
                                        const TAttributesSettings& settings,
                                        TGrantsChecker& checker) {
        if (!checker.GetRequest().HasArg(TStrings::GET_PHONES)) {
            return;
        }

        const TString& arg = checker.GetRequest().GetArg(TStrings::PHONE_ATTRIBUTES);
        const TString& toProcess = arg == TStrings::ALL ? ALL_PHONE_ATTRIBUTES : arg;

        NUtils::Transform(toProcess, ',', [&](TStringBuf buf) {
            NUtils::Trim(buf);
            if (buf.empty()) {
                return;
            }

            const TString attr(buf);
            const bool needGrant = settings.RequiresGrantPhone(attr);
            if (needGrant && !checker.GetConsumer().IsPhoneAttrAllowed(attr, rank)) {
                checker.Add(NUtils::CreateStr("no grants for phone attribute '", attr, "'"));
            }
        });
    }

    std::unique_ptr<TExtAttrsChunk> TPhoneAttrsHelper::Result(const TDbProfile* profile) const {
        if (nullptr == profile) {
            return std::unique_ptr<TExtAttrsChunk>();
        }
        TExtAttrsChunk::TExtAttrs resAttrs;
        resAttrs.reserve(profile->ExtendedPhoneAttrs().size());

        for (const auto& [extId, extAttrs] : profile->ExtendedPhoneAttrs()) {
            if (extId.empty()) {
                continue;
            }

            if (Mode_ == EMode::ModeBound) {
                const TDbProfile::TAttrs::const_iterator itBound = extAttrs.find(TPhoneAttr::BOUND);
                if (itBound == extAttrs.cend() || !itBound->second.Exists) {
                    continue;
                }

                // Data in db can be inconsistent: for 'bound' phones such data must be skipped
                // For 'all' phones it is ok: this parameter is for passport - he is clever
                const TDbProfile::TAttrs::const_iterator itNumber = extAttrs.find(TPhoneAttr::NUMBER);
                if (itNumber == extAttrs.cend() || !itNumber->second.Exists) {
                    continue;
                }
            }

            resAttrs.push_back(std::make_pair(extId, TExtAttrsChunk::TAttrs()));
            TExtAttrsChunk::TAttrs& attrs = resAttrs.back().second;
            attrs.reserve(Attrs_.size());

            for (const TString& attr : Attrs_) {
                TDbProfile::TAttrs::const_iterator pval = extAttrs.find(attr);
                if (pval == extAttrs.cend() || !pval->second.Exists) {
                    continue;
                }
                attrs.push_back(std::make_pair(attr, pval->second.Value));
            }
        }

        return std::make_unique<TExtAttrsChunk>(std::move(resAttrs));
    }

}
