#include "plus_subscriber_state_helper.h"

#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/output/plus_subscriber_state_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <passport/protobuf/plus_subscriber_state/plus_subscriber_state.pb.h>

namespace NPassport::NBb {
    TPlusSubscriberStateHelper::TPlusSubscriberStateHelper(TDbFetcher& fetcher,
                                                           const NCommon::TRequest& request) {
        if (!request.HasArg(TStrings::GET_PLUS_SUBSCRIBER_STATE)) {
            return;
        }

        const TString& getPlusSubscriptionState = request.GetArg(TStrings::GET_PLUS_SUBSCRIBER_STATE);
        if (getPlusSubscriptionState != TStrings::ALL) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Unsupported get_plus_subscriber_state value: " << InvalidValue(getPlusSubscriptionState);
        }

        StateAttr_ = fetcher.AddAttr(TAttr::ACCOUNT_PLUS_SUBSCRIBER_STATE);
    }

    std::unique_ptr<TPlusSubscriberStateChunk> TPlusSubscriberStateHelper::Result(const TDbProfile* profile,
                                                                                  TInstant now) const {
        if (!profile || StateAttr_ < 0) {
            return {};
        }

        google::protobuf::Arena arena;
        auto& proto = *google::protobuf::Arena::CreateMessage<plus_subscriber_state::PlusSubscriberState>(&arena);

        const TString& value = profile->Get(StateAttr_)->Value;
        if (!proto.ParseFromArray(value.data(), value.size())) {
            TLog::Error() << "Broken plus subscriber state protobuf value, uid=" << profile->Uid()
                          << ", attr value=" << NUtils::BinToBase64(value);
            return {};
        }

        TInstant availableFeaturesEnd = TInstant::Seconds(proto.GetAvailableFeaturesEnd());

        std::unique_ptr<TPlusSubscriberStateChunk> stateChunk = std::make_unique<TPlusSubscriberStateChunk>();

        stateChunk->AvailableFeatures.reserve(proto.AvailableFeaturesSize());
        for (const auto& feature : proto.GetAvailableFeatures()) {
            TInstant featureEnd = feature.GetEnd() ? TInstant::Seconds(feature.GetEnd()) : availableFeaturesEnd;
            if (featureEnd <= now) {
                continue;
            }

            stateChunk->AvailableFeatures.emplace_back(TPlusSubscriberStateChunk::TAvailableFeature{
                .Id = feature.GetId(),
                .Value = feature.GetValue(),
                .End = featureEnd,
            });
        }

        stateChunk->FrozenFeatures.reserve(proto.FrozenFeaturesSize());
        for (const auto& feature : proto.GetFrozenFeatures()) {
            stateChunk->FrozenFeatures.emplace_back(TPlusSubscriberStateChunk::TFrozenFeature{
                .Id = feature.GetId(),
                .Value = feature.GetValue(),
            });
        }

        return stateChunk;
    }
}
