#include "public_id_helper.h"

#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>

#include <passport/infra/libs/cpp/auth_core/public_id.h>
#include <passport/infra/libs/cpp/auth_core/public_id_encryptor.h>
#include <passport/infra/libs/cpp/request/request.h>

namespace NPassport::NBb {
    TPublicIdHelper::TPublicIdHelper(TDbFetcher& fetcher,
                                     const NCommon::TRequest& request,
                                     const NAuth::TPublicIdEncryptor* encryptor)
        : Encryptor_(encryptor)
    {
        if (!Encryptor_ || !TUtils::GetBoolArg(request, TStrings::GET_PUBLIC_ID)) {
            return;
        }

        Version_ = fetcher.AddAttr(TAttr::ACCOUNT_PUBLIC_ID_VERSION);
        PublicIdAlias_ = fetcher.AddAlias(TAlias::PUBLICID);
        UserDefinedPublicId_ = fetcher.AddAttr(TAttr::ACCOUNT_USER_DEFINED_PUBLIC_ID);
    }

    std::unique_ptr<TString>
    TPublicIdHelper::Result(const TDbProfile* profile) const {
        if (!profile || PublicIdAlias_ < 0 || UserDefinedPublicId_ < 0) {
            return {};
        }

        // first check the user-defined public_id
        const TString& alias = profile->Get(PublicIdAlias_)->Value;
        if (!alias.empty()) {
            const TString& userDefinedValue = profile->Get(UserDefinedPublicId_)->Value;
            if (userDefinedValue.empty()) { // user only has normalized value in alias
                return std::make_unique<TString>(alias);
            }
            // we should compare user-defined value if it matches the alias
            // because if edit operation fails half-way, we get old value in alias and new value in attribute
            // and this new attribute value whould be ignored because edit failed
            const TString normalizedValue = TUtils::NormalizeLogin(userDefinedValue);
            if (normalizedValue == alias) {
                // ok, user-defined value is consistent
                return std::make_unique<TString>(userDefinedValue);
            }

            return std::make_unique<TString>(alias);
        }

        // second, check for generated public_id
        if (!Encryptor_ || Version_ < 0) {
            return {};
        }

        ui64 uid;
        if (!TryIntFromString<10>(profile->Uid(), uid)) {
            TLog::Error() << "Invalid uid for user " << profile->Uid();
            return {};
        }

        ui8 version;
        const TString& dbVersion = profile->Get(Version_)->Value;
        if (!TryIntFromString<10>(dbVersion, version)) {
            TLog::Error() << "Bad PublicId version for user " << profile->Uid()
                          << " version=" << dbVersion << " taking 0 instead";
            version = 0;
        }

        TString publicId = Encryptor_->Encrypt(NAuth::TPublicId(uid, version));
        if (publicId.empty()) {
            TLog::Error() << "Failed to generate PublicId for user " << profile->Uid();
            return {};
        }
        return std::make_unique<TString>(publicId);
    }

}
