#include "webauthn_attrs_helper.h"

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/misc/db_fetcher.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>
#include <passport/infra/daemons/blackbox/src/output/ext_attrs_chunk.h>

#include <passport/infra/libs/cpp/request/request.h>

namespace NPassport::NBb {
    static const TString ALL_WEBAUTHN_ATTRIBUTES = "1,2,3,4,5,6,7,8";

    TWebauthnAttrsHelper::TWebauthnAttrsHelper(TDbFetcher& fetcher, const NCommon::TRequest& request) {
        const TString& mode = request.GetArg(TStrings::GET_WEBAUTHN_CREDENTIALS);
        if (mode != TStrings::ALL) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "unknown " << TStrings::GET_WEBAUTHN_CREDENTIALS << " parameter value: " << InvalidValue(mode);
        }

        // add one hidden attr to fetch at least webauthn credential ids
        fetcher.AddExtendedWebauthnAttr(TWebauthnAttr::EXTERNAL_ID);

        const TString& attrs = request.GetArg(TStrings::WEBAUTHN_CREDENTIAL_ATTRIBUTES);
        Attrs_ = TUtils::GetNumbersArgChecked(attrs == TStrings::ALL ? ALL_WEBAUTHN_ATTRIBUTES : attrs,
                                              "webauthn attribute");

        for (const TString& attr : Attrs_) {
            fetcher.AddExtendedWebauthnAttr(attr);
        }
    }

    void TWebauthnAttrsHelper::CheckGrants(TGrantsChecker& checker) {
        checker.CheckHasArgAllowed(TStrings::GET_WEBAUTHN_CREDENTIALS, TBlackboxFlags::GetWebauthnCredentials);
    }

    std::unique_ptr<TExtAttrsChunk> TWebauthnAttrsHelper::Result(const TDbProfile* profile) const {
        if (nullptr == profile) {
            return std::unique_ptr<TExtAttrsChunk>();
        }
        TExtAttrsChunk::TExtAttrs resAttrs;
        resAttrs.reserve(profile->ExtendedWebauthnAttrs().size());

        for (const auto& [extId, extAttrs] : profile->ExtendedWebauthnAttrs()) {
            if (extId.empty()) {
                continue;
            }

            TExtAttrsChunk::TAttrs attrs;
            attrs.reserve(Attrs_.size());

            for (const TString& attr : Attrs_) {
                TDbProfile::TAttrs::const_iterator pval = extAttrs.find(attr);
                if (pval == extAttrs.cend() || !pval->second.Exists) {
                    continue;
                }
                attrs.push_back(std::make_pair(attr, pval->second.Value));
            }
            resAttrs.emplace_back(extId, std::move(attrs));
        }

        return std::make_unique<TExtAttrsChunk>(std::move(resAttrs));
    }
}
