#include "usernets_list.h"

#include "passport/infra/daemons/blackbox/src/ip/ipacl.h"
#include "passport/infra/daemons/blackbox/src/ip/network_descriprion.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>

#include <passport/infra/libs/cpp/request/request.h>

namespace NPassport::NBb {
    TUsernetsList::TUsernetsList(const TString& jsonBody)
        : InternalExplicit_("BlackBox: usernets internal_explicit list:", "BlackBox: internal_explicit network:")
        , RobotExplicit_("BlackBox: usernets robot_explicit list:", "BlackBox: robot_explicit network:")
        , Allow_("BlackBox: usernets allowed list:", "BlackBox: allowed network:")
        , Deny_("BlackBox: usernets deny list:", "BlackBox: deny network:")
        , Robots_("BlackBox: usernets robots list:", "BlackBox: robots network:")
    {
        rapidjson::Document doc;

        if (!NJson::TReader::DocumentAsObject(jsonBody, doc)) {
            throw yexception() << "Usernets: broken json: " << jsonBody;
        }

        try {
            BuildIpAclEntity(doc, InternalExplicit_, "internal_explicit");
            BuildIpAclEntity(doc, RobotExplicit_, "robot_explicit");
            BuildIpAclEntity(doc, Allow_, "allow");
            BuildIpAclEntity(doc, Deny_, "deny");
            BuildIpAclEntity(doc, Robots_, "robots");
        } catch (const yexception& e) {
            throw yexception(e) << ": " << jsonBody;
        }
    }

    void TUsernetsList::BuildIpAclEntity(const rapidjson::Document& doc,
                                         TIpAcl<TNetworkDescription>& nets,
                                         const TZtStringBuf& usernetsListType) {
        const rapidjson::Value* arr;

        Y_ENSURE(NJson::TReader::MemberAsArray(doc, usernetsListType.c_str(), arr),
                 "Usernets: broken usernets json member '" << usernetsListType << "'; expected array");
        std::shared_ptr aclMap = std::make_shared<TIpAclMap<TNetworkDescription>>();

        for (size_t idx = 0; idx < arr->Size(); ++idx) {
            const auto& jsonIpDescription = (*arr)[idx];
            auto netData = UnpackJsonParameters(jsonIpDescription, usernetsListType);
            aclMap->ParseAndAddEntry(TStringBuf(netData.Ip), netData.Description);
        }

        nets.SetAcl(aclMap);
        nets.Print();
    };

    TUsernetsList::TNetData TUsernetsList::UnpackJsonParameters(const rapidjson::Value& jsonIpDescription,
                                                                const TStringBuf& name) {
        TStringBuf ipJsonFieldName = "ip";
        TStringBuf macroJsonFieldName = "macro_name";
        TStringBuf workRateJsonFieldName = "work_rate";

        TNetData netData{
            .Description = std::make_shared<TNetworkDescription>(),
        };
        TNetworkDescription& netDescription = *netData.Description;

        Y_ENSURE(jsonIpDescription.IsObject(), "Usernets: broken usernets json member '" << name << "'; expected object");

        Y_ENSURE(NJson::TReader::MemberAsString(jsonIpDescription, ipJsonFieldName.data(), netData.Ip),
                 "Usernets: broken usernets json member '" << name << "'; expected string in '" << ipJsonFieldName << "'");

        Y_ENSURE(NJson::TReader::MemberAsString(jsonIpDescription, macroJsonFieldName.data(), netDescription.MacroName),
                 "Usernets: broken usernets json member '" << name << "'; expected string in '" << macroJsonFieldName << "'");

        if (jsonIpDescription.HasMember(workRateJsonFieldName.data())) {
            ui32 workRate = 0;
            Y_ENSURE(NJson::TReader::MemberAsUInt(jsonIpDescription, workRateJsonFieldName.data(), workRate),
                     "Usernets: broken usernets json member '" << name << "' expected int in '" << workRateJsonFieldName << "'");
            Y_ENSURE(workRate <= 100,
                     "Usernets: work rate is out of range: " << workRate << "; json member '" << name << "'");
            netDescription.WorkRate = workRate;
        }

        return netData;
    }

    static const THashSet<TString> SKIPABLE_METHODS = {
        "oauth",
        "sessionid",
        "", // sessionid
    };

    ENetworkKind TUsernetsList::FindNetwork(const NUtils::TIpAddr& ip,
                                            const NCommon::TRequest& req,
                                            const TStringBuf description) const {
        auto mapContainsIp = [&](const TIpAcl<TNetworkDescription>& nets, TStringBuf ipset, bool isLogginEnabled = true) {
            auto ipDescription = nets.Find(ip);
            if (!ipDescription) {
                // unknown ip
                return false;
            }

            if (!ipDescription->WorkRate) {
                // ip is not under experiment
                return true;
            }

            const size_t hash = std::hash<NUtils::TIpAddr>()(ip);
            if ((hash % 100) < *ipDescription->WorkRate) {
                if (isLogginEnabled) {
                    LogExperiment(ip, req, description, "wet-run", ipset, ipDescription->MacroName);
                }
                return true; // yandex ip but allowed only for robots
            }

            if (isLogginEnabled) {
                LogExperiment(ip, req, description, "dry-run", ipset, ipDescription->MacroName);
            }
            return false;
        };

        const bool enableLogs = !SKIPABLE_METHODS.contains(req.GetArg(TStrings::METHOD));

        if (mapContainsIp(InternalExplicit_, "internal_explicit", enableLogs)) {
            return ENetworkKind::Internal; // internal whitelist
        }

        if (mapContainsIp(RobotExplicit_, "robot_explicit", enableLogs)) {
            return ENetworkKind::Robot; // robot whitelist
        }

        if (!mapContainsIp(Allow_, "allow")) {
            return ENetworkKind::External; // surely not yandex ip
        }

        if (mapContainsIp(Deny_, "deny")) {
            return ENetworkKind::External; // yandex ip but access denied (guest etc)
        }

        if (mapContainsIp(Robots_, "robots", enableLogs)) {
            return ENetworkKind::Robot;
        }

        return ENetworkKind::Internal;
    }

    static const TString AUTH_TYPE("authtype");

    void TUsernetsList::LogExperiment(const NUtils::TIpAddr& ip,
                                      const NCommon::TRequest& req,
                                      const TStringBuf description,
                                      const TStringBuf branch,
                                      const TStringBuf ipset,
                                      const TStringBuf macroName) {
        const TString& method = req.GetArg(TStrings::METHOD);

        TStringBuf arg;
        if (method == "login") {
            arg = req.GetArg(TStrings::LOGIN);
            if (arg.empty()) {
                arg = req.GetArg(TStrings::UID);
            }
        } else {
            arg = req.GetArg(TStrings::SESSION_ID);
            arg = arg.RBefore('.');
        }

        const TStringBuf authtype = req.GetArg(AUTH_TYPE);

        TLog::Debug() << "UsernetsList: ip was got from robots nets: " << branch
                      << "; ip=" << ip
                      << "; method=" << method
                      << "; consumer=" << req.GetConsumerFormattedName()
                      << "; consumer_ip=" << req.GetRemoteAddr()
                      << "; arg value='" << arg << "'"
                      << "; decription='" << description << "'"
                      << "; ipset=" << ipset
                      << "; macro_name='" << macroName << "'"
                      << "; authtype='" << authtype << "'";
    }
}
