#include "authlog.h"

#include <passport/infra/libs/cpp/utils/log/logger.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

#include <util/generic/yexception.h>

namespace NPassport::NBb {
    static const TString EMPTY;
    static const TString ZERO("0");
    static const TString REV("1");
    static const TString ABSENT("-");
    static const TString BB_SOURCE("bb");
    static const TString UNKNOWN("unknown");
    static const TString VERIFY("verify");

    static const TString CAPTCHA_ENTERED_COMMENT("cpt=1");
    static const TString CAPTCHA_REQUESTED_COMMENT("askcpt=1");

    static const TString NEED_ESCAPING_CHARS(" \r\n\t\v`");

    static const TString& FormatOptional(const TString& s,
                                         const TString& absent = ABSENT) {
        return s.empty() ? absent : s;
    }

    TString& TAuthLog::FormatTimestamp(TString& out, TInstant now) {
        // Required: 2010-03-16T11:02:55.123456+03

        const TString t = now.ToStringLocal(); // 2010-03-16T11:02:55.123456+0300
        out.append(t.begin(), t.end() - 2);    // '-2' to cut off minutes part of the time zone

        return out;
    }

    static TString& Sanitize(TString& out, const TString& s) {
        if (s == ABSENT) {
            out.append(ABSENT);
            return out;
        }

        TString::size_type pos = s.find_first_of(NEED_ESCAPING_CHARS);
        if (pos == TString::npos) {
            pos = s.find('\0');
        }
        if (pos == TString::npos) {
            out.append(s);
            return out;
        }

        out.push_back('`');
        for (const char c : s) {
            switch (c) {
                case '`':
                    out.push_back('`');
                    out.push_back('`');
                    break;
                case '\t':
                    out.push_back('\\');
                    out.push_back('t');
                    break;
                case '\v':
                    out.push_back('\\');
                    out.push_back('v');
                    break;
                case '\r':
                    out.push_back('\\');
                    out.push_back('r');
                    break;
                case '\n':
                    out.push_back('\\');
                    out.push_back('n');
                    break;
                case '\0':
                    out.push_back('\\');
                    out.push_back('0');
                    break;
                default:
                    out.push_back(c);
            }
        }
        out.push_back('`');

        return out;
    }

    static TString FormatComment(const TString& comment, const TString& attribs, bool captchaEntered, bool captchaRequested) {
        TString ret;
        if (!comment.empty()) {
            ret.assign(comment);
            ret.push_back(';');
        }
        if (!attribs.empty()) {
            ret.append(attribs);
            ret.push_back(';');
        }
        if (captchaEntered) {
            ret.append(CAPTCHA_ENTERED_COMMENT);
        } else if (captchaRequested) {
            ret.append(CAPTCHA_REQUESTED_COMMENT);
        }

        return FormatOptional(ret);
    }

    const std::map<int, TString> TAuthLog::flags_ = {
        {TAuthLog::OK, "successful"},
        {TAuthLog::BAD, "failed"},
        {TAuthLog::CAPTCHA, "blocked"},
        {TAuthLog::DISABLED, "disabled"},
        {TAuthLog::BLOCKED, "blocked"},
        {TAuthLog::NEWSESS, "ses_create"},
        {TAuthLog::UPDATESESS, "ses_update"},
        {TAuthLog::BRUTEFORCE, "bruteforce"},
        {TAuthLog::SECONDSTEP, "second_step"},
    };

    TAuthLog::TAuthLog(std::unique_ptr<NUtils::ILogger> logger, const TString& hostid)
        : Logger_(std::move(logger))
        , Hostid_(hostid)
    {
        Y_ENSURE(!hostid.empty());
    }

    TAuthLog::~TAuthLog() = default;

    void TAuthLog::Write(
        const TString& uid,
        const TString& login,
        const TString& sid,
        const TString& authType,
        EFlag flag,
        const TString& comment,
        const TString& attribs,
        bool captchaEntered,
        const TString& userIP,
        const TString& origIP,
        const TString& yandexuid,
        const TString& referer,
        const TString& retpath,
        const TString& useragent) const {
        // Assume these records are a simply waste for history sapce.
        // They persist in login.log anyway so there they are still
        // availbale if required.
        if (flag == TAuthLog::BLOCKED) {
            return;
        }

        // This type is used by Passport and PayPassport to verify password
        // in an non-authorizing contexts. For instance, PapPassport needs
        // to check whether pay-password equals to regular password. These
        // occasions shall not appear in auth.log (PASSP-1847)
        if (authType == VERIFY) {
            return;
        }

        TString record;
        record.reserve(1024);

        // Format all fields one after another. formatOptional simple selects original
        // string if it's not empty or " - " otherwise; sanitize() performs all required
        // escaping and quoting, and it does so in-place.
        record.assign(REV).push_back(' ');
        FormatTimestamp(record).append(1, ' ');
        record.append(Hostid_).append(1, ' ');
        record.append(BB_SOURCE).append(1, ' ');
        record.append(FormatOptional(uid == ZERO ? EMPTY : uid)).append(1, ' ');
        Sanitize(record, FormatOptional(login)).append(1, ' ');
        Sanitize(record, FormatOptional(sid)).append(1, ' ');
        record.append(FormatOptional(authType, UNKNOWN)).append(1, ' ');
        record.append(FormatOptional(flags_.find(flag)->second)).append(1, ' ');
        Sanitize(record, FormatComment(comment, attribs, captchaEntered, flag == TAuthLog::CAPTCHA)).append(1, ' ');
        Sanitize(record, FormatOptional(userIP)).append(1, ' ');
        Sanitize(record, FormatOptional(origIP)).append(1, ' ');
        Sanitize(record, FormatOptional(yandexuid)).append(1, ' ');
        Sanitize(record, FormatOptional(referer)).append(1, ' ');
        Sanitize(record, FormatOptional(retpath)).append(1, ' ');
        Sanitize(record, FormatOptional(useragent));

        Logger_->Log(std::move(record));
    }
}
