#include "tskvlog.h"

#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

#include <util/string/cast.h>

namespace NPassport::NBb {
    class TLine {
    public:
        explicit TLine(const TStringBuf tskvFormat) {
            Str_.reserve(512);
            Str_.append("tskv");
            AddValue("tskv_format", tskvFormat);
            AddValue("unixtime", IntToString<10>(time(nullptr)));
            AddValue("req_id", NUtils::GetThreadLocalRequestId());
        }

        enum class EQuoting {
            Yes,
            No
        };

        void AddValue(const TStringBuf key, const TStringBuf val, EQuoting q = EQuoting::No) {
            if (!val.empty()) {
                Str_.push_back('\t');
                Str_.append(key.begin(), key.end()).push_back('=');
                if (q == EQuoting::Yes) {
                    Str_.append(Quote(val));
                } else {
                    Str_.append(val.data(), val.size());
                }
            }
        }

        TString Finish() {
            return std::move(Str_);
        }

    private:
        static TString Quote(const TStringBuf value) {
            TString esc_val(value);
            NUtils::EscapeEol(esc_val);
            return esc_val;
        }

    private:
        TString Str_;
    };

    TTskvLog::TTskvLog(std::unique_ptr<NUtils::ILogger> logger)
        : Logger_(std::move(logger))
    {
    }

    void TTskvLog::LogBadauthBan(const TStringBuf login,
                                 const TStringBuf uid,
                                 const TStringBuf yandexuid,
                                 const TStringBuf rule,
                                 const TStringBuf count,
                                 const TStringBuf ip,
                                 bool show_captcha,
                                 bool halfIp) const {
        TLine line("blackbox-log");
        line.AddValue("action", halfIp ? "bruteforce_half_ip_ban" : "bruteforce_ban");
        line.AddValue("login", login, TLine::EQuoting::Yes);
        line.AddValue("uid", uid);
        line.AddValue("yandexuid", yandexuid, TLine::EQuoting::Yes);
        line.AddValue("rule", rule);
        line.AddValue("count", count);
        line.AddValue("ip", ip);
        line.AddValue("show_captcha", show_captcha ? "1" : "0");

        Logger_->Log(line.Finish());
    }

    void TTskvLog::LogDeadSession(const TStringBuf uid,
                                  const TStringBuf authid,
                                  long int time_delta) const {
        TLine line("blackbox-log");
        line.AddValue("action", "dead_session");
        line.AddValue("uid", uid);
        line.AddValue("authid", authid);
        line.AddValue("delta", IntToString<10>(time_delta));

        Logger_->Log(line.Finish());
    }

    void TTskvLog::LogOAuth(const TStringBuf tokenId,
                            bool byAlias,
                            const TStringBuf status,
                            const TStringBuf uid,
                            const TStringBuf clientId,
                            const TStringBuf scopes,
                            const TStringBuf createTime,
                            const TStringBuf issueTime,
                            const TStringBuf expireTime,
                            const TStringBuf userIp,
                            const TStringBuf deviceId,
                            const TStringBuf deviceName,
                            bool needsRefresh,
                            const TStringBuf reason,
                            const TStringBuf consumer,
                            const TStringBuf consumerIp) const {
        TLine line("oauth-log");
        line.AddValue("mode", "verify_token");
        line.AddValue("full_check", "1");
        line.AddValue("by_alias", byAlias ? "1" : "0");
        line.AddValue("status", status);
        line.AddValue("token_id", tokenId);
        line.AddValue("uid", uid);
        line.AddValue("client_id", clientId);
        line.AddValue("scopes", scopes);
        line.AddValue("create_time", createTime);
        line.AddValue("issue_time", issueTime);
        line.AddValue("expire_time", expireTime);
        line.AddValue("user_ip", userIp);
        line.AddValue("device_id", deviceId, TLine::EQuoting::Yes);
        line.AddValue("device_name", deviceName, TLine::EQuoting::Yes);
        line.AddValue("needs_refresh", needsRefresh ? "1" : "0");
        line.AddValue("reason", reason);
        line.AddValue("__consumer", consumer, TLine::EQuoting::Yes);
        line.AddValue("__consumer_ip", consumerIp, TLine::EQuoting::Yes);

        Logger_->Log(line.Finish());
    }

    void TTskvLog::LogSesCheck(const TStringBuf uids,
                               const TStringBuf defaultUid,
                               const TStringBuf host,
                               const TStringBuf authid,
                               const TStringBuf idnt,
                               const TStringBuf yandexuid,
                               const TStringBuf userip,
                               const TStringBuf userport,
                               const TStringBuf consumer,
                               const TStringBuf consumerIp,
                               const ui32 consumerTvmId) const {
        TLine line("blackbox-log");
        line.AddValue("action", "sescheck");
        line.AddValue("uids", uids);
        line.AddValue("def_uid", defaultUid);
        line.AddValue("host", host, TLine::EQuoting::Yes);
        line.AddValue("authid", authid);
        line.AddValue("idnt", idnt, TLine::EQuoting::Yes);
        line.AddValue("yuid", yandexuid, TLine::EQuoting::Yes);
        line.AddValue("userip", userip);
        line.AddValue("user_port", userport);
        line.AddValue("__consumer", consumer, TLine::EQuoting::Yes);
        line.AddValue("__consumer_ip", consumerIp, TLine::EQuoting::Yes);
        if (consumerTvmId) {
            line.AddValue("__consumer_tvm_id", IntToString<10>(consumerTvmId));
        }

        Logger_->Log(line.Finish());
    }
}
